﻿/*
 * -----------------------------------------------------------------------------
 * Tagged Stream Format V1.1.19
 * Copyright (c) 2017-2019 Rudy Tellert Elektronik
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy 
 * of this software and associated documentation files (the "Software"), to deal 
 * in the Software without restriction, including without limitation the rights 
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 * SOFTWARE.
 * -----------------------------------------------------------------------------
 */

/*- Requirements: .NET Framework 2 (or better) -------------------------------*/
#define TSF_SUPPORT_EXT

using System;
using System.IO;

namespace Tellert.Format
{
    public class Uuid
    {
        byte[] uuid;

        public static readonly Uuid Empty = new Uuid(new byte[16]);
        public bool IsValid { get { return uuid != null && uuid.Length == 16 && (uuid[6] & 0xf0) > 0; } }

        public Uuid(byte[] bytes)
        {
            uuid = (byte[])bytes.Clone();
        }

        public Uuid(Guid guid)
        {
            byte[] bytes = guid.ToByteArray();

            if (bytes.Length != 16)
            {
                bytes = Empty.uuid;
            }
            else if (BitConverter.IsLittleEndian)
            {
                Array.Reverse(bytes, 0, 4);
                Array.Reverse(bytes, 4, 2);
                Array.Reverse(bytes, 6, 2);
            }
            uuid = bytes;
        }

        public Uuid(string s)
        {
            uuid = Empty.uuid;
            if (string.IsNullOrEmpty(s)) return;

            // remove optional "urn:uuid:" prefix
            if (s.Substring(0, 9).ToLower() == "urn:uuid:") s = s.Substring(9);

            // remove optional braces or parentheses
            if ((s[0] == '{' && s[s.Length - 1] == '}') ||
              (s[0] == '(' && s[s.Length - 1] == ')'))
            {
                s = s.Substring(1, s.Length - 2);
            }

            // remove optional dashes
            if (s[8] == '-' && s[13] == '-' && s[18] == '-' && s[23] == '-')
            {
                s = s.Substring(0, 8) + s.Substring(9, 4) + s.Substring(14, 4) + s.Substring(19, 4) + s.Substring(24, 12);
            }

            if (s.Length != 32 || !IsHexString(s)) return;
            uuid = HexStringToByteArray(s);
        }

        public override bool Equals(object obj)
        {
            if (obj == null) return !IsValid;

            Uuid rhs = obj as Uuid;

            if ((object)rhs == null) return !IsValid;

            return Equals(rhs);
        }

        public bool Equals(Uuid rhs)
        {
            if ((object)rhs == null) return !IsValid;
            if (!IsValid) return !rhs.IsValid;
            if (rhs.uuid.Length != uuid.Length) return false;
            for (int i = 0; i < uuid.Length; i++) if (rhs.uuid[i] != uuid[i]) return false;
            return true;
        }

        public bool Good()
        {
            return uuid != null && (uuid[6] & 0xf0) == 0x40 && (uuid[8] & 0xc0) == 0x80;
        }

        public bool Bad()
        {
            return !Good();
        }

        public override int GetHashCode()
        {
            if (!IsValid) return base.GetHashCode();
            return BitConverter.ToInt32(uuid, 0) ^
                BitConverter.ToInt32(uuid, 4) ^
                BitConverter.ToInt32(uuid, 8) ^
                BitConverter.ToInt32(uuid, 12);
        }

        public static bool operator ==(Uuid a, Uuid b)
        {
            if ((object)a == null)
            {
                if ((object)b == null) { return true; } else { return !b.IsValid; }
            }
            return a.Equals(b);
        }

        public static bool operator !=(Uuid a, Uuid b)
        {
            return !(a == b);
        }

        public static Uuid NewUuid()
        {
            Guid guid = Guid.NewGuid();
            byte[] bytes = new byte[16];
            Array.Copy(guid.ToByteArray(), bytes, 16);
            if (BitConverter.IsLittleEndian)
            {
                Array.Reverse(bytes, 0, 4);
                Array.Reverse(bytes, 4, 2);
                Array.Reverse(bytes, 6, 2);
            }
            return new Uuid(bytes);
        }

        static bool IsHexString(string s)
        {
            foreach (char ch in s) if (!char.IsDigit(ch) && !(ch >= 'A' && ch <= 'F') && !(ch >= 'a' && ch <= 'f')) return false;
            return true;
        }

        static string HexString(byte[] data, int startIndex, int Length)
        {
            string s = string.Empty;

            for (int i = 0; i < Length; i++) s += data[startIndex + i].ToString("x2");
            return s;
        }

        public static byte[] HexStringToByteArray(string s)
        {
            byte[] bytes = new byte[s.Length / 2];

            for (int i = 0; i < bytes.Length; i++) bytes[i] = (byte)Convert.ToInt16(s.Substring(i * 2, 2), 16);
            return bytes;
        }

        public override string ToString()
        {
            byte[] bytes = IsValid ? uuid : Empty.uuid;
            return HexString(bytes, 0, 4) + "-" + HexString(bytes, 4, 2) + "-" + HexString(bytes, 6, 2) +
              "-" + HexString(bytes, 8, 2) + "-" + HexString(bytes, 10, 6);
        }

        public byte[] ToBytes()
        {
            return (byte[])uuid.Clone();
        }

        public Guid ToGuid()
        {
            byte[] bytes = ToBytes();

            if (BitConverter.IsLittleEndian)
            {
                Array.Reverse(bytes, 0, 4);
                Array.Reverse(bytes, 4, 2);
                Array.Reverse(bytes, 6, 2);
            }
            return new Guid(bytes);
        }
    }

    public class Tsf
    {
        public enum HeaderType { None, Normal, Rev, LittleEndian, BigEndian };
        public enum Status { Data, End, EndOfStream, Error };
        public enum Extension { Default, Null, Redef, FixedArray = 0x11, VarArray = 0x12 };
        public enum Token
        {
            None, End, EndOfStream, Error, EndOfArray, EndOfCollection,
            Data, Collection, Array, Ext
        }
        static byte[] Prefix = new byte[7] { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };
        struct LevelData
        {
            public int Count;
            public int CurrId;
        }

        byte[] Data;
        HeaderType Ht;
        int pos;
        bool reverse;
        public bool Reverse { get { return reverse; } set { reverse = value; } }

        Status status;
        int dataSize;
        int dataCount;
        byte Ext;

        int Level;
        LevelData[] levelData;
        int Count;
        int Id;
        public Tsf() { }
        public Tsf(int size, HeaderType ht = HeaderType.Normal, bool writeHeader = true) { Create(size, ht, writeHeader); }
        public Tsf(byte[] data) { Open(data); }
        public Tsf(byte[] data, int offset, int size) { Open(data, offset, size); }
        public Tsf(string fileName) { Open(fileName); }
        public bool Good { get { return status != Status.Error; } }
        public bool Bad { get { return !Good; } }

        void Init()
        {
            Data = null;
            Ht = HeaderType.None;
            pos = 0;
            reverse = false;
            status = Status.Data;
            dataSize = 0;
            dataCount = 0;
            Ext = 0;
            Level = 0;
            levelData = new LevelData[8];
            Count = 0;
            Id = 0;
        }
        public int DataSize { get { return dataSize; } }
        public int DataCount { get { return dataCount; } }

        public int Pos
        {
            get { return pos; }
            set { pos = value; }
        }

        public void Create(int size, HeaderType ht = HeaderType.Normal, bool writeHeader = true)
        {
            Init();
            Data = new byte[size];

            if (ht == HeaderType.LittleEndian || ht == HeaderType.BigEndian)
            {
                bool isHtLittleEndian = ht == HeaderType.LittleEndian;
                ht = (isHtLittleEndian ^ BitConverter.IsLittleEndian) ? HeaderType.Rev : HeaderType.Normal;
            }
            Ht = ht;

            if (ht != HeaderType.None && size >= 4)
            {
                if (writeHeader)
                {
                    Data[0] = 0x01;
                    Data[1] = 0xe1;
                    if (ht == HeaderType.Rev ^ BitConverter.IsLittleEndian)
                    {
                        Data[2] = 0x74;
                        Data[3] = 0x73;
                    }
                    else
                    {
                        Data[2] = 0x73;
                        Data[3] = 0x74;
                    }
                    pos = 4;
                }
                reverse = ht == HeaderType.Rev;
            }
        }

        public byte[] ToBytes()
        {
            byte[] dest = new byte[pos];
            Array.Copy(Data, dest, pos);
            return dest;
        }

        bool Write(byte b)
        {
            if (status == Status.Error) return false;
            if (pos + 1 >= Data.Length)
            {
                status = Status.Error;
                return false;
            }
            Data[pos++] = b;
            return true;
        }

        bool Write(byte[] src)
        {
            return Write(src, 0, src.Length);
        }

        bool Write(byte[] src, int srcOffset, int count)
        {
            if (status == Status.Error) return false;
            if (pos + count >= Data.Length)
            {
                status = Status.Error;
                return false;
            }
            Array.Copy(src, srcOffset, Data, pos, count);
            pos += count;
            return true;
        }

        bool WriteSorted(byte[] src)
        {
            return WriteSorted(src, 0, src.Length);
        }

        bool WriteSorted(byte[] src, int srcOffset, int count)
        {
            if (reverse)
            {
                if (!Write(src, srcOffset, count)) return false;
                Array.Reverse(Data, pos - count, count);
                return true;
            }
            else
            {
                return Write(src, srcOffset, count);
            }
        }

        bool WriteNumber(ulong num)
        {
            if (num < byte.MaxValue)
            {
                byte[] arr = new byte[1];
                arr[0] = (byte)num;
                return Write(arr, 0, 1);
            }
            else if (num < ushort.MaxValue)
            {
                byte[] arr = BitConverter.GetBytes((ushort)num);

                if (!Write((byte)0xff)) return false;
                return WriteSorted(arr, 0, 2);
            }
            else if (num < uint.MaxValue)
            {
                byte[] arr = BitConverter.GetBytes((uint)num);

                if (!Write(Prefix, 0, 3)) return false;
                return WriteSorted(arr, 0, 4);
            }
            else if (num < ulong.MaxValue)
            {
                byte[] arr = BitConverter.GetBytes(num);

                if (!Write(Prefix, 0, 7)) return false;
                return WriteSorted(arr, 0, 8);
            }

            status = Status.Error;
            return false;
        }

        bool WriteId(int id, int type)
        {
            if (id == 0)
            {
                status = Status.Error;
                return false;
            }

            byte idByte = (byte)((id < 0x1f) ? id : 0x1f);
            idByte = (byte)((idByte << 3) | type);
            if (!Write(idByte)) return false;
            if (id >= 0x1f)
            {
                if (!WriteNumber((ulong)id)) return false;
            }
            return true;
        }

        public bool WriteArray(int id, int size)
        {
            if (size == 0) return false;

            if (!WriteId(id, (size == 1) ? 0x05 : 0x06)) return false;
            if (size > 1) return WriteNumber((ulong)size);
            return true;
        }

        public bool WriteEnd()
        {
            return Write((byte)0);
        }

        public bool Write(int id, byte[] src)
        {
            return Write(id, src, 0, src.Length);
        }

        public bool Write(int id, byte[] src, int offset, int count)
        {
#if TSF_WRITE_VAL0 != true
            uint c;

            for (c = 0; c < count; c++) if (src[offset + c] != 0) break;
            if (c == count) return true;
#endif
            switch (count)
            {
                case 0:
#if TSF_WRITE_SIZE0
#  if TSF_SUPPORT_EXT
                    if (!WriteId(id, 7)) return false;
                    return WriteEnd();
#  else 
                    if (!WriteId(id, 4)) return false;
                    return WriteNumber(0);
#  endif
#else
                    return true;
#endif
                case 1:
                    if (!WriteId(id, 0)) return false;
                    break;
                case 2:
                    if (!WriteId(id, 1)) return false;
                    break;
                case 4:
                    if (!WriteId(id, 2)) return false;
                    break;
                case 8:
                    if (!WriteId(id, 3)) return false;
                    break;
                default:
                    if (!WriteId(id, 4)) return false;
                    if (!WriteNumber((ulong)count)) return false;
                    break;
            }

            return Write(src, offset, count);
        }

        public bool Write0(int id, byte[] src)
        {
            return Write0(id, src, 0, src.Length);
        }

        public bool Write0(int id, byte[] src, int offset, int count)
        {
            switch (count)
            {
                case 0:
#if TSF_WRITE_SIZE0
#  if TSF_SUPPORT_EXT
                    if (!WriteId(id, 7)) return false;
                    return WriteEnd();
#  else 
                    if (!WriteId(id, 4)) return false;
                    return WriteNumber(0);
#  endif
#else
                    return true;
#endif
                case 1:
                    if (!WriteId(id, 0)) return false;
                    break;
                case 2:
                    if (!WriteId(id, 1)) return false;
                    break;
                case 4:
                    if (!WriteId(id, 2)) return false;
                    break;
                case 8:
                    if (!WriteId(id, 3)) return false;
                    break;
                default:
                    if (!WriteId(id, 4)) return false;
                    if (!WriteNumber((ulong)count)) return false;
                    break;
            }

            return Write(src, offset, count);
        }

        public bool SortedWrite(int id, byte[] src)
        {
            return SortedWrite(id, src, 0, src.Length);
        }

        public bool SortedWrite(int id, byte[] src, int offset, int count)
        {
#if TSF_WRITE_VAL0 != true
            uint c;

            for (c = 0; c < count; c++) if (src[offset + c] != 0) break;
            if (c == count) return true;
#endif
            switch (count)
            {
                case 0:
#if TSF_WRITE_SIZE0
#  if TSF_SUPPORT_EXT
                    if (!WriteId(id, 7)) return false;
                    return WriteEnd();
#  else 
                    if (!WriteId(id, 4)) return false;
                    return WriteNumber(0);
#  endif
#else
                    return true;
#endif
                case 1:
                    if (!WriteId(id, 0)) return false;
                    break;
                case 2:
                    if (!WriteId(id, 1)) return false;
                    break;
                case 4:
                    if (!WriteId(id, 2)) return false;
                    break;
                case 8:
                    if (!WriteId(id, 3)) return false;
                    break;
                default:
                    if (!WriteId(id, 4)) return false;
                    if (!WriteNumber((ulong)count)) return false;
                    break;
            }

            return WriteSorted(src, offset, count);
        }

        public bool SortedWrite0(int id, byte[] src, int offset, int count)
        {
            switch (count)
            {
                case 0:
#if TSF_WRITE_SIZE0
#  if TSF_SUPPORT_EXT
                    if (!WriteId(id, 7)) return false;
                    return WriteEnd();
#  else 
                    if (!WriteId(id, 4)) return false;
                    return WriteNumber(0);
#  endif
#else
                    return true;
#endif
                case 1:
                    if (!WriteId(id, 0)) return false;
                    break;
                case 2:
                    if (!WriteId(id, 1)) return false;
                    break;
                case 4:
                    if (!WriteId(id, 2)) return false;
                    break;
                case 8:
                    if (!WriteId(id, 3)) return false;
                    break;
                default:
                    if (!WriteId(id, 4)) return false;
                    if (!WriteNumber((ulong)count)) return false;
                    break;
            }

            return WriteSorted(src, offset, count);
        }

        public bool Write(int id, sbyte val)
        {
            byte[] arr = new byte[1];
            arr[0] = (byte)val;
            return Write(id, arr, 0, 1);
        }

        public bool Write(int id, byte val)
        {
            byte[] arr = new byte[1];
            arr[0] = val;
            return Write(id, arr, 0, 1);
        }

        public bool Write(int id, short val)
        {
            if (val >= sbyte.MinValue && val <= sbyte.MaxValue) return Write(id, (sbyte)val);

            byte[] arr = BitConverter.GetBytes(val);
            return SortedWrite(id, arr, 0, 2);
        }

        public bool Write(int id, ushort val)
        {
            if (val >= byte.MinValue && val <= byte.MaxValue) return Write(id, (byte)val);

            byte[] arr = BitConverter.GetBytes(val);
            return SortedWrite(id, arr, 0, 2);
        }

        public bool Write(int id, int val)
        {
            if (val >= short.MinValue && val <= short.MaxValue) return Write(id, (short)val);

            byte[] arr = BitConverter.GetBytes(val);
            return SortedWrite(id, arr, 0, 4);
        }

        public bool Write(int id, uint val)
        {
            if (val >= ushort.MinValue && val <= ushort.MaxValue) return Write(id, (ushort)val);

            byte[] arr = BitConverter.GetBytes(val);
            return SortedWrite(id, arr, 0, 4);
        }

        public bool Write(int id, long val)
        {
            if (val >= int.MinValue && val <= int.MaxValue) return Write(id, (int)val);

            byte[] arr = BitConverter.GetBytes(val);
            return SortedWrite(id, arr, 0, 8);
        }

        public bool Write(int id, ulong val)
        {
            if (val >= uint.MinValue && val <= uint.MaxValue) return Write(id, (uint)val);

            byte[] arr = BitConverter.GetBytes(val);
            return SortedWrite(id, arr, 0, 8);
        }

        public bool Write(int id, float val)
        {
            byte[] arr = BitConverter.GetBytes(val);
            return SortedWrite(id, arr, 0, 4);
        }

        public bool Write(int id, double val)
        {
            byte[] arr = BitConverter.GetBytes(val);
            return SortedWrite(id, arr, 0, 8);
        }

        public bool Write(int id, Uuid uuid)
        {
            return Write(id, uuid.ToBytes(), 0, 16);
        }

        public void WriteFile(string fileName)
        {
            if (status == Status.Error) throw new FormatException("Invalid TSF document");

            using (FileStream fs = new FileStream(fileName, FileMode.Create))
            {
                fs.Write(Data, 0, pos);
            }
        }

        public bool TryWriteFile(string fileName)
        {
            bool result = true;

            try
            {
                WriteFile(fileName);
            }
            catch
            {
                result = false;
            }

            return result;
        }

        public bool WriteFixedArray(int id, int count, int itemSize)
        {
            dataSize = itemSize;
            if (!WriteId(id, 7)) return false;
            if (!Write((byte)Extension.FixedArray)) return false;
            if (!WriteNumber((ulong)itemSize)) return false;
            return WriteNumber((ulong)count);
        }

        public bool WriteFixedArray(sbyte data)
        {
            switch (dataSize)
            {
                case 1:
                    return Write((byte)data);
                case 2:
                    return WriteSorted(BitConverter.GetBytes((short)data), 0, 2);
                case 4:
                    return WriteSorted(BitConverter.GetBytes((int)data), 0, 4);
                case 8:
                    return WriteSorted(BitConverter.GetBytes((long)data), 0, 8);
            }
            status = Status.Error;
            return false;
        }

        public bool WriteFixedArray(short data)
        {
            switch (dataSize)
            {
                case 1:
                    if (data < sbyte.MinValue || data > sbyte.MaxValue) goto Error;
                    return Write((byte)data);
                case 2:
                    return WriteSorted(BitConverter.GetBytes(data), 0, 2);
                case 4:
                    return WriteSorted(BitConverter.GetBytes((int)data), 0, 4);
                case 8:
                    return WriteSorted(BitConverter.GetBytes((long)data), 0, 8);
            }

        Error:
            status = Status.Error;
            return false;
        }

        public bool WriteFixedArray(int data)
        {
            switch (dataSize)
            {
                case 1:
                    if (data < sbyte.MinValue || data > sbyte.MaxValue) goto Error;
                    return Write((byte)data);
                case 2:
                    if (data < short.MinValue || data > short.MaxValue) goto Error;
                    return WriteSorted(BitConverter.GetBytes((short)data), 0, 2);
                case 4:
                    return WriteSorted(BitConverter.GetBytes(data), 0, 4);
                case 8:
                    return WriteSorted(BitConverter.GetBytes((long)data), 0, 8);
            }

        Error:
            status = Status.Error;
            return false;
        }

        public bool WriteFixedArray(long data)
        {
            switch (dataSize)
            {
                case 1:
                    if (data < sbyte.MinValue || data > sbyte.MaxValue) goto Error;
                    return Write((byte)data);
                case 2:
                    if (data < short.MinValue || data > short.MaxValue) goto Error;
                    return WriteSorted(BitConverter.GetBytes((short)data), 0, 2);
                case 4:
                    if (data < int.MinValue || data > int.MaxValue) goto Error;
                    return WriteSorted(BitConverter.GetBytes((int)data), 0, 4);
                case 8:
                    return WriteSorted(BitConverter.GetBytes(data), 0, 8);
            }

        Error:
            status = Status.Error;
            return false;
        }

        public bool WriteFixedArray(byte data)
        {
            switch (dataSize)
            {
                case 1:
                    return Write(data);
                case 2:
                    return WriteSorted(BitConverter.GetBytes((ushort)data), 0, 2);
                case 4:
                    return WriteSorted(BitConverter.GetBytes((uint)data), 0, 4);
                case 8:
                    return WriteSorted(BitConverter.GetBytes((ulong)data), 0, 8);
            }
            status = Status.Error;
            return false;
        }

        public bool WriteFixedArray(ushort data)
        {
            switch (dataSize)
            {
                case 1:
                    if (data > byte.MaxValue) goto Error;
                    return Write((byte)data);
                case 2:
                    return WriteSorted(BitConverter.GetBytes(data), 0, 2);
                case 4:
                    return WriteSorted(BitConverter.GetBytes((uint)data), 0, 4);
                case 8:
                    return WriteSorted(BitConverter.GetBytes((ulong)data), 0, 8);
            }

        Error:
            status = Status.Error;
            return false;
        }

        public bool WriteFixedArray(uint data)
        {
            switch (dataSize)
            {
                case 1:
                    if (data > byte.MaxValue) goto Error;
                    return Write((byte)data);
                case 2:
                    if (data > ushort.MaxValue) goto Error;
                    return WriteSorted(BitConverter.GetBytes((ushort)data), 0, 2);
                case 4:
                    return WriteSorted(BitConverter.GetBytes(data), 0, 4);
                case 8:
                    return WriteSorted(BitConverter.GetBytes((ulong)data), 0, 8);
            }

        Error:
            status = Status.Error;
            return false;
        }

        public bool WriteFixedArray(ulong data)
        {
            switch (dataSize)
            {
                case 1:
                    if (data > byte.MaxValue) goto Error;
                    return Write((byte)data);
                case 2:
                    if (data > ushort.MaxValue) goto Error;
                    return WriteSorted(BitConverter.GetBytes((ushort)data), 0, 2);
                case 4:
                    if (data > uint.MaxValue) goto Error;
                    return WriteSorted(BitConverter.GetBytes((uint)data), 0, 4);
                case 8:
                    return WriteSorted(BitConverter.GetBytes(data), 0, 8);
            }

        Error:
            status = Status.Error;
            return false;
        }

        public bool WriteFixedArray(float data)
        {
            switch (dataSize)
            {
                case 4:
                    return WriteSorted(BitConverter.GetBytes(data), 0, 4);
                case 8:
                    return WriteSorted(BitConverter.GetBytes((double)data), 0, 8);
            }

            status = Status.Error;
            return false;
        }

        public bool WriteFixedArray(double data)
        {
            switch (dataSize)
            {
                case 4:
                    return WriteSorted(BitConverter.GetBytes((float)data), 0, 4);
                case 8:
                    return WriteSorted(BitConverter.GetBytes(data), 0, 8);
            }

            status = Status.Error;
            return false;
        }

        public bool WriteFixedArray(Uuid data)
        {
            if (dataSize != 16)
            {
                status = Status.Error;
                return false;
            }
            return Write(data.ToBytes(), 0, 16);
        }

        public bool WriteVarArray(int id, int count)
        {
            if (!WriteId(id, 7)) return false;
            if (!Write((byte)Extension.VarArray)) return false;
            return WriteNumber((ulong)count);
        }

        public bool WriteVarArray(byte[] data)
        {
            return WriteVarArray(data, 0, data.Length);
        }

        public bool WriteVarArray(byte[] data, int offset, int size)
        {
            if (!WriteNumber((ulong)size)) return false;
            return Write(data, offset, size);
        }

        public bool WriteVarArraySorted(byte[] data)
        {
            return WriteVarArraySorted(data, 0, data.Length);
        }

        public bool WriteVarArraySorted(byte[] data, int offset, int size)
        {
            if (!WriteNumber((ulong)size)) return false;
            return WriteSorted(data, offset, size);
        }

        public bool WriteVarArray(sbyte data)
        {
            if (!WriteNumber((ulong)1)) return false;
            return Write((byte)data);
        }

        public bool WriteVarArray(short data)
        {
            if (data >= sbyte.MinValue && data <= sbyte.MaxValue)
            {
                return WriteVarArray((sbyte)data);
            }
            if (!WriteNumber((ulong)2)) return false;
            return WriteSorted(BitConverter.GetBytes(data), 0, 2);
        }

        public bool WriteVarArray(int data)
        {
            if (data >= short.MinValue && data <= short.MaxValue)
            {
                return WriteVarArray((short)data);
            }
            if (!WriteNumber((ulong)4)) return false;
            return WriteSorted(BitConverter.GetBytes(data), 0, 4);
        }

        public bool WriteVarArray(long data)
        {
            if (data >= int.MinValue && data <= int.MaxValue)
            {
                return WriteVarArray((int)data);
            }
            if (!WriteNumber((ulong)8)) return false;
            return WriteSorted(BitConverter.GetBytes(data), 0, 8);
        }

        public bool WriteVarArray(byte data)
        {
            if (!WriteNumber((ulong)1)) return false;
            return Write(data);
        }

        public bool WriteVarArray(ushort data)
        {
            if (data <= byte.MaxValue)
            {
                return WriteVarArray((byte)data);
            }
            if (!WriteNumber((ulong)2)) return false;
            return WriteSorted(BitConverter.GetBytes(data), 0, 2);
        }

        public bool WriteVarArray(uint data)
        {
            if (data <= ushort.MaxValue)
            {
                return WriteVarArray((ushort)data);
            }
            if (!WriteNumber((ulong)4)) return false;
            return WriteSorted(BitConverter.GetBytes(data), 0, 4);
        }

        public bool WriteVarArray(ulong data)
        {
            if (data <= uint.MaxValue)
            {
                return WriteVarArray((uint)data);
            }
            if (!WriteNumber((ulong)8)) return false;
            return WriteSorted(BitConverter.GetBytes(data), 0, 8);
        }

        public bool WriteVarArray(float data)
        {
            if (!WriteNumber((ulong)4)) return false;
            return WriteSorted(BitConverter.GetBytes(data), 0, 4);
        }

        public bool WriteVarArray(double data)
        {
            if (!WriteNumber((ulong)8)) return false;
            return WriteSorted(BitConverter.GetBytes(data), 0, 8);
        }

        public bool WriteVarArray(Uuid data)
        {
            if (!WriteNumber((ulong)16)) return false;
            return Write(data.ToBytes(), 0, 16);
        }

        //======================================================================

        public void Open(byte[] data)
        {
            Open(data, 0, data.Length);
        }

        public void Open(byte[] data, int offset, int size)
        {
            Init();
            if (size >= 0)
            {
                this.Data = new byte[size];
                if (size != 0) Array.Copy(data, offset, this.Data, 0, size);
                ReadHeader();
            }
        }

        public void Open(string fileName)
        {
            ReadFile(fileName);
        }

        public bool TryOpen(string fileName)
        {
            bool result = true;

            try
            {
                Open(fileName);
            }
            catch
            {
                result = false;
            }

            return result;
        }

        public void Close()
        {
        }

        void ReadHeader()
        {
            if (Data.Length >= 4)
            {
                if (Data[0] == 0x01 && Data[1] == 0xe1)
                {
                    if (Data[2] == 0x74 && Data[3] == 0x73)
                    {
                        reverse = !BitConverter.IsLittleEndian;
                        pos = 4;
                    }
                    else if (Data[2] == 0x73 && Data[3] == 0x74)
                    {
                        reverse = BitConverter.IsLittleEndian;
                        pos = 4;
                    }
                    Ht = reverse ? HeaderType.Rev : HeaderType.Normal;
                }
            }
        }

        public void ReadFile(string fileName)
        {
            Init();
            using (FileStream fs = new FileStream(fileName, FileMode.Open))
            {
                Data = new byte[fs.Length];
                fs.Read(Data, 0, Data.Length);
                ReadHeader();
            }
        }

        public int Available
        {
            get { return Data.Length - pos; }
        }

        public bool Read(byte[] data, int offset, int size)
        {
            if (status == Status.Error) return false;
            if (size == 0) return true;
            if (Available < size)
            {
                status = Status.Error;
                return false;
            }
            Array.Copy(this.Data, pos, data, offset, size);
            pos += size;
            return true;
        }

        public bool Read(ref byte[] bytes)
        {
            bytes = new byte[DataSize];
            return Read(bytes, 0, DataSize);
        }

        public void SkipData()
        {
            if (Available >= dataSize) pos += dataSize;
        }

        public bool ReadSorted(byte[] data)
        {
            return ReadSorted(data, 0, data.Length);
        }

        public bool ReadSorted(byte[] data, int offset, int size)
        {
            if (!Read(data, offset, size)) return false;
            if (reverse) Array.Reverse(data, offset, size);
            return true;
        }

        bool GetNumber(ref ulong num)
        {
            byte[] tmp = new byte[8];

            if (!Read(tmp, 0, 1)) return false;
            if (tmp[0] != byte.MaxValue)
            {
                num = tmp[0];
                return true;
            }
            if (!ReadSorted(tmp, 0, 2)) return false;

            ushort val16 = BitConverter.ToUInt16(tmp, 0);

            if (val16 != ushort.MaxValue)
            {
                num = val16;
                return true;
            }
            if (!ReadSorted(tmp, 0, 4)) return false;

            uint val32 = BitConverter.ToUInt32(tmp, 0);

            if (val32 != uint.MaxValue)
            {
                num = val32;
                return true;
            }
            if (!ReadSorted(tmp, 0, 8)) return false;

            ulong val64 = BitConverter.ToUInt64(tmp, 0);

            if (val64 != ulong.MaxValue)
            {
                num = val64;
                return true;
            }
            status = Status.Error;
            return false;
        }

        public bool IsAvailable(int size)
        {
            return (Available >= size);
        }

        public Token ReadToken()
        {
            byte[] t = new byte[1];
            byte id;
            ulong num = 0;

            if (status >= Status.End) return (Token)status;

            dataSize = 0;
            Count = 0;
            Ext = 0;
            dataCount = 0;

            for (;;)
            {
                if (!Read(t, 0, 1))
                {
                    if (Level != 0) goto Error;
                    status = Status.EndOfStream;
                    return Token.EndOfStream;
                }
                this.Id = id = (byte)(t[0] >> 3);
                if (id != 0)
                {
                    if (id == 0x1f)
                    {
                        if (!GetNumber(ref num)) goto Error;
                        this.Id = (int)num;
                        if (this.Id < 0x1f || num > int.MaxValue) goto Error;
                    }
                    if (this.Id <= levelData[Level].CurrId) goto Error;
                    levelData[Level].CurrId = this.Id;

                    switch (t[0] & 7)
                    {
                        case 0:
                            if (!IsAvailable(dataSize = 1)) goto Error;
                            return Token.Data;
                        case 1:
                            if (!IsAvailable(dataSize = 2)) goto Error;
                            return Token.Data;
                        case 2:
                            if (!IsAvailable(dataSize = 4)) goto Error;
                            return Token.Data;
                        case 3:
                            if (!IsAvailable(dataSize = 8)) goto Error;
                            return Token.Data;
                        case 4:
                            if (!GetNumber(ref num)) goto Error;
                            if (num > int.MaxValue) goto Error;
                            dataSize = (int)num;
                            if (dataSize == 1 || dataSize == 2 || dataSize == 4 ||
                                dataSize == 8) goto Error;
                            if (!IsAvailable(dataSize)) goto Error;
                            return Token.Data;
                        case 5:
                            levelData[Level].Count = Count = 1;
                            if (++Level == levelData.Length) goto Error;
                            return Token.Collection;
                        case 6:
                            if (!GetNumber(ref num)) goto Error;
                            if (num > int.MaxValue) goto Error;
                            Count = (int)num;
                            if (Count <= 1) goto Error;
                            levelData[Level].Count = Count;
                            if (++Level == levelData.Length) goto Error;
                            return Token.Array;
                        case 7:
                            if (!Read(t, 0, 1)) goto Error;
                            Ext = t[0];
                            if ((Extension)Ext == Extension.FixedArray)
                            {
                                if (!GetNumber(ref num)) goto Error;
                                if (num > int.MaxValue) goto Error;
                                dataSize = (int)num;
                                if (!GetNumber(ref num)) goto Error;
                                if (num > int.MaxValue) goto Error;
                                dataCount = (int)num;
                                if (DataCount == 0) goto Error;
                                if (!IsAvailable(dataSize)) goto Error;
                                return Token.Ext;
                            }
                            else if ((Extension)Ext == Extension.VarArray)
                            {
                                if (!GetNumber(ref num)) goto Error;
                                if (num > int.MaxValue) goto Error;
                                dataCount = (int)num;
                                if (DataCount == 0) goto Error;
                                if (!GetNumber(ref num)) goto Error;
                                if (num > int.MaxValue) goto Error;
                                dataSize = (int)num;
                                return Token.Ext;
                            }
                            else if ((Ext & 0x0f) != 0) goto Error;
                            return Token.Ext;
                    }
                }
                else
                {
                    switch (t[0])
                    {
                        case 0:
                            levelData[Level].CurrId = 0;
                            if (Level != 0)
                            {
                                if (--levelData[Level - 1].Count == 0)
                                {
                                    Level--;
                                    return Token.EndOfArray;
                                }
                                else
                                {
                                    return Token.EndOfCollection;
                                }
                            }
                            else
                            {
                                status = Status.End;
                                return Token.End;
                            }

                        case 2: /* NOP */
                            continue;
                    }
                }
                break;
            }

        Error:
            status = Status.Error;
            return Token.Error;
        }

        public bool SelectNextItem()
        {
            if (dataCount == 0) return false;
            if (--dataCount == 0) return false;

            switch ((Extension)Ext)
            {
                case Extension.VarArray:

                    ulong num = 0;

                    if (!GetNumber(ref num)) goto Error;
                    if (num > int.MaxValue) goto Error;
                    dataSize = (int)num;
                    if (IsAvailable(dataSize)) return true;
                    break;
                case Extension.FixedArray:
                    if (IsAvailable(dataSize)) return true;
                    break;
            }

        Error:
            status = Status.Error;
            return false;
        }

        public bool Read(ref sbyte data)
        {
            long tmp64 = 0;

            if (!Read(ref tmp64)) return false;

            if (tmp64 > sbyte.MaxValue) return false;
            if (tmp64 < sbyte.MinValue) return false;
            data = (sbyte)tmp64;

            return true;
        }

        public bool Read(ref short data)
        {
            long tmp64 = 0;

            if (!Read(ref tmp64)) return false;

            if (tmp64 > short.MaxValue) return false;
            if (tmp64 < short.MinValue) return false;
            data = (short)tmp64;

            return true;
        }

        public bool Read(ref int data)
        {
            long tmp64 = 0;

            if (!Read(ref tmp64)) return false;

            if (tmp64 > int.MaxValue) return false;
            if (tmp64 < int.MinValue) return false;
            data = (int)tmp64;

            return true;
        }

        public bool Read(ref long data)
        {
            byte[] b = new byte[8];

            if (dataSize == 1)
            {
                Read(b, 0, 1);
                data = b[0];
                return true;
            }
            else if (dataSize == 2)
            {
                ReadSorted(b, 0, 2);
                data = BitConverter.ToInt16(b, 0);
                return true;
            }
            else if (dataSize == 4)
            {
                ReadSorted(b, 0, 4);
                data = BitConverter.ToInt32(b, 0);
                return true;
            }
            else if (dataSize == 8)
            {
                ReadSorted(b, 0, 8);
                data = BitConverter.ToInt64(b, 0);
                return true;
            }

            SkipData();
            return false;
        }

        public bool Read(ref byte data)
        {
            ulong tmp64 = 0;

            if (!Read(ref tmp64)) return false;

            if (tmp64 > byte.MaxValue) return false;
            data = (byte)tmp64;

            return true;
        }

        public bool Read(ref ushort data)
        {
            ulong tmp64 = 0;

            if (!Read(ref tmp64)) return false;

            if (tmp64 > ushort.MaxValue) return false;
            data = (ushort)tmp64;

            return true;
        }

        public bool Read(ref uint data)
        {
            ulong tmp64 = 0;

            if (!Read(ref tmp64)) return false;

            if (tmp64 > uint.MaxValue) return false;
            data = (uint)tmp64;

            return true;
        }

        public bool Read(ref ulong data)
        {
            byte[] b = new byte[8];

            if (dataSize == 1)
            {
                Read(b, 0, 1);
                data = b[0];
                return true;
            }
            else if (dataSize == 2)
            {
                ReadSorted(b, 0, 2);
                data = BitConverter.ToUInt16(b, 0);
                return true;
            }
            else if (dataSize == 4)
            {
                ReadSorted(b, 0, 4);
                data = BitConverter.ToUInt32(b, 0);
                return true;
            }
            else if (dataSize == 8)
            {
                ReadSorted(b, 0, 8);
                data = BitConverter.ToUInt64(b, 0);
                return true;
            }

            SkipData();
            return false;
        }

        public bool Read(ref float data)
        {
            byte[] b = new byte[8];

            if (dataSize == 4)
            {
                ReadSorted(b, 0, 4);
                data = BitConverter.ToSingle(b, 0);
                return true;
            }
            else if (dataSize == 8)
            {
                ReadSorted(b, 0, 8);
                data = (float)BitConverter.ToDouble(b, 0);
                return true;
            }
            else if (dataSize == 1)
            {
                ReadSorted(b, 0, 1);
                data = (sbyte)b[0];
                return true;
            }
            else if (dataSize == 2)
            {
                ReadSorted(b, 0, 2);
                data = BitConverter.ToInt16(b, 0);
                return true;
            }

            SkipData();
            return false;
        }

        public bool Read(ref double data)
        {
            byte[] b = new byte[8];

            if (dataSize == 8)
            {
                ReadSorted(b, 0, 8);
                data = BitConverter.ToDouble(b, 0);
                return true;
            }
            else if (dataSize == 4)
            {
                ReadSorted(b, 0, 4);
                data = BitConverter.ToSingle(b, 0);
                return true;
            }
            else if (dataSize == 1)
            {
                ReadSorted(b, 0, 1);
                data = (sbyte)b[0];
                return true;
            }
            else if (dataSize == 2)
            {
                ReadSorted(b, 0, 2);
                data = BitConverter.ToInt16(b, 0);
                return true;
            }

            SkipData();
            return false;
        }

        public bool Read(ref Uuid data)
        {
            byte[] b = new byte[16];

            if (dataSize == 16)
            {
                Read(b, 0, 16);
                data = new Uuid(b);
                return true;
            }

            SkipData();
            return false;
        }

        public bool Select(int id)
        {
            int level;
            int pos;

            if (status >= Status.End) return false;
            if (DataCount != 0) while (SelectNextItem()) SkipData();

            level = this.Level;

            for (;;)
            {
                pos = Pos;
                switch (ReadToken())
                {
                    case Token.Ext:
                        goto case Token.Data;
                    case Token.Data:
                        if (this.Level == level && this.Id >= id)
                        {
                            if (this.Id == id) return true;
                            Pos = pos;
                            dataCount = 0;
                            levelData[this.Level].CurrId = 0;
                            return false;
                        }
                        do
                        {
                            SkipData();
                        } while (SelectNextItem());
                        continue;
                    case Token.Collection:
                        goto case Token.Array;
                    case Token.Array:
                        if (this.Level == level + 1 && this.Id >= id)
                        {
                            Pos = pos;
                            --this.Level;
                            levelData[this.Level].CurrId = 0;
                            return false;
                        }
                        continue;
                    case Token.EndOfCollection:
                        if (this.Level == level)
                        {
                            Pos = pos;
                            levelData[this.Level - 1].Count++;
                            levelData[this.Level].CurrId = 0;
                            return false;
                        }
                        continue;
                    case Token.EndOfArray:
                        if (this.Level + 1 == level)
                        {
                            Pos = pos;
                            levelData[this.Level++].Count++;
                            levelData[this.Level].CurrId = 0;
                            return false;
                        }
                        continue;
                }
                return false;
            }
        }

        public bool SelectObject(int id)
        {
            return Select(id);
        }

        public int ReadArraySize(int id)
        {
            int level;
            int pos;

            if (status >= Status.End) return 0;
            if (DataCount != 0) while (SelectNextItem()) SkipData();

            level = this.Level;

            for (;;)
            {
                pos = Pos;
                switch (ReadToken())
                {
                    case Token.Ext:
                        goto case Token.Data;
                    case Token.Data:
                        if (this.Level == level && this.Id >= id)
                        {
                            Pos = pos;
                            dataCount = 0;
                            levelData[this.Level].CurrId = 0;
                            return 0;
                        }
                        do
                        {
                            SkipData();
                        } while (SelectNextItem());
                        continue;
                    case Token.Collection:
                        goto case Token.Array;
                    case Token.Array:
                        if (this.Level == level + 1 && this.Id >= id)
                        {
                            if (this.Id == id) return Count;
                            Pos = pos;
                            --this.Level;
                            levelData[this.Level].CurrId = 0;
                            return 0;
                        }
                        continue;
                    case Token.EndOfCollection:
                        if (this.Level == level)
                        {
                            Pos = pos;
                            levelData[this.Level - 1].Count++;
                            levelData[this.Level].CurrId = 0;
                            return 0;
                        }
                        continue;
                    case Token.EndOfArray:
                        if (this.Level + 1 == level)
                        {
                            Pos = pos;
                            levelData[this.Level++].Count++;
                            levelData[this.Level].CurrId = 0;
                            return 0;
                        }
                        continue;
                }
                return 0;
            }
        }

        public bool SelectArray(int id)
        {
            return ReadArraySize(id) != 0;
        }

        bool SkipCollectionOrArray(bool skipCollectionOnly)
        {
            int level;

            if (status >= Status.End) return false;
            if (DataCount != 0) while (SelectNextItem()) SkipData();

            level = this.Level;

            for (;;)
            {
                switch (ReadToken())
                {
                    case Token.Ext:
                        goto case Token.Data;
                    case Token.Data:
                        do
                        {
                            SkipData();
                        } while (SelectNextItem());
                        continue;
                    case Token.Collection:
                        goto case Token.Array;
                    case Token.Array:
                        continue;
                    case Token.EndOfCollection:
                        if (skipCollectionOnly && this.Level == level) return true;
                        continue;
                    case Token.EndOfArray:
                        if (this.Level + 1 == level) return true;
                        continue;
                }
                return false;
            }
        }

        public bool SkipCollection()
        {
            return SkipCollectionOrArray(true);
        }

        public bool SkipArray()
        {
            return SkipCollectionOrArray(false);
        }
    }
}