/*
 * -----------------------------------------------------------------------------
 * Tagged Stream Format V1.1.19
 * Copyright (c) 2017-2019 Rudy Tellert Elektronik
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy 
 * of this software and associated documentation files (the "Software"), to deal 
 * in the Software without restriction, including without limitation the rights 
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 * SOFTWARE.
 * -----------------------------------------------------------------------------
 */

#ifndef __TSFCLASS_H
#define __TSFCLASS_H

#include "tsf.h"

/*- Additional switches for target platform ----------------------------------*/

#define TSF_USE_DYNAMIC_MEMORY /* Support "new", "delete" and std::vector */
#define TSF_SUPPORT_BYTES /* Support for TsfBytes */
//#define TSF_USE_FSTREAM /* Use fstream, otherwise fallback to TSF C API */

/*----------------------------------------------------------------------------*/

#ifdef TSF_SUPPORT_BYTES
#ifdef TSF_USE_DYNAMIC_MEMORY
#include <vector>
typedef std::vector<TsfByte> TsfBytes;
#elif defined(ETL_VECTOR_INCLUDED) && defined(TSF_BYTES_CAPACITY)
/* see also https://www.etlcpp.com */
typedef etl::vector<TsfByte, TSF_BYTES_CAPACITY> TsfBytes;
#else
#undef TSF_SUPPORT_BYTES
#endif
#endif

class Tsf {
    TsfObject *tsf;
    char *data;
    size_t dataSize;
    bool isOpen;
    void Init(void *buffer, size_t bufferSize);
    bool BufferIsUsed() const { return dataSize != 0; }
    bool DataDeletionRequired() const { return data != NULL && dataSize == 0; }

public:
    enum HeaderType { NoneHeader, NormalHeader, RevHeader };
    
    enum TokenType {
        NoneToken, EndToken, EndOfStreamToken, ErrorToken,
        EndOfArrayToken, EndOfCollectionToken,
        DataToken,
        CollectionToken, ArrayToken,
        ExtToken
    };

    enum ExtType {
        DefaultExt = 0, NullExt = 1, ReDefExt = 2, FixedArrayExt = 0x11, VarArrayExt = 0x12
    };

    enum StatusType {
        Data, End, EndOfStream, Error
    };

#ifdef TSF_SUPPORT_UUID
    class Uuid {
        TsfUuid uuid;
    public:
        Uuid() { Empty(); }
        Uuid(const void *data) { TsfUuidFromData(uuid, data); }
        Uuid(const char *data) { TsfUuidFromString(uuid, data); }
#ifdef TSF_SUPPORT_WCHAR_T
        Uuid(const wchar_t *data) { TsfUuidFromWideString(uuid, data); }
#endif
        Uuid(TsfUInt64 nonce) { TsfUuidFromNonce(uuid, nonce); }
        void Set(const void *data) { TsfUuidFromData(uuid, data); }
        void Set(const char *data) { TsfUuidFromString(uuid, data); }
#ifdef TSF_SUPPORT_WCHAR_T
        void Set(const wchar_t *data) { TsfUuidFromWideString(uuid, data); }
#endif
        void Set(unsigned nonce) { TsfUuidFromNonce(uuid, nonce); }
        void Empty() { TsfUuidEmpty(uuid); }
#ifdef TSF_SUPPORT_WIN32
        void New() { TsfUuidNew(uuid); }
#endif
        bool Good() const { return TsfUuidIsGood(uuid) != TSF_FALSE; }
        bool Bad() const { return TsfUuidIsBad(uuid) != TSF_FALSE; }
        bool IsEqual(const Uuid &rhs) const { return TsfUuidIsEqual(uuid, rhs.uuid) != TSF_FALSE; }
        const void *Get() const { return (const void*)uuid; }
        operator const void *() const { return Get(); }

#ifdef TSF_SUPPORT_OUTPUT
        bool TsfWrite(TsfObject *tsf, int id) const { return TsfWriteUuid(tsf, id, uuid) != TSF_FALSE; }
#ifdef TSF_SUPPORT_SIMPLE_ARRAY
        bool TsfWriteFixedArray(TsfObject *tsf) const { return TsfWriteFixedArrayUuid(tsf, uuid) != TSF_FALSE; }
        bool TsfWriteVarArray(TsfObject *tsf) const { return TsfWriteVarArrayUuid(tsf, uuid) != TSF_FALSE; }
#endif
#endif
    };
#endif

#ifdef TSF_USE_DYNAMIC_MEMORY
    Tsf(void *buffer = NULL, size_t bufferSize = 0) { Init(buffer, bufferSize); }
#else
    Tsf(void *buffer, size_t bufferSize) { Init(buffer, bufferSize); }
#endif
#ifdef TSF_SUPPORT_OUTPUT
#ifdef TSF_USE_DYNAMIC_MEMORY
    Tsf(size_t size, HeaderType ht = HeaderType::NormalHeader, bool writeHeader = true) { Init(NULL, 0); Create(size, ht, writeHeader); };
#endif
    Tsf(HeaderType ht, bool writeHeader, void *buffer, size_t bufferSize) { Init(buffer, bufferSize); Create(0, ht, writeHeader); };
#endif
#ifdef TSF_SUPPORT_INPUT
#ifdef TSF_USE_DYNAMIC_MEMORY
    Tsf(const void *data, size_t size, void *buffer = NULL, size_t bufferSize = 0) { Init(buffer, bufferSize); Open((const char *)data, size); }
    Tsf(const TSF_TCHAR *fileName, void *buffer = NULL, size_t bufferSize = 0) { Init(buffer, bufferSize); Open(fileName); }
#else
    Tsf(const void *data, size_t size, void *buffer, size_t bufferSize) { Init(buffer, bufferSize); Open((const char *)data, size); }
    Tsf(const TSF_TCHAR *fileName, void *buffer, size_t bufferSize) { Init(buffer, bufferSize); Open(fileName); }
#endif
#endif
    ~Tsf() { 
        Close(); 
#ifdef TSF_USE_DYNAMIC_MEMORY
        if (!BufferIsUsed()) delete[] (char*)tsf; 
#endif
    }
    void Close();

    StatusType GetStatus() const { return (StatusType)TsfGetStatus(tsf); }
    bool Good() const { return TsfIsGood(tsf) != TSF_FALSE; }
    bool Bad() const { return TsfIsBad(tsf) != TSF_FALSE; }
    bool Reverse() const { return TsfIsReverse(tsf) != TSF_FALSE; }
    HeaderType GetHeaderType() const { return (HeaderType)TsfGetHeaderType(tsf); }
    TsfId GetId() const { return TsfGetId(tsf); }
    size_t GetCount() const { return TsfGetCount(tsf); }
    TsfLevel GetLevel() const { return TsfGetLevel(tsf); }
    TsfByte GetExt() const { return TsfGetExt(tsf); }
    const void *GetData() const { return TsfGetData(tsf); }
    const char *GetDataChars() const { return TsfGetDataChars(tsf); }
    const TsfByte *GetDataBytes() const { return TsfGetDataBytes(tsf); }
    size_t GetDataSize() const { return TsfGetDataSize(tsf); }
#ifdef TSF_SUPPORT_SIMPLE_ARRAY
    size_t GetDataCount() const { return TsfGetDataCount(tsf); }
#endif
#ifdef TSF_SUPPORT_REV_ACCESS
    bool GetReverse() const { return TsfGetReverse(tsf) != TSF_FALSE; }
    void SetReverse(bool isRev) { TsfSetReverse(tsf, isRev ? TSF_TRUE : TSF_FALSE); }
#endif

#ifdef TSF_SUPPORT_INPUT
    size_t GetPos() const { return TsfGetPos(tsf); }
    bool SetPos(size_t pos) { return TsfSetPos(tsf, pos) != TSF_FALSE; }
    void Open(const void *data, size_t size) { Close(); TsfOpen(tsf, (const char *)data, size); }
    bool Open(const TSF_TCHAR *fileName);
    TokenType ReadToken() { return (TokenType)TsfReadToken(tsf); }
#ifdef TSF_SUPPORT_SIMPLE_ARRAY
    bool SelectNextItem() { return TsfSelectNextItem(tsf) != TSF_FALSE; }
#endif
    bool Read(void *data, size_t size) { return TsfReadData(tsf, data, size) != TSF_FALSE; }
    bool ReadObject(void *data, size_t size) { return TsfReadObject(tsf, data, size) != TSF_FALSE; }
#ifdef TSF_SUPPORT_BYTES
    bool Read(TsfBytes &data) {
        data.resize(GetDataSize());
        return Read(&data.at(0), GetDataSize());
    }
#endif
    bool Skip() { return TsfSkipData(tsf) != TSF_FALSE; }
    void Sort(void *data, size_t size) { TsfSortData(tsf, data, size); }
    bool ReadSorted(void *data, size_t size) { return TsfReadSortedData(tsf, data, size) != TSF_FALSE; }
    bool ReadSortedObject(void *data, size_t size) { return TsfReadSortedObject(tsf, data, size) != TSF_FALSE; }
#ifdef TSF_SUPPORT_UINT
    bool Read(TsfUInt8 &data) { return TsfReadUInt8(tsf, &data) != TSF_FALSE; }
    bool Read(TsfUInt16 &data) { return TsfReadUInt16(tsf, &data) != TSF_FALSE; }
    bool Read(TsfUInt32 &data) { return TsfReadUInt32(tsf, &data) != TSF_FALSE; }
    bool Read(TsfUInt64 &data) { return TsfReadUInt64(tsf, &data) != TSF_FALSE; }
#endif
#ifdef TSF_SUPPORT_INT
    bool Read(TsfInt8 &data) { return TsfReadInt8(tsf, &data) != TSF_FALSE; }
    bool Read(TsfInt16 &data) { return TsfReadInt16(tsf, &data) != TSF_FALSE; }
    bool Read(TsfInt32 &data) { return TsfReadInt32(tsf, &data) != TSF_FALSE; }
    bool Read(TsfInt64 &data) { return TsfReadInt64(tsf, &data) != TSF_FALSE; }
#endif
#ifdef TSF_SUPPORT_FLOAT
    bool Read(float &data) { return TsfReadFloat32(tsf, &data) != TSF_FALSE; }
    bool Read(double &data) { return TsfReadFloat64(tsf, &data) != TSF_FALSE; }
#endif
#ifdef TSF_SUPPORT_UUID
    bool Read(Uuid &data) { return TsfReadUuid(tsf, (TsfUuid*)data.Get()) != TSF_FALSE; }
#endif
    bool SelectObject(int id) { return TsfSelectObject(tsf, id) != TSF_FALSE; }
    bool Select(int id) { return TsfSelect(tsf, id) != TSF_FALSE; }
    size_t ReadArraySize(int id) { return TsfReadArraySize(tsf, id); }
    bool SelectArray(int id) { return TsfSelectArray(tsf, id) != TSF_FALSE; }
    bool SkipArray() { return TsfSkipArray(tsf) != TSF_FALSE; }
    bool SkipCollection() { return TsfSkipCollection(tsf) != TSF_FALSE; }
#endif

#ifdef TSF_SUPPORT_OUTPUT
    size_t DataSize() const { return TsfDataSize(tsf); }
    const TsfByte *DataBase() const { return TsfDataBase(tsf); }
    bool WriteFile(const TSF_TCHAR *fileName) const;
    bool Create(size_t size, HeaderType ht = HeaderType::NormalHeader, bool writeHeader = true);
    bool Write(const char *data, size_t size) { return TsfWriteData(tsf, data, size) != TSF_FALSE; }
    bool WriteSorted(const char *data, size_t size) { return TsfWriteSortedData(tsf, data, size) != TSF_FALSE; }
    bool Write(int id, const void *data, size_t size) { return TsfWriteObject(tsf, id, data, size) != TSF_FALSE; }
#ifdef TSF_SUPPORT_BYTES
    bool Write(int id, TsfBytes data) { return Write(id, &data.at(0), data.size()); }
#endif
    bool Write0(int id, const void *data, size_t size) { return TsfWriteObject0(tsf, id, data, size) != TSF_FALSE; }
    bool WriteSorted(int id, const void *data, size_t size) { return TsfWriteSortedObject(tsf, id, data, size) != TSF_FALSE; }
    bool WriteSorted0(int id, const void *data, size_t size) { return TsfWriteSortedObject0(tsf, id, data, size) != TSF_FALSE; }
    bool WriteArray(int id, size_t size) { return TsfWriteArray(tsf, id, size) != TSF_FALSE; }
    bool WriteEnd() { return TsfWriteEnd(tsf) != TSF_FALSE; }
#ifdef TSF_SUPPORT_UINT
    bool Write(int id, TsfUInt8 data) { return TsfWriteUInt8(tsf, id, data) != TSF_FALSE; }
    bool Write(int id, TsfUInt16 data) { return TsfWriteUInt16(tsf, id, data) != TSF_FALSE; }
    bool Write(int id, TsfUInt32 data) { return TsfWriteUInt32(tsf, id, data) != TSF_FALSE; }
    bool Write(int id, TsfUInt64 data) { return TsfWriteUInt64(tsf, id, data) != TSF_FALSE; }
#endif
#ifdef TSF_SUPPORT_INT
    bool Write(int id, TsfInt8 data) { return TsfWriteInt8(tsf, id, data) != TSF_FALSE; }
    bool Write(int id, TsfInt16 data) { return TsfWriteInt16(tsf, id, data) != TSF_FALSE; }
    bool Write(int id, TsfInt32 data) { return TsfWriteInt32(tsf, id, data) != TSF_FALSE; }
    bool Write(int id, TsfInt64 data) { return TsfWriteInt64(tsf, id, data) != TSF_FALSE; }
#endif
#ifdef TSF_SUPPORT_FLOAT
    bool Write(int id, float data) { return TsfWriteFloat32(tsf, id, data) != TSF_FALSE; }
    bool Write(int id, double data) { return TsfWriteFloat64(tsf, id, data) != TSF_FALSE; }
#endif
#ifdef TSF_SUPPORT_UUID
    bool Write(int id, Uuid data) { return data.TsfWrite(tsf, id); }
#endif
#ifdef TSF_SUPPORT_SIMPLE_ARRAY
    bool WriteFixedArray(int id, size_t count, size_t itemSize) { return TsfWriteFixedArray(tsf, id, count, itemSize) != TSF_FALSE; }
    bool WriteFixedArrayData(const void *data) { return TsfWriteFixedArrayData(tsf, data) != TSF_FALSE; }
    bool WriteFixedArraySortedData(const void *data) { return TsfWriteFixedArraySortedData(tsf, data) != TSF_FALSE; }
#  ifdef TSF_SUPPORT_UINT
    bool WriteFixedArray(TsfUInt8 data) { return TsfWriteFixedArrayUInt8(tsf, data) != TSF_FALSE; }
    bool WriteFixedArray(TsfUInt16 data) { return TsfWriteFixedArrayUInt16(tsf, data) != TSF_FALSE; }
    bool WriteFixedArray(TsfUInt32 data) { return TsfWriteFixedArrayUInt32(tsf, data) != TSF_FALSE; }
    bool WriteFixedArray(TsfUInt64 data) { return TsfWriteFixedArrayUInt64(tsf, data) != TSF_FALSE; }
#  endif
#  ifdef TSF_SUPPORT_INT
    bool WriteFixedArray(TsfInt8 data) { return TsfWriteFixedArrayInt8(tsf, data) != TSF_FALSE; }
    bool WriteFixedArray(TsfInt16 data) { return TsfWriteFixedArrayInt16(tsf, data) != TSF_FALSE; }
    bool WriteFixedArray(TsfInt32 data) { return TsfWriteFixedArrayInt32(tsf, data) != TSF_FALSE; }
    bool WriteFixedArray(TsfInt64 data) { return TsfWriteFixedArrayInt64(tsf, data) != TSF_FALSE; }
#  endif
#  ifdef TSF_SUPPORT_FLOAT
    bool WriteFixedArray(float data) { return TsfWriteFixedArrayFloat32(tsf, data) != TSF_FALSE; }
    bool WriteFixedArray(double data) { return TsfWriteFixedArrayFloat64(tsf, data) != TSF_FALSE; }
#  endif
#  ifdef TSF_SUPPORT_UUID
    bool WriteFixedArray(Uuid data) { return data.TsfWriteFixedArray(tsf); }
#  endif
    bool WriteVarArray(int id, size_t count) { return TsfWriteVarArray(tsf, id, count) != TSF_FALSE; }
    bool WriteVarArrayData(const void *data, size_t size) { return TsfWriteVarArrayData(tsf, data, size) != TSF_FALSE; }
    bool WriteVarArraySortedData(const void *data, size_t size) { return TsfWriteVarArraySortedData(tsf, data, size) != TSF_FALSE; }
#  ifdef TSF_SUPPORT_UINT
    bool WriteVarArray(TsfUInt8 data) { return TsfWriteVarArrayUInt8(tsf, data) != TSF_FALSE; }
    bool WriteVarArray(TsfUInt16 data) { return TsfWriteVarArrayUInt16(tsf, data) != TSF_FALSE; }
    bool WriteVarArray(TsfUInt32 data) { return TsfWriteVarArrayUInt32(tsf, data) != TSF_FALSE; }
    bool WriteVarArray(TsfUInt64 data) { return TsfWriteVarArrayUInt64(tsf, data) != TSF_FALSE; }
#  endif
#  ifdef TSF_SUPPORT_INT
    bool WriteVarArray(TsfInt8 data) { return TsfWriteVarArrayInt8(tsf, data) != TSF_FALSE; }
    bool WriteVarArray(TsfInt16 data) { return TsfWriteVarArrayInt16(tsf, data) != TSF_FALSE; }
    bool WriteVarArray(TsfInt32 data) { return TsfWriteVarArrayInt32(tsf, data) != TSF_FALSE; }
    bool WriteVarArray(TsfInt64 data) { return TsfWriteVarArrayInt64(tsf, data) != TSF_FALSE; }
#  endif
#  ifdef TSF_SUPPORT_FLOAT
    bool WriteVarArray(float data) { return TsfWriteVarArrayFloat32(tsf, data) != TSF_FALSE; }
    bool WriteVarArray(double data) { return TsfWriteVarArrayFloat64(tsf, data) != TSF_FALSE; }
#  endif
#  ifdef TSF_SUPPORT_UUID
    bool WriteVarArray(Uuid data) { return data.TsfWriteVarArray(tsf); }
#  endif
#endif
#endif
};

inline bool operator == (const Tsf::Uuid &lhs, const Tsf::Uuid &rhs)
{
    return lhs.IsEqual(rhs);
}
inline bool operator != (const Tsf::Uuid &lhs, const Tsf::Uuid &rhs)
{
    return !(lhs == rhs);
}

#endif
