/*
 * -----------------------------------------------------------------------------
 * Tagged Stream Format V1.1.19
 * Copyright (c) 2017-2019 Rudy Tellert Elektronik
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy 
 * of this software and associated documentation files (the "Software"), to deal 
 * in the Software without restriction, including without limitation the rights 
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 * SOFTWARE.
 * -----------------------------------------------------------------------------
 */

#include "tsf_priv.h"
#include "tsfclass.h"

void Tsf::Init(void *buffer, size_t bufferSize) 
{
    isOpen = false;
    if (bufferSize == 0) {
#ifdef TSF_USE_DYNAMIC_MEMORY
        tsf = (TsfObject*) new char[sizeof(TsfObject)];
        if (tsf) memset(tsf, '\0', sizeof(TsfObject));
#else
        tsf = NULL;
#endif
        data = NULL;
        dataSize = 0;
    }
    else {
        size_t tsfSize = sizeof(TsfObject);
        if (buffer == NULL || tsfSize >= bufferSize) {
            tsf = NULL;
            data = NULL;
            dataSize = 0;
        }
        else {
            tsf = (TsfObject*)buffer;
            data = (char*)buffer + tsfSize;
            dataSize = bufferSize - tsfSize;
            memset(buffer, '\0', bufferSize);
        }
    }
}

void Tsf::Close() 
{
#ifdef TSF_USE_DYNAMIC_MEMORY
    if (DataDeletionRequired()) {
        delete[] data;
        data = NULL;
    }
#endif
    if (isOpen) {
        isOpen = false;
        TsfAssert(tsf);
        TsfClose(tsf);
        memset(tsf, '\0', sizeof(TsfObject));
    }
}

#ifdef TSF_SUPPORT_OUTPUT
bool Tsf::Create(size_t size, Tsf::HeaderType ht, bool writeHeader) 
{
    if (tsf == NULL) return false;
    Close();
    if (BufferIsUsed()) {
        if (size == 0) {
            // size == 0: keep current data region
            size = dataSize;
        }
        else if (dataSize < size) {
            return false;
        }
    }
    else {
#ifdef TSF_USE_DYNAMIC_MEMORY
        if (size == 0) return false;
        data = new char[size];
        if (data == NULL) return false;
        memset(data, '\0', size);
#else
        return false;
#endif
    }
    TsfCreate(tsf, data, size, ht, writeHeader);
    isOpen = true;

    return true;
}
#endif

#ifndef TSF_USE_FSTREAM

#ifdef TSF_SUPPORT_INPUT
bool Tsf::Open(const TSF_TCHAR *fileName)
{
    size_t size = 0;

    if (tsf == NULL) return false;
    Close();
    TsfReadFile(fileName, NULL, 0, &size);
    if (size == 0) return false;

    if (BufferIsUsed()) {
        if (size > dataSize) return false;
        memset(data, '\0', dataSize);
    } 
    else {
#ifdef TSF_USE_DYNAMIC_MEMORY
        data = new char[size];
        if (data == NULL) return false;
        memset(data, '\0', size);
#else
        return false;
#endif
    }
    isOpen = TsfOpenFile(tsf, fileName, data, size) != TSF_FALSE;
    
    return isOpen;
}
#endif

#ifdef TSF_SUPPORT_OUTPUT
bool Tsf::WriteFile(const TSF_TCHAR *fileName) const
{
    if (data == NULL || tsf == NULL) return false;

    return (TsfWriteFile(tsf, fileName)) != TSF_FALSE;
}
#endif

#else

#include <iostream>
#include <fstream>
#include <limits>

#ifdef TSF_SUPPORT_INPUT
bool Tsf::Open(const TSF_TCHAR *fileName)
{
    std::ifstream fs;
    bool result = true;

    if (tsf == NULL) return false;
    Close();
    fs.open(fileName, std::ios_base::in | std::ios_base::binary);
    if (fs.good()) {
        fs.ignore(std::numeric_limits<std::streamsize>::max());
        std::streamsize length = fs.gcount();
        fs.clear();
        fs.seekg(0, std::ios_base::beg);
        size_t size = (size_t)length;
        if (size && length == (std::streamsize)size) {
            fs.seekg(0, std::ios_base::beg);
            if (BufferIsUsed()) {
                if (size <= dataSize) {
                    memset(data, '\0', dataSize);
                    fs.read(data, size);
                }
                else {
                    result = false;
                }
            }
            else {
#ifdef TSF_USE_DYNAMIC_MEMORY
                data = new char[size];
                if (data) {
                    memset(data, '\0', size);
                    fs.read(data, size);
                }
                else {
                    result = false;
                }
#else
                result = false;
#endif
            }
            if (fs.bad()) result = false;
            fs.close();
            if (result) {
                TsfOpen(tsf, data, size);
                isOpen = true;
            }
        }
        else {
            fs.close();
            result = false;
        }
    }
    else {
        result = false;
    }

    return result;
}
#endif

#ifdef TSF_SUPPORT_OUTPUT
bool Tsf::WriteFile(const TSF_TCHAR *fileName) const
{
    std::ofstream fs;
    bool result = true;

    TsfStdAssert(tsf);
    if (data == NULL) return false;
    if (GetStatus() == StatusType::Error) return false;

    fs.open(fileName, std::ios_base::out | std::ios_base::binary);
    if (fs.good()) {
        fs.write(data, DataSize());
        if (fs.bad()) result = false;
        fs.close();
    }
    else {
        result = false;
    }

    return result;
}
#endif

#endif
