/*
 * -----------------------------------------------------------------------------
 * Tagged Stream Format V1.1.19
 * Copyright (c) 2017-2019 Rudy Tellert Elektronik
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy 
 * of this software and associated documentation files (the "Software"), to deal 
 * in the Software without restriction, including without limitation the rights 
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 * SOFTWARE.
 * -----------------------------------------------------------------------------
 */

/* tsfwrite.c is endian neutral */

#define TSF_POINTER_MODIFIER
#include "tsf_priv.h"

#ifdef TSF_SUPPORT_OUTPUT

/*- stream access ------------------------------------------------------------*/

void TSF_API TsfCreate(TsfObject *tsf, void *data, size_t availableSize, TsfHeaderType h, TsfBool writeHeader)
{
    TsfInit(tsf);
    TsfAssert(data);

#ifdef TSF_SUPPORT_REV_CHECK
    if (h >= TSFH_LITTLE_ENDIAN) {
        TsfBool htIsLittleEndian = (h == TSFH_LITTLE_ENDIAN) ? TSF_TRUE : TSF_FALSE;
        h = (htIsLittleEndian ^ TsfIsLittleEndian()) ? TSFH_REV : TSFH_NORMAL;
    }
#endif
#ifdef TSF_SUPPORT_REV
    TsfAssert(h <= TSFH_REV);
#endif

    tsf->Data = (TsfByte*)data;
    tsf->RemainingSize = tsf->TotalSize = availableSize;

    if (h) {
        if (writeHeader) {
            tsf->Data[0] = TSF_HEADER0;
            tsf->Data[1] = TSF_HEADER1;
            *(unsigned short*)(tsf->Data + 2) =
                (h == TSFH_NORMAL) ? TSF_HEADER2 : TSF_HEADER2_REV;
            tsf->Data += 4;
            tsf->RemainingSize -= 4;
        }
        tsf->Header = h;
#ifdef TSF_SUPPORT_REV_FIELD
        if (h == TSFH_REV) tsf->Reverse = TSF_TRUE;
#endif
    }
}

TsfBool TSF_API TsfWriteData(TsfObject *tsf, const void *data, size_t size)
{
    TsfStdAssert(tsf);
    TsfAssert(data);

    if (tsf->Status == TSFS_ERROR) return TSF_FALSE;

    if (size > tsf->RemainingSize) {
        tsf->Status = TSFS_ERROR;
        return TSF_FALSE;
    }

    memcpy(tsf->Data, data, size);
    tsf->RemainingSize -= size;
    tsf->Data += size;

    return TSF_TRUE;
}

#ifdef TSF_SUPPORT_REV
TsfBool TSF_API TsfWriteSortedData(TsfObject *tsf, const void *data, size_t size)
{
    TsfStdAssert(tsf);
    TsfAssert(data);

    if (tsf->Status == TSFS_ERROR) return TSF_FALSE;

    if (size > tsf->RemainingSize) {
        tsf->Status = TSFS_ERROR;
        return TSF_FALSE;
    }

    memcpy(tsf->Data, data, size);

#ifdef TSF_SUPPORT_REV_FIELD
    if (tsf->Reverse) TsfReverse(tsf->Data, size);
#else
    if (tsf->Header == TSFH_REV) TsfReverse(tsf->Data, size);
#endif

    tsf->RemainingSize -= size;
    tsf->Data += size;

    return TSF_TRUE;
}
#endif

/*- layer 1 ------------------------------------------------------------------*/

static TsfBool TSF_API TsfWriteNumber(TsfObject *tsf, size_t num)
{
    static const char prefix[7] = { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };

    TsfStdAssert(tsf);

    if (num < UINT8_MAX) {
        TsfUInt8 tmp8 = (TsfUInt8)num;
        return TsfWriteData(tsf, &tmp8, 1);
    }
    else if (num < UINT16_MAX) {
        TsfUInt16 tmp16 = (TsfUInt16)num;
        if (!TsfWriteData(tsf, prefix, 1)) return TSF_FALSE;
        return TsfWriteSortedData(tsf, &tmp16, 2);
    }
    else if (num < UINT32_MAX) {
        TsfUInt32 tmp32 = (TsfUInt32)num;
        if (!TsfWriteData(tsf, prefix, 3)) return TSF_FALSE;
        return TsfWriteSortedData(tsf, &tmp32, 4);
    }
    else if (num < UINT64_MAX) {
        TsfUInt64 tmp64 = (TsfUInt64)num;
        if (!TsfWriteData(tsf, prefix, 7)) return TSF_FALSE;
        return TsfWriteSortedData(tsf, &tmp64, 8);
    }
    
    tsf->Status = TSFS_ERROR;
    return TSF_FALSE;
}

static TsfBool TSF_API TsfWriteId(TsfObject *tsf, TsfId id, TsfByte type)
{
    TsfByte idByte;

    TsfStdAssert(tsf);
    TsfAssert(id);
    TsfAssert((type & 0xf8) == 0);

    if (id == 0) {
        tsf->Status = TSFS_ERROR;
        return TSF_FALSE;
    }

#ifdef TSF_SUPPORT_LONG_ID
    idByte = (TsfByte)((id < 0x1f) ? id : 0x1f);
#else
    if (id >= 0x1f) {
        tsf->Status = TSFS_ERROR;
        return TSF_FALSE;
    }
    idByte = id;
#endif
    idByte = (idByte << 3) | type;
    if (!TsfWriteData(tsf, &idByte, 1)) return TSF_FALSE;
#ifdef TSF_SUPPORT_LONG_ID
    if (id >= 0x1f) {
        if (!TsfWriteNumber(tsf, id)) return TSF_FALSE;
    }
#endif

    return TSF_TRUE;
}

TsfBool TSF_API TsfWriteArray(TsfObject *tsf, TsfId id, size_t size)
{
    TsfStdAssert(tsf);

    if (size == 0) return TSF_FALSE;

    if (!TsfWriteId(tsf, id, (size == 1) ? 0x05 : 0x06)) return TSF_FALSE;
    if (size > 1) {
        if (!TsfWriteNumber(tsf, size)) return TSF_FALSE;
    }

    return TSF_TRUE;
}

TsfBool TSF_API TsfWriteEnd(TsfObject *tsf)
{
    static const char nul = '\0';

    return TsfWriteData(tsf, &nul, 1);
}

TsfBool TSF_API TsfWriteObject(TsfObject *tsf, TsfId id, const void *data, size_t size)
{
#ifndef TSF_WRITE_VAL0
    unsigned c;
    const unsigned char *p = (const unsigned char *)data;
    for (c = 0; c < size; c++) if (p[c] != 0) break;
    if (c == size) return TSF_TRUE;
#endif
    TsfStdAssert(tsf);

    switch (size) {
    case 0: 
#ifdef TSF_WRITE_SIZE0
#ifdef TSF_SUPPORT_EXT
        if (!TsfWriteId(tsf, id, 7)) return TSF_FALSE;
        return TsfWriteEnd(tsf);
#else
        if (!TsfWriteId(tsf, id, 4)) return TSF_FALSE;
        return TsfWriteNumber(tsf, 0);
#endif 
#else
        return TSF_TRUE;
#endif
    case 1:
        if (!TsfWriteId(tsf, id, 0)) return TSF_FALSE;
        break;
    case 2:
        if (!TsfWriteId(tsf, id, 1)) return TSF_FALSE;
        break;
    case 4:
        if (!TsfWriteId(tsf, id, 2)) return TSF_FALSE;
        break;
    case 8:
        if (!TsfWriteId(tsf, id, 3)) return TSF_FALSE;
        break;
    default:
        if (!TsfWriteId(tsf, id, 4)) return TSF_FALSE;
        if (!TsfWriteNumber(tsf, size)) return TSF_FALSE;
        break;
    }

    return TsfWriteData(tsf, data, size);
}

TsfBool TSF_API TsfWriteObject0(TsfObject *tsf, TsfId id, const void *data, size_t size)
{
    TsfStdAssert(tsf);

    switch (size) {
    case 0: 
#ifdef TSF_WRITE_SIZE0
#ifdef TSF_SUPPORT_EXT
        if (!TsfWriteId(tsf, id, 7)) return TSF_FALSE;
        return TsfWriteEnd(tsf);
#else
        if (!TsfWriteId(tsf, id, 4)) return TSF_FALSE;
        return TsfWriteNumber(tsf, 0);
#endif 
#else
        return TSF_TRUE;
#endif
    case 1:
        if (!TsfWriteId(tsf, id, 0)) return TSF_FALSE;
        break;
    case 2:
        if (!TsfWriteId(tsf, id, 1)) return TSF_FALSE;
        break;
    case 4:
        if (!TsfWriteId(tsf, id, 2)) return TSF_FALSE;
        break;
    case 8:
        if (!TsfWriteId(tsf, id, 3)) return TSF_FALSE;
        break;
    default:
        if (!TsfWriteId(tsf, id, 4)) return TSF_FALSE;
        if (!TsfWriteNumber(tsf, size)) return TSF_FALSE;
        break;
    }

    return TsfWriteData(tsf, data, size);
}

TsfBool TSF_API TsfWriteSortedObject(TsfObject *tsf, TsfId id, const void *data, size_t size)
{
#ifndef TSF_WRITE_VAL0
    unsigned c;
    const unsigned char *p = (const unsigned char *)data;
    for (c = 0; c < size; c++) if (p[c] != 0) break;
    if (c == size) return TSF_TRUE;
#endif
    TsfStdAssert(tsf);

    switch (size) {
    case 0:
#ifdef TSF_WRITE_SIZE0
#ifdef TSF_SUPPORT_EXT
        if (!TsfWriteId(tsf, id, 7)) return TSF_FALSE;
        return TsfWriteEnd(tsf);
#else
        if (!TsfWriteId(tsf, id, 4)) return TSF_FALSE;
        return TsfWriteNumber(tsf, 0);
#endif 
#else
        return TSF_TRUE;
#endif
    case 1:
        if (!TsfWriteId(tsf, id, 0)) return TSF_FALSE;
        break;
    case 2:
        if (!TsfWriteId(tsf, id, 1)) return TSF_FALSE;
        break;
    case 4:
        if (!TsfWriteId(tsf, id, 2)) return TSF_FALSE;
        break;
    case 8:
        if (!TsfWriteId(tsf, id, 3)) return TSF_FALSE;
        break;
    default:
        if (!TsfWriteId(tsf, id, 4)) return TSF_FALSE;
        if (!TsfWriteNumber(tsf, size)) return TSF_FALSE;
        break;
    }

    return TsfWriteSortedData(tsf, data, size);
}

TsfBool TSF_API TsfWriteSortedObject0(TsfObject *tsf, TsfId id, const void *data, size_t size)
{
    TsfStdAssert(tsf);

    switch (size) {
    case 0:
#ifdef TSF_WRITE_SIZE0
#ifdef TSF_SUPPORT_EXT
        if (!TsfWriteId(tsf, id, 7)) return TSF_FALSE;
        return TsfWriteEnd(tsf);
#else
        if (!TsfWriteId(tsf, id, 4)) return TSF_FALSE;
        return TsfWriteNumber(tsf, 0);
#endif 
#else
        return TSF_TRUE;
#endif
    case 1:
        if (!TsfWriteId(tsf, id, 0)) return TSF_FALSE;
        break;
    case 2:
        if (!TsfWriteId(tsf, id, 1)) return TSF_FALSE;
        break;
    case 4:
        if (!TsfWriteId(tsf, id, 2)) return TSF_FALSE;
        break;
    case 8:
        if (!TsfWriteId(tsf, id, 3)) return TSF_FALSE;
        break;
    default:
        if (!TsfWriteId(tsf, id, 4)) return TSF_FALSE;
        if (!TsfWriteNumber(tsf, size)) return TSF_FALSE;
        break;
    }

    return TsfWriteSortedData(tsf, data, size);
}

#ifdef TSF_SUPPORT_INT
TsfBool TSF_API TsfWriteInt8(TsfObject *tsf, TsfId id, TsfInt8 data)
{
    return TsfWriteObject(tsf, id, &data, 1);
}

TsfBool TSF_API TsfWriteInt16(TsfObject *tsf, TsfId id, TsfInt16 data)
{
    if (data >= INT8_MIN && data <= INT8_MAX) {
        TsfInt8 tmp8 = (TsfInt8)data;
        return TsfWriteInt8(tsf, id, tmp8);
    }
    return TsfWriteSortedObject(tsf, id, &data, 2);
}

TsfBool TSF_API TsfWriteInt32(TsfObject *tsf, TsfId id, TsfInt32 data)
{
    if (data >= INT16_MIN && data <= INT16_MAX) {
        TsfInt16 tmp16 = (TsfInt16)data;
        return TsfWriteInt16(tsf, id, tmp16);
    }
    return TsfWriteSortedObject(tsf, id, &data, 4);
}

TsfBool TSF_API TsfWriteInt64(TsfObject *tsf, TsfId id, TsfInt64 data)
{
    if (data >= INT32_MIN && data <= INT32_MAX) {
        TsfInt32 tmp32 = (TsfInt32)data;
        return TsfWriteInt32(tsf, id, tmp32);
    }
    return TsfWriteSortedObject(tsf, id, &data, 8);
}

#endif

#ifdef TSF_SUPPORT_UINT
TsfBool TSF_API TsfWriteUInt8(TsfObject *tsf, TsfId id, TsfUInt8 data)
{
    return TsfWriteObject(tsf, id, &data, 1);
}

TsfBool TSF_API TsfWriteUInt16(TsfObject *tsf, TsfId id, TsfUInt16 data)
{
    if (data <= UINT8_MAX) {
        TsfUInt8 tmp8 = (TsfUInt8)data;
        return TsfWriteUInt8(tsf, id, tmp8);
    }
    return TsfWriteSortedObject(tsf, id, &data, 2);
}

TsfBool TSF_API TsfWriteUInt32(TsfObject *tsf, TsfId id, TsfUInt32 data)
{
    if (data <= UINT16_MAX) {
        TsfUInt16 tmp16 = (TsfUInt16)data;
        return TsfWriteUInt16(tsf, id, tmp16);
    }
    return TsfWriteSortedObject(tsf, id, &data, 4);
}

TsfBool TSF_API TsfWriteUInt64(TsfObject *tsf, TsfId id, TsfUInt64 data)
{
    if (data <= UINT32_MAX) {
        TsfUInt32 tmp32 = (TsfUInt32)data;
        return TsfWriteUInt32(tsf, id, tmp32);
    }
    return TsfWriteSortedObject(tsf, id, &data, 8);
}
#endif

#ifdef TSF_SUPPORT_FLOAT
TsfBool TSF_API TsfWriteFloat32(TsfObject *tsf, TsfId id, float data)
{
    return TsfWriteSortedObject(tsf, id, &data, 4);
}

TsfBool TSF_API TsfWriteFloat64(TsfObject *tsf, TsfId id, double data)
{
    return TsfWriteSortedObject(tsf, id, &data, 8);
}
#endif

#ifdef TSF_SUPPORT_UUID
TsfBool TSF_API TsfWriteUuid(TsfObject *tsf, TsfId id, const TsfUuid data)
{
    return TsfWriteObject(tsf, id, data, sizeof(TsfUuid));
}
#endif

#ifdef TSF_SUPPORT_SIMPLE_ARRAY
TsfBool TSF_API TsfWriteFixedArray(TsfObject *tsf, TsfId id, size_t count, size_t itemSize)
{
    static const char FixedArray = TSFE_FIXED_ARRAY;

    TsfStdAssert(tsf);

    tsf->DataSize = itemSize;
    if (!TsfWriteId(tsf, id, 7)) return TSF_FALSE;
    if (!TsfWriteData(tsf, &FixedArray, 1)) return TSF_FALSE;
    if (!TsfWriteNumber(tsf, itemSize)) return TSF_FALSE;
    return TsfWriteNumber(tsf, count);
}

#ifdef TSF_SUPPORT_INT
TsfBool TSF_API TsfWriteFixedArrayInt8(TsfObject *tsf, TsfInt8 data)
{
    TsfStdAssert(tsf);

    switch (tsf->DataSize)
    {
    case 1: 
        return TsfWriteData(tsf, &data, 1); 
    case 2: {
        TsfInt16 tmp16 = (TsfInt16)data;
        return TsfWriteFixedArrayInt16(tsf, tmp16); }
    case 4: {
        TsfInt32 tmp32 = (TsfInt32)data;
        return TsfWriteFixedArrayInt32(tsf, tmp32); }
    case 8: {
        TsfInt64 tmp64 = (TsfInt64)data;
        return TsfWriteFixedArrayInt64(tsf, tmp64); }
    }

    tsf->Status = TSFS_ERROR;
    return TSF_FALSE;
}

TsfBool TSF_API TsfWriteFixedArrayInt16(TsfObject *tsf, TsfInt16 data)
{
    TsfStdAssert(tsf);

    switch (tsf->DataSize)
    {
    case 1: {
        TsfInt8 tmp8 = (TsfInt8)data;
        if (data < INT8_MIN || data > INT8_MAX) goto error;
        return TsfWriteFixedArrayInt8(tsf, tmp8); }
    case 2: {
        return TsfWriteSortedData(tsf, &data, 2); }
    case 4: {
        TsfInt32 tmp32 = (TsfInt32)data;
        return TsfWriteFixedArrayInt32(tsf, tmp32); }
    case 8: {
        TsfInt64 tmp64 = (TsfInt64)data;
        return TsfWriteFixedArrayInt64(tsf, tmp64); }
    }

error:
    tsf->Status = TSFS_ERROR;
    return TSF_FALSE;
}

TsfBool TSF_API TsfWriteFixedArrayInt32(TsfObject *tsf, TsfInt32 data)
{
    TsfStdAssert(tsf);

    switch (tsf->DataSize)
    {
    case 1: {
        TsfInt8 tmp8 = (TsfInt8)data;
        if (data < INT8_MIN || data > INT8_MAX) goto error;
        return TsfWriteFixedArrayInt8(tsf, tmp8); }
    case 2: {
        TsfInt16 tmp16 = (TsfInt16)data;
        if (data < INT16_MIN || data > INT16_MAX) goto error;
        return TsfWriteFixedArrayInt16(tsf, tmp16); }
    case 4: {
        return TsfWriteSortedData(tsf, &data, 4); }
    case 8: {
        TsfInt64 tmp64 = (TsfInt64)data;
        return TsfWriteFixedArrayInt64(tsf, tmp64); }
    }

error:
    tsf->Status = TSFS_ERROR;
    return TSF_FALSE;
}

TsfBool TSF_API TsfWriteFixedArrayInt64(TsfObject *tsf, TsfInt64 data)
{
    TsfStdAssert(tsf);

    switch (tsf->DataSize)
    {
    case 1: {
        TsfInt8 tmp8 = (TsfInt8)data;
        if (data < INT8_MIN || data > INT8_MAX) goto error;
        return TsfWriteFixedArrayInt8(tsf, tmp8); }
    case 2: {
        TsfInt16 tmp16 = (TsfInt16)data;
        if (data < INT16_MIN || data > INT16_MAX) goto error;
        return TsfWriteFixedArrayInt16(tsf, tmp16); }
    case 4: {
        TsfInt32 tmp32 = (TsfInt32)data;
        if (data < INT32_MIN || data > INT32_MAX) goto error;
        return TsfWriteFixedArrayInt32(tsf, tmp32); }
    case 8: {
        return TsfWriteSortedData(tsf, &data, 8); }
    }

error:
    tsf->Status = TSFS_ERROR;
    return TSF_FALSE;
}
#endif

#ifdef TSF_SUPPORT_INT
TsfBool TSF_API TsfWriteFixedArrayUInt8(TsfObject *tsf, TsfUInt8 data)
{
    TsfStdAssert(tsf);

    switch (tsf->DataSize)
    {
    case 1:
        return TsfWriteData(tsf, &data, 1);
    case 2: {
        TsfUInt16 tmp16 = (TsfUInt16)data;
        return TsfWriteFixedArrayUInt16(tsf, tmp16); }
    case 4: {
        TsfUInt32 tmp32 = (TsfUInt32)data;
        return TsfWriteFixedArrayUInt32(tsf, tmp32); }
    case 8: {
        TsfUInt64 tmp64 = (TsfUInt64)data;
        return TsfWriteFixedArrayUInt64(tsf, tmp64); }
    }

    tsf->Status = TSFS_ERROR;
    return TSF_FALSE;
}

TsfBool TSF_API TsfWriteFixedArrayUInt16(TsfObject *tsf, TsfUInt16 data)
{
    TsfStdAssert(tsf);

    switch (tsf->DataSize)
    {
    case 1: {
        TsfUInt8 tmp8 = (TsfUInt8)data;
        if (data > UINT8_MAX) goto error;
        return TsfWriteFixedArrayUInt8(tsf, tmp8); }
    case 2: {
        return TsfWriteSortedData(tsf, &data, 2); }
    case 4: {
        TsfUInt32 tmp32 = (TsfUInt32)data;
        return TsfWriteFixedArrayUInt32(tsf, tmp32); }
    case 8: {
        TsfUInt64 tmp64 = (TsfUInt64)data;
        return TsfWriteFixedArrayUInt64(tsf, tmp64); }
    }

error:
    tsf->Status = TSFS_ERROR;
    return TSF_FALSE;
}

TsfBool TSF_API TsfWriteFixedArrayUInt32(TsfObject *tsf, TsfUInt32 data)
{
    TsfStdAssert(tsf);

    switch (tsf->DataSize)
    {
    case 1: {
        TsfUInt8 tmp8 = (TsfUInt8)data;
        if (data > UINT8_MAX) goto error;
        return TsfWriteFixedArrayUInt8(tsf, tmp8); }
    case 2: {
        TsfUInt16 tmp16 = (TsfUInt16)data;
        if (data > UINT16_MAX) goto error;
        return TsfWriteFixedArrayUInt16(tsf, tmp16); }
    case 4: {
        return TsfWriteSortedData(tsf, &data, 4); }
    case 8: {
        TsfUInt64 tmp64 = (TsfUInt64)data;
        return TsfWriteFixedArrayUInt64(tsf, tmp64); }
    }

error:
    tsf->Status = TSFS_ERROR;
    return TSF_FALSE;
}

TsfBool TSF_API TsfWriteFixedArrayUInt64(TsfObject *tsf, TsfUInt64 data)
{
    TsfStdAssert(tsf);

    switch (tsf->DataSize)
    {
    case 1: {
        TsfUInt8 tmp8 = (TsfUInt8)data;
        if (data > UINT8_MAX) goto error;
        return TsfWriteFixedArrayUInt8(tsf, tmp8); }
    case 2: {
        TsfUInt16 tmp16 = (TsfUInt16)data;
        if (data > UINT16_MAX) goto error;
        return TsfWriteFixedArrayUInt16(tsf, tmp16); }
    case 4: {
        TsfUInt32 tmp32 = (TsfUInt32)data;
        if (data > UINT32_MAX) goto error;
        return TsfWriteFixedArrayUInt32(tsf, tmp32); }
    case 8: {
        return TsfWriteSortedData(tsf, &data, 8); }
    }

error:
    tsf->Status = TSFS_ERROR;
    return TSF_FALSE;
}
#endif

#ifdef TSF_SUPPORT_FLOAT
TsfBool TSF_API TsfWriteFixedArrayFloat32(TsfObject *tsf, float data)
{
    TsfStdAssert(tsf);

    switch (tsf->DataSize)
    {
    case 4:
        return TsfWriteSortedData(tsf, &data, 4);
    case 8: {
        double tmp64 = (double)data;
        return TsfWriteSortedData(tsf, &tmp64, 8); }
    }

    tsf->Status = TSFS_ERROR;
    return TSF_FALSE;
}

TsfBool TSF_API TsfWriteFixedArrayFloat64(TsfObject *tsf, double data)
{
    TsfStdAssert(tsf);

    switch (tsf->DataSize)
    {
    case 4: {
        float tmp32 = (float)data;
        return TsfWriteSortedData(tsf, &tmp32, 4); }
    case 8: 
        return TsfWriteSortedData(tsf, &data, 8);
    }

    tsf->Status = TSFS_ERROR;
    return TSF_FALSE;
}
#endif

#ifdef TSF_SUPPORT_UUID
TsfBool TSF_API TsfWriteFixedArrayUuid(TsfObject *tsf, const TsfUuid data)
{
    TsfStdAssert(tsf);

    if (tsf->DataSize != sizeof(TsfUuid)) {
        tsf->Status = TSFS_ERROR;
        return TSF_FALSE;
    }
    return TsfWriteData(tsf, &data, sizeof(TsfUuid));
}
#endif

TsfBool TSF_API TsfWriteVarArray(TsfObject *tsf, TsfId id, size_t count)
{
    static const char VarArray = TSFE_VAR_ARRAY;

    TsfStdAssert(tsf);

    if (!TsfWriteId(tsf, id, 7)) return TSF_FALSE;
    if (!TsfWriteData(tsf, &VarArray, 1)) return TSF_FALSE;
    return TsfWriteNumber(tsf, count);
}

TsfBool TSF_API TsfWriteVarArrayData(TsfObject *tsf, const void *data, size_t size)
{
    TsfStdAssert(tsf);

    if (!TsfWriteNumber(tsf, size)) return TSF_FALSE;
    return TsfWriteData(tsf, data, size);
}

TsfBool TSF_API TsfWriteVarArraySortedData(TsfObject *tsf, const void *data, size_t size)
{
    TsfStdAssert(tsf);

    if (!TsfWriteNumber(tsf, size)) return TSF_FALSE;
    return TsfWriteSortedData(tsf, data, size);
}

#ifdef TSF_SUPPORT_INT
TsfBool TSF_API TsfWriteVarArrayInt8(TsfObject *tsf, TsfInt8 data)
{
    if (!TsfWriteNumber(tsf, 1)) return TSF_FALSE;
    return TsfWriteData(tsf, &data, 1);
}

TsfBool TSF_API TsfWriteVarArrayInt16(TsfObject *tsf, TsfInt16 data)
{
    if (data >= INT8_MIN && data <= INT8_MAX) {
        TsfInt8 tmp8 = (TsfInt8)data;
        return TsfWriteVarArrayInt8(tsf, tmp8);
    }
    if (!TsfWriteNumber(tsf, 2)) return TSF_FALSE;
    return TsfWriteSortedData(tsf, &data, 2);
}

TsfBool TSF_API TsfWriteVarArrayInt32(TsfObject *tsf, TsfInt32 data)
{
    if (data >= INT16_MIN && data <= INT16_MAX) {
        TsfInt16 tmp16 = (TsfInt16)data;
        return TsfWriteVarArrayInt16(tsf, tmp16);
    }
    if (!TsfWriteNumber(tsf, 4)) return TSF_FALSE;
    return TsfWriteSortedData(tsf, &data, 4);
}

TsfBool TSF_API TsfWriteVarArrayInt64(TsfObject *tsf, TsfInt64 data)
{
    if (data >= INT32_MIN && data <= INT32_MAX) {
        TsfInt32 tmp32 = (TsfInt32)data;
        return TsfWriteVarArrayInt32(tsf, tmp32);
    }
    if (!TsfWriteNumber(tsf, 8)) return TSF_FALSE;
    return TsfWriteSortedData(tsf, &data, 8);
}
#endif

#ifdef TSF_SUPPORT_UINT
TsfBool TSF_API TsfWriteVarArrayUInt8(TsfObject *tsf, TsfUInt8 data)
{
    if (!TsfWriteNumber(tsf, 1)) return TSF_FALSE;
    return TsfWriteData(tsf, &data, 1);
}

TsfBool TSF_API TsfWriteVarArrayUInt16(TsfObject *tsf, TsfUInt16 data)
{
    if (data <= UINT8_MAX) {
        TsfUInt8 tmp8 = (TsfUInt8)data;
        return TsfWriteVarArrayUInt8(tsf, tmp8);
    }
    if (!TsfWriteNumber(tsf, 2)) return TSF_FALSE;
    return TsfWriteSortedData(tsf, &data, 2);
}

TsfBool TSF_API TsfWriteVarArrayUInt32(TsfObject *tsf, TsfUInt32 data)
{
    if (data <= UINT16_MAX) {
        TsfUInt16 tmp16 = (TsfUInt16)data;
        return TsfWriteVarArrayUInt16(tsf, tmp16);
    }
    if (!TsfWriteNumber(tsf, 4)) return TSF_FALSE;
    return TsfWriteSortedData(tsf, &data, 4);
}

TsfBool TSF_API TsfWriteVarArrayUInt64(TsfObject *tsf, TsfUInt64 data)
{
    if (data <= UINT32_MAX) {
        TsfUInt32 tmp32 = (TsfUInt32)data;
        return TsfWriteVarArrayUInt32(tsf, tmp32);
    }
    if (!TsfWriteNumber(tsf, 8)) return TSF_FALSE;
    return TsfWriteSortedData(tsf, &data, 8);
}
#endif

#ifdef TSF_SUPPORT_FLOAT
TsfBool TSF_API TsfWriteVarArrayFloat32(TsfObject *tsf, float data)
{
    if (!TsfWriteNumber(tsf, 4)) return TSF_FALSE;
    return TsfWriteSortedData(tsf, &data, 4);
}

TsfBool TSF_API TsfWriteVarArrayFloat64(TsfObject *tsf, double data)
{
    if (!TsfWriteNumber(tsf, 8)) return TSF_FALSE;
    return TsfWriteSortedData(tsf, &data, 8);
}
#endif

#ifdef TSF_SUPPORT_UUID
TsfBool TSF_API TsfWriteVarArrayUuid(TsfObject *tsf, const TsfUuid data)
{
    if (!TsfWriteNumber(tsf, sizeof(TsfUuid))) return TSF_FALSE;
    return TsfWriteData(tsf, &data, sizeof(TsfUuid));
}
#endif

#endif

#endif