/*
 * -----------------------------------------------------------------------------
 * Tagged Stream Format V1.1.19
 * Copyright (c) 2017-2019 Rudy Tellert Elektronik
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy 
 * of this software and associated documentation files (the "Software"), to deal 
 * in the Software without restriction, including without limitation the rights 
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 * SOFTWARE.
 * -----------------------------------------------------------------------------
 */

#include "tsf_priv.h"

#ifdef TSF_SUPPORT_UUID

static int TsfHexDigit(char ch)
{
    if (ch >= '0' && ch <= '9') return ch - '0';
    if (ch >= 'a' && ch <= 'f') return ch - ('a' - 10);
    if (ch >= 'A' && ch <= 'F') return ch - ('A' - 10);
    return -1;
}

static const char *TsfHexByteA(const char *p, unsigned char *val)
{
    int lo, hi;
    if (p == NULL || val == NULL) return NULL;
    if (_strnicmp(p, "urn:uuid:", 9) == 0) p += 9;
    do {
        if (*p == '\0') return NULL;
        hi = TsfHexDigit(*p++);
    } while (hi < 0);
    do {
        if (*p == '\0') return NULL;
        lo = TsfHexDigit(*p++);
    } while (lo < 0);
    *val = (unsigned char)((hi << 4) | lo);
    return p;
}

#ifdef TSF_SUPPORT_WCHAR_T
static const wchar_t *TsfHexByteW(const wchar_t *p, unsigned char *val)
{
    int lo, hi;
    if (p == NULL || val == NULL) return NULL;
    if (_wcsnicmp(p, L"urn:uuid:", 9) == 0) p += 9;
    do {
        if (*p == '\0') return NULL;
        hi = TsfHexDigit((char)*p++);
    } while (hi < 0);
    do {
        if (*p == '\0') return NULL;
        lo = TsfHexDigit((char)*p++);
    } while (lo < 0);
    *val = (unsigned char)((hi << 4) | lo);
    return p;
}
#endif

void TSF_API TsfUuidFromData(TsfUuid uuid, const void *data)
{
    if (uuid != NULL && data != NULL) {
        memcpy(uuid, data, sizeof(TsfUuid));
    }
}

void TSF_API TsfUuidFromString(TsfUuid uuid, const char *str)
{
    if (uuid != NULL) {
        memset(uuid, '\0', sizeof(TsfUuid));
        if (str) {
            int i;
            for (i = 0; i < sizeof(TsfUuid); i++) {
                str = TsfHexByteA(str, uuid + i);
                if (str == NULL) break;
            }
        }
    }
}

#ifdef TSF_SUPPORT_WCHAR_T
void TSF_API TsfUuidFromWideString(TsfUuid uuid, const wchar_t *wstr)
{
    if (uuid != NULL) {
        memset(uuid, '\0', sizeof(TsfUuid));
        if (wstr) {
            int i;
            for (i = 0; i < sizeof(TsfUuid); i++) {
                wstr = TsfHexByteW(wstr, uuid + i);
                if (wstr == NULL) break;
            }
        }
    }
}
#endif

/* source: http://xoshiro.di.unimi.it/xorshift.php */

static TsfUInt64 s[2];

static TsfUInt64 NextXorShift128Plus(void) 
{
    uint64_t s1 = s[0];
    const uint64_t s0 = s[1];
    const uint64_t result = s0 + s1;

    s[0] = s0;
    s1 ^= s1 << 23; // a
    s[1] = s1 ^ s0 ^ (s1 >> 18) ^ (s0 >> 5); // b, c

    return result;
}

static void JumpXorShift128Plus(void) 
{
    static const uint64_t jump[] = { 0x8a5cd789635d2dff, 0x121fd2155c472f96 };

    uint64_t s0 = 0;
    uint64_t s1 = 0;

    for (int i = 0; i < 2; i++) {
        for (int b = 0; b < 64; b++) {
            if (jump[i] & (UINT64_C(1) << b)) {
                s0 ^= s[0];
                s1 ^= s[1];
            }
            NextXorShift128Plus();
        }
    }

    s[0] = s0;
    s[1] = s1;
}

/* source: http://xoshiro.di.unimi.it/splitmix64.c */

static uint64_t x; /* The state can be seeded with any value. */

static uint64_t NextSplitMix64() 
{
    uint64_t z = (x += 0x9e3779b97f4a7c15);

    z = (z ^ (z >> 30)) * 0xbf58476d1ce4e5b9;
    z = (z ^ (z >> 27)) * 0x94d049bb133111eb;

    return z ^ (z >> 31);
}

void TSF_API TsfUuidFromNonce(TsfUuid uuid, TsfUInt64 nonce)
{
    /* The next line can be commented out if the nonce and uuid are limited to this platform's byte order */
    TsfAssert(TsfIsLittleEndian());
    
    if (uuid != NULL) {

        x = nonce;
        s[0] = NextSplitMix64();
        s[1] = NextSplitMix64();

        JumpXorShift128Plus();

        ((TsfUInt64*)uuid)[0] = NextXorShift128Plus();
        ((TsfUInt64*)uuid)[1] = NextXorShift128Plus();

        uuid[6] = (TsfByte)(0x40 | (uuid[6] & 0x0f)); /* = 0b0100xxxx */
        uuid[8] = (TsfByte)(0x80 | (uuid[8] & 0x3f)); /* = 0b10xxxxxx */
    }
}

TsfBool TSF_API TsfUuidIsEqual(const TsfUuid uuid1, const TsfUuid uuid2)
{
    if (uuid1 == NULL || uuid2 == NULL) return TSF_FALSE;
    return (memcmp(uuid1, uuid2, sizeof(TsfUuid)) == 0) ? TSF_TRUE : TSF_FALSE;
}

TsfBool TSF_API TsfUuidIsGood(const TsfUuid uuid)
{
    return (uuid != NULL && (uuid[6] & 0xf0) == 0x40 && (uuid[8] & 0xc0) == 0x80) ? TSF_TRUE : TSF_FALSE;
}

void TSF_API TsfUuidEmpty(TsfUuid uuid)
{
    if (uuid) {
        memset(uuid, '\0', sizeof(TsfUuid));
    }
}

#endif
