/*
 * -----------------------------------------------------------------------------
 * Tagged Stream Format V1.1.19
 * Copyright (c) 2017-2019 Rudy Tellert Elektronik
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy 
 * of this software and associated documentation files (the "Software"), to deal 
 * in the Software without restriction, including without limitation the rights 
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 * SOFTWARE.
 * -----------------------------------------------------------------------------
 */

/* tsfread.c is endian neutral */

#include "tsf_priv.h"

#ifdef TSF_SUPPORT_INPUT

/*- stream access ------------------------------------------------------------*/

static size_t TSF_API TsfGetRemainingStreamSize(TsfObject *tsf)
{
    TsfStdAssert(tsf);

    return tsf->RemainingSize;
}

/* skip data if data is null */
static TsfBool TSF_API TsfRead(TsfObject *tsf, void *data, size_t size)
{
    TsfStdAssert(tsf);

    if (tsf->Status == TSFS_ERROR) return TSF_FALSE;

    if (tsf->RemainingSize < size) {
        tsf->Status = TSFS_ERROR;
        return TSF_FALSE;
    }
    tsf->RemainingSize -= size;

    if (data) memcpy(data, tsf->Data, size);
    tsf->Data += size;

    return TSF_TRUE;
}

void TSF_API TsfOpen(TsfObject *tsf, const void *data, size_t size)
{
    const char *p;
    size_t headerSize;

    TsfInit(tsf);
    TsfAssert(data);

    p = (const char *)data;
    headerSize = 0;
    if (size >= 4) {
        if (p[0] == TSF_HEADER0 && p[1] == TSF_HEADER1) {
            switch (*(const unsigned short*)(p + 2)) {
            case TSF_HEADER2:
                tsf->Header = TSFH_NORMAL;
                headerSize = 4;
                break;
#ifdef TSF_SUPPORT_REV
            case TSF_HEADER2_REV:
                tsf->Header = TSFH_REV;
#ifdef TSF_SUPPORT_REV_FIELD
                tsf->Reverse = TSF_TRUE;
#endif
                headerSize = 4;
                break;
#endif
            }
        }
    }

    tsf->Data = (const TsfByte*)p + headerSize;
    tsf->RemainingSize = size - headerSize;
    tsf->TotalSize = size;
}

/*- layer 1 ------------------------------------------------------------------*/

size_t TSF_API TsfGetPos(TsfObject *tsf)
{
    TsfStdAssert(tsf);

    return tsf->TotalSize - tsf->RemainingSize;
}

TsfBool TSF_API TsfSetPos(TsfObject *tsf, size_t pos)
{
    TsfStdAssert(tsf);

    if (pos > tsf->TotalSize) return TSF_FALSE;

    tsf->Data += pos - (tsf->TotalSize - tsf->RemainingSize);
    tsf->RemainingSize = tsf->TotalSize - pos;

    return TSF_TRUE;
}

TsfBool TSF_API TsfReadData(TsfObject *tsf, void *data, size_t size)
{
    TsfStdAssert(tsf);

    return TsfRead(tsf, data, size);
}

TsfBool TSF_API TsfReadObject(TsfObject *tsf, void *data, size_t size)
{
    TsfStdAssert(tsf);

    if (size < tsf->DataSize) {
        TsfSkipData(tsf);
        return TSF_FALSE;
    }

    return TsfReadData(tsf, data, tsf->DataSize);
}

#ifdef TSF_SUPPORT_REV
void TSF_API TsfSortData(TsfObject *tsf, void *data, size_t size)
{
    TsfStdAssert(tsf);

#ifdef TSF_SUPPORT_REV_FIELD
    if (tsf->Reverse) TsfReverse(data, size);
#else
    if (tsf->Header == TSFH_REV) TsfReverse(data, size);
#endif
}

TsfBool TSF_API TsfReadSortedData(TsfObject *tsf, void *data, size_t size)
{
    /* TsfReadData() */
    TsfStdAssert(tsf);

    if (!TsfRead(tsf, data, size)) return TSF_FALSE;

    /* TsfSortData() */
#ifdef TSF_SUPPORT_REV_FIELD
    if (tsf->Reverse) TsfReverse(data, size);
#else
    if (tsf->Header == TSFH_REV) TsfReverse(data, size);
#endif

    return TSF_TRUE;
}

TsfBool TSF_API TsfReadSortedObject(TsfObject *tsf, void *data, size_t size)
{
    TsfStdAssert(tsf);
    
    if (size < tsf->DataSize) {
        TsfSkipData(tsf);
        return TSF_FALSE;
    }

    return TsfReadSortedData(tsf, data, tsf->DataSize);
}
#endif

static TsfBool TSF_API TsfGetNumber(TsfObject *tsf, size_t *num)
{
    TsfUInt8 tmp8;

    TsfStdAssert(tsf);
    TsfAssert(num);

    if (!TsfReadData(tsf, &tmp8, 1)) return TSF_FALSE;
    if ((*num = tmp8) == UINT8_MAX) {
#if TSF_SIZE_BITS >= 16
        TsfUInt16 tmp16;
        if (!TsfReadSortedData(tsf, &tmp16, 2)) return TSF_FALSE;
#  ifdef TSF_SUPPORT_STRICT_PAR_CHECK
        if (tmp16 < UINT8_MAX) goto error;
#  endif
        if ((*num = tmp16) == UINT16_MAX) {
#  if TSF_SIZE_BITS >= 32
            TsfUInt32 tmp32;
            if (!TsfReadSortedData(tsf, &tmp32, 4)) return TSF_FALSE;
#  ifdef TSF_SUPPORT_STRICT_PAR_CHECK
            if (tmp32 < UINT16_MAX) goto error;
#  endif
            if ((*num = tmp32) == UINT32_MAX) {
#    if TSF_SIZE_BITS >= 64
                TsfUInt64 tmp64;
                if (!TsfReadSortedData(tsf, &tmp64, 8)) return TSF_FALSE;
#  ifdef TSF_SUPPORT_STRICT_PAR_CHECK
                if (tmp64 < UINT32_MAX) goto error;
#  endif
                if ((*num = tmp64) == UINT64_MAX) goto error;
#    else
                goto error;
#    endif
            }
#  else
            goto error;
#  endif
        }
#else
        goto error;
#endif
    }

    return TSF_TRUE;

error:
    tsf->Status = TSFS_ERROR;
    return TSFT_ERROR;
}

static TsfBool TSF_API TsfIsDataAvailable(TsfObject *tsf, size_t size)
{
    TsfStdAssert(tsf);

    return size <= TsfGetRemainingStreamSize(tsf);
}

TsfTokenType TSF_API TsfReadToken(TsfObject *tsf)
{
    TsfByte t, id;

    TsfStdAssert(tsf);

    if (tsf->Status >= TSFS_END) return tsf->Status;

    tsf->DataSize = 0;
    tsf->Count = 0;
#ifdef TSF_SUPPORT_EXT
    tsf->Ext = 0;
#endif
#ifdef TSF_SUPPORT_SIMPLE_ARRAY
    tsf->DataCount = 0;
#endif

    for (;;) {

        if (!TsfReadData(tsf, &t, 1)) {
            if (tsf->Level) goto error;
            tsf->Status = TSFS_END_OF_STREAM;
            return TSFT_END_OF_STREAM;
        }
        tsf->Id = id = t >> 3;
        if (id) {
#ifdef TSF_SUPPORT_LONG_ID
            if (id == 0x1f) {
                if (!TsfGetNumber(tsf, &tsf->Id)) goto error;
#  ifdef TSF_SUPPORT_STRICT_PAR_CHECK
                if (tsf->Id < 0x1f) goto error;
#  endif
            }
#else
            if (id == 0x1f) goto error;
#endif
#ifdef TSF_SUPPORT_TAG_ORDER_CHECK
            if (tsf->Id <= tsf->LevelData[tsf->Level].CurrId) goto error;
            tsf->LevelData[tsf->Level].CurrId = tsf->Id;
#endif
            switch (t & 7) {
            case 0:
                if (!TsfIsDataAvailable(tsf, tsf->DataSize = 1)) goto error;
                return TSFT_DATA;
            case 1:
                if (!TsfIsDataAvailable(tsf, tsf->DataSize = 2)) goto error;
                return TSFT_DATA;
            case 2:
                if (!TsfIsDataAvailable(tsf, tsf->DataSize = 4)) goto error;
                return TSFT_DATA;
            case 3:
                if (!TsfIsDataAvailable(tsf, tsf->DataSize = 8)) goto error;
                return TSFT_DATA;
            case 4:
                if (!TsfGetNumber(tsf, &tsf->DataSize)) goto error;
#ifdef TSF_SUPPORT_STRICT_PAR_CHECK
#  ifdef TSF_SUPPORT_EXT
                if (tsf->DataSize == 0) goto error;
#  endif
                if (tsf->DataSize == 1 || tsf->DataSize == 2 ||
                    tsf->DataSize == 4 || tsf->DataSize == 8) goto error;
#endif
                if (!TsfIsDataAvailable(tsf, tsf->DataSize)) goto error;
                return TSFT_DATA;
            case 5:
                tsf->LevelData[tsf->Level].Count = tsf->Count = 1;
                if (++tsf->Level == TSF_LEVEL_COUNT) goto error;
                return TSFT_COLLECTION;
            case 6:
                if (!TsfGetNumber(tsf, &tsf->Count)) goto error;
#ifdef TSF_SUPPORT_STRICT_PAR_CHECK
                if (tsf->Count <= 1) goto error;
#endif
                tsf->LevelData[tsf->Level].Count = tsf->Count;
                if (++tsf->Level == TSF_LEVEL_COUNT) goto error;
                return TSFT_ARRAY;
            case 7:
#ifdef TSF_SUPPORT_EXT
                if (!TsfReadData(tsf, &tsf->Ext, 1)) goto error;
#  ifdef TSF_SUPPORT_SIMPLE_ARRAY
                if (tsf->Ext == TSFE_FIXED_ARRAY) {
                    if (!TsfGetNumber(tsf, &tsf->DataSize)) goto error;
                    if (!TsfGetNumber(tsf, &tsf->DataCount)) goto error;
                    if (tsf->DataCount == 0) goto error;
                    if (!TsfIsDataAvailable(tsf, tsf->DataSize)) goto error;
                    return TSFT_EXT;
                }
                else if (tsf->Ext == TSFE_VAR_ARRAY) {
                    if (!TsfGetNumber(tsf, &tsf->DataCount)) goto error;
                    if (tsf->DataCount == 0) goto error;
                    if (!TsfGetNumber(tsf, &tsf->DataSize)) goto error;
                    if (!TsfIsDataAvailable(tsf, tsf->DataSize)) goto error;
                    return TSFT_EXT;
                } 
                else 
#  endif
                if ((tsf->Ext & 0x0f) != 0) goto error;
                return TSFT_EXT;
#else
                goto error;
#endif
            }
        }
        else {
            switch (t) {
            case 0:
#ifdef TSF_SUPPORT_TAG_ORDER_CHECK
                tsf->LevelData[tsf->Level].CurrId = 0;
#endif
                if (tsf->Level) {
                    if (--tsf->LevelData[tsf->Level - 1].Count == 0) {
                        tsf->Level--;
                        return TSFT_END_OF_ARRAY;
                    }
                    else {
                        return TSFT_END_OF_COLLECTION;
                    }
                }
                else {
                    tsf->Status = TSFT_END;
                    return TSFT_END;
                }
#if 0 /* unexpected header token */
            case 1:
                if (!TsfIsDataAvailable(tsf, tsf->DataSize = 3)) goto error;
                return TSFT_HEADER;
#endif
#ifdef TSF_SUPPORT_NOP
            case 2:
                /* TSFT_NOP */
                continue;
#endif
            }
        }
        break;
    }

error:
    tsf->Status = TSFS_ERROR;
    return TSFT_ERROR;
}

#ifdef TSF_SUPPORT_SIMPLE_ARRAY
TsfBool TSF_API TsfSelectNextItem(TsfObject *tsf)
{
    if (tsf->DataCount == 0) return TSF_FALSE;
    if (--tsf->DataCount == 0) return TSF_FALSE;

    switch (tsf->Ext) {
    case TSFE_VAR_ARRAY:
        if (!TsfGetNumber(tsf, &tsf->DataSize)) goto error;
        /* fall through */
    case TSFE_FIXED_ARRAY:
        if (TsfIsDataAvailable(tsf, tsf->DataSize)) return TSF_TRUE;
        break;
    }

error:
    tsf->Status = TSFS_ERROR;
    return TSF_FALSE;
}
#endif

#ifdef TSF_SUPPORT_INT
TsfBool TSF_API TsfReadInt8(TsfObject *tsf, TsfInt8 *data)
{
    TsfInt64 tmp64;

    TsfStdAssert(tsf);
    TsfAssert(data);

    if (!TsfReadInt64(tsf, &tmp64)) return TSF_FALSE;

    if (tmp64 > INT8_MAX) return TSF_FALSE;
    if (tmp64 < INT8_MIN) return TSF_FALSE;
    *data = (TsfInt8)tmp64;

    return TSF_TRUE;
}

TsfBool TSF_API TsfReadInt16(TsfObject *tsf, TsfInt16 *data)
{
    TsfInt64 tmp64;

    TsfStdAssert(tsf);
    TsfAssert(data);

    if (!TsfReadInt64(tsf, &tmp64)) return TSF_FALSE;

    if (tmp64 > INT16_MAX) return TSF_FALSE;
    if (tmp64 < INT16_MIN) return TSF_FALSE;
    *data = (TsfInt16)tmp64;

    return TSF_TRUE;
}

TsfBool TSF_API TsfReadInt32(TsfObject *tsf, TsfInt32 *data)
{
    TsfInt64 tmp64;

    TsfStdAssert(tsf);
    TsfAssert(data);

    if (!TsfReadInt64(tsf, &tmp64)) return TSF_FALSE;

    if (tmp64 > INT32_MAX) return TSF_FALSE;
    if (tmp64 < INT32_MIN) return TSF_FALSE;
    *data = (TsfInt32)tmp64;

    return TSF_TRUE;
}

TsfBool TSF_API TsfReadInt64(TsfObject *tsf, TsfInt64 *data)
{
    TsfStdAssert(tsf);
    TsfAssert(data);

    if (tsf->DataSize == 1) {
        TsfInt8 tmp8;
        TsfReadData(tsf, &tmp8, 1);
        *data = tmp8;
        return TSF_TRUE;
    }
    else if (tsf->DataSize == 2) {
        TsfInt16 tmp16;
        TsfReadSortedData(tsf, &tmp16, 2);
        *data = tmp16;
        return TSF_TRUE;
    }
    else if (tsf->DataSize == 4) {
        TsfInt32 tmp32;
        TsfReadSortedData(tsf, &tmp32, 4);
        *data = tmp32;
        return TSF_TRUE;
    }
    else if (tsf->DataSize == 8) {
        TsfReadSortedData(tsf, data, 8);
        return TSF_TRUE;
    }

    TsfSkipData(tsf);
    return TSF_FALSE;
}
#endif

#ifdef TSF_SUPPORT_UINT
TsfBool TSF_API TsfReadUInt8(TsfObject *tsf, TsfUInt8 *data)
{
    TsfUInt64 tmp64;

    TsfStdAssert(tsf);
    TsfAssert(data);

    if (!TsfReadUInt64(tsf, &tmp64)) return TSF_FALSE;

    if (tmp64 > UINT8_MAX) return TSF_FALSE;
    *data = (TsfUInt8)tmp64;

    return TSF_TRUE;
}

TsfBool TSF_API TsfReadUInt16(TsfObject *tsf, TsfUInt16 *data)
{
    TsfUInt64 tmp64;

    TsfStdAssert(tsf);
    TsfAssert(data);

    if (!TsfReadUInt64(tsf, &tmp64)) return TSF_FALSE;

    if (tmp64 > UINT16_MAX) return TSF_FALSE;
    *data = (TsfUInt16)tmp64;

    return TSF_TRUE;
}

TsfBool TSF_API TsfReadUInt32(TsfObject *tsf, TsfUInt32 *data)
{
    TsfUInt64 tmp64;

    TsfStdAssert(tsf);
    TsfAssert(data);

    if (!TsfReadUInt64(tsf, &tmp64)) return TSF_FALSE;

    if (tmp64 > UINT32_MAX) return TSF_FALSE;
    *data = (TsfUInt32)tmp64;

    return TSF_TRUE;
}

TsfBool TSF_API TsfReadUInt64(TsfObject *tsf, TsfUInt64 *data)
{
    TsfStdAssert(tsf);
    TsfAssert(data);

    if (tsf->DataSize == 1) {
        TsfUInt8 tmp8;
        TsfReadData(tsf, &tmp8, 1);
        *data = tmp8;
        return TSF_TRUE;
    }
    else if (tsf->DataSize == 2) {
        TsfUInt16 tmp16;
        TsfReadSortedData(tsf, &tmp16, 2);
        *data = tmp16;
        return TSF_TRUE;
    }
    else if (tsf->DataSize == 4) {
        TsfUInt32 tmp32;
        TsfReadSortedData(tsf, &tmp32, 4);
        *data = tmp32;
        return TSF_TRUE;
    }
    else if (tsf->DataSize == 8) {
        TsfReadSortedData(tsf, data, 8);
        return TSF_TRUE;
    }

    TsfSkipData(tsf);
    return TSF_FALSE;
}
#endif

#ifdef TSF_SUPPORT_FLOAT
TsfBool TSF_API TsfReadFloat32(TsfObject *tsf, float *data)
{
    TsfStdAssert(tsf);
    TsfAssert(data);

    if (tsf->DataSize == 4) {
        TsfReadSortedData(tsf, data, 4);
        return TSF_TRUE;
    }
    else if (tsf->DataSize == 8) {
        double tmp64;
        TsfReadSortedData(tsf, &tmp64, 8);
        *data = (float)tmp64;
        return TSF_TRUE;
    }
#ifdef TSF_SUPPORT_FLOAT_FROM_INT
    else if (tsf->DataSize == 1) {
        TsfInt8 tmp8;
        TsfReadData(tsf, &tmp8, 1);
        *data = (float)tmp8;
        return TSF_TRUE;
    }
    else if (tsf->DataSize == 2) {
        TsfInt16 tmp16;
        TsfReadSortedData(tsf, &tmp16, 2);
        *data = (float)tmp16;
        return TSF_TRUE;
    }
#endif

    TsfSkipData(tsf);
    return TSF_FALSE;
}

TsfBool TSF_API TsfReadFloat64(TsfObject *tsf, double *data)
{
    TsfStdAssert(tsf);
    TsfAssert(data);

    if (tsf->DataSize == 8) {
        TsfReadSortedData(tsf, data, 8);
        return TSF_TRUE;
    }
    else if (tsf->DataSize == 4) {
        float tmp32;
        TsfReadSortedData(tsf, &tmp32, 4);
        *data = (double)tmp32;
        return TSF_TRUE;
    }
#ifdef TSF_SUPPORT_FLOAT_FROM_INT
    else if (tsf->DataSize == 1) {
        TsfInt8 tmp8;
        TsfReadData(tsf, &tmp8, 1);
        *data = (double)tmp8;
        return TSF_TRUE;
    }
    else if (tsf->DataSize == 2) {
        TsfInt16 tmp16;
        TsfReadSortedData(tsf, &tmp16, 2);
        *data = (double)tmp16;
        return TSF_TRUE;
    }
#endif

    TsfSkipData(tsf);
    return TSF_FALSE;
}
#endif

#ifdef TSF_SUPPORT_UUID
TsfBool TSF_API TsfReadUuid(TsfObject *tsf, TsfUuid *data)
{
    TsfStdAssert(tsf);
    TsfAssert(data);

    if (tsf->DataSize == sizeof(TsfUuid)) {
        TsfReadData(tsf, data, sizeof(TsfUuid));
        return TSF_TRUE;
    }

    TsfSkipData(tsf);
    return TSF_FALSE;
}
#endif

/*- layer 2 ------------------------------------------------------------------*/

TsfBool TSF_API TsfSelectObject(TsfObject *tsf, TsfId id)
{
    int level;
    size_t pos;

    TsfStdAssert(tsf);
    TsfAssert(id);

    if (tsf->Status >= TSFS_END) return TSF_FALSE;

#ifdef TSF_SUPPORT_SIMPLE_ARRAY
    /* skip half-processed array */
    if (tsf->DataCount) while (TsfSelectNextItem(tsf)) TsfSkipData(tsf);
#endif

    level = tsf->Level;

    for (;;) {
        pos = TsfGetPos(tsf);
        switch (TsfReadToken(tsf)) {
        case TSFT_EXT:
        case TSFT_DATA:
            if (tsf->Level == level && tsf->Id >= id) {
                if (tsf->Id == id) return TSF_TRUE;
                TsfSetPos(tsf, pos);
#ifdef TSF_SUPPORT_SIMPLE_ARRAY
                tsf->DataCount = 0;
#endif
                tsf->LevelData[tsf->Level].CurrId = 0;
                return TSF_FALSE;
            }
#ifdef TSF_SUPPORT_SIMPLE_ARRAY
            do {
#endif
                TsfSkipData(tsf);
#ifdef TSF_SUPPORT_SIMPLE_ARRAY
            } while (TsfSelectNextItem(tsf));
#endif
                continue;
        case TSFT_COLLECTION:
        case TSFT_ARRAY:
            if (tsf->Level == level + 1 && tsf->Id >= id) {
                TsfSetPos(tsf, pos);
                --tsf->Level;
                tsf->LevelData[tsf->Level].CurrId = 0;
                return TSF_FALSE;
            }
            continue;
        case TSFT_END_OF_COLLECTION:
            if (tsf->Level == level) {
                TsfSetPos(tsf, pos);
                tsf->LevelData[tsf->Level-1].Count++;
                tsf->LevelData[tsf->Level].CurrId = 0;
                return TSF_FALSE;
            }
            continue;
        case TSFT_END_OF_ARRAY:
            if (tsf->Level + 1 == level) {
                TsfSetPos(tsf, pos);
                tsf->LevelData[tsf->Level++].Count++;
                tsf->LevelData[tsf->Level].CurrId = 0;
                return TSF_FALSE;
            }
            continue;
        }
        return TSF_FALSE;
    }
}

size_t TSF_API TsfReadArraySize(TsfObject *tsf, TsfId id)
{
    int level;
    size_t pos;

    TsfStdAssert(tsf);
    TsfAssert(id);

    if (tsf->Status >= TSFS_END) return 0;

#ifdef TSF_SUPPORT_SIMPLE_ARRAY
    /* skip half-processed array */
    if (tsf->DataCount) while (TsfSelectNextItem(tsf)) TsfSkipData(tsf);
#endif

    level = tsf->Level;

    for (;;) {
        pos = TsfGetPos(tsf);
        switch (TsfReadToken(tsf)) {
        case TSFT_EXT:
        case TSFT_DATA:
            if (tsf->Level == level && tsf->Id >= id) {
                TsfSetPos(tsf, pos);
#ifdef TSF_SUPPORT_SIMPLE_ARRAY
                tsf->DataCount = 0;
#endif
                tsf->LevelData[tsf->Level].CurrId = 0;
                return 0;
            }
#ifdef TSF_SUPPORT_SIMPLE_ARRAY
            do {
#endif
                TsfSkipData(tsf);
#ifdef TSF_SUPPORT_SIMPLE_ARRAY
            } while (TsfSelectNextItem(tsf));
#endif
            continue;
        case TSFT_COLLECTION:
        case TSFT_ARRAY:
            if (tsf->Level == level + 1 && tsf->Id >= id) {
                if (tsf->Id == id) return tsf->Count;
                TsfSetPos(tsf, pos);
                --tsf->Level;
                tsf->LevelData[tsf->Level].CurrId = 0;
                return 0;
            }
            continue;
        case TSFT_END_OF_COLLECTION:
            if (tsf->Level == level) {
                TsfSetPos(tsf, pos);
                tsf->LevelData[tsf->Level-1].Count++;
                tsf->LevelData[tsf->Level].CurrId = 0;
                return 0;
            }
            continue;
        case TSFT_END_OF_ARRAY:
            if (tsf->Level + 1 == level) {
                TsfSetPos(tsf, pos);
                tsf->LevelData[tsf->Level++].Count++;
                tsf->LevelData[tsf->Level].CurrId = 0;
                return 0;
            }
            continue;
        }
        return 0;
    }
}

TsfBool TSF_API TsfSkipCollectionOrArray(TsfObject *tsf, TsfBool skipCollectionOnly)
{
    int level;

    TsfStdAssert(tsf);

    if (tsf->Status >= TSFS_END) return TSF_FALSE;

#ifdef TSF_SUPPORT_SIMPLE_ARRAY
    /* skip half-processed array */
    if (tsf->DataCount) while (TsfSelectNextItem(tsf)) TsfSkipData(tsf);
#endif

    level = tsf->Level;

    for (;;) {
        switch (TsfReadToken(tsf)) {
        case TSFT_EXT:
        case TSFT_DATA:
#ifdef TSF_SUPPORT_SIMPLE_ARRAY
            do {
#endif
                TsfSkipData(tsf);
#ifdef TSF_SUPPORT_SIMPLE_ARRAY
            } while (TsfSelectNextItem(tsf));
#endif
            continue;
        case TSFT_COLLECTION:
        case TSFT_ARRAY:
            continue;
        case TSFT_END_OF_COLLECTION:
            if (skipCollectionOnly && tsf->Level == level) return TSF_TRUE;
            continue;
        case TSFT_END_OF_ARRAY:
            if (tsf->Level + 1 == level) return TSF_TRUE;
            continue;
        }
        return TSF_FALSE;
    }
}

#endif
