/*
 * -----------------------------------------------------------------------------
 * Tagged Stream Format V1.1.19
 * Copyright (c) 2017-2019 Rudy Tellert Elektronik
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy 
 * of this software and associated documentation files (the "Software"), to deal 
 * in the Software without restriction, including without limitation the rights 
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 * SOFTWARE.
 * -----------------------------------------------------------------------------
 */

#define _CRT_SECURE_NO_WARNINGS
#include "tsf_priv.h"

#ifdef TSF_SUPPORT_FILE

#ifdef TSF_USE_LOW_LEVEL_FILE_ACCESS

#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <io.h>

#ifdef TSF_SUPPORT_OUTPUT
TsfBool TSF_API TsfWriteFile(TsfObject *tsf, const TSF_TCHAR *fileName)
{
    int h;
    TsfBool result = TSF_TRUE;
    const TsfByte *p = TsfDataBase(tsf);
    size_t size = TsfDataSize(tsf);
    unsigned n;
    int i;

    TsfStdAssert(tsf);
    if (tsf->Status == TSFS_ERROR) return TSF_FALSE;

#ifdef TSF_UNICODE
    h = _wopen(fileName, _O_WRONLY | _O_BINARY | _O_CREAT | _O_TRUNC, _S_IREAD | _S_IWRITE);
#else
    h = _open(fileName, _O_WRONLY | _O_BINARY | _O_CREAT | _O_TRUNC, _S_IREAD | _S_IWRITE);
#endif
    if (h != -1) {
        while (size) {
            n = (size > INT_MAX) ? INT_MAX : (unsigned)size;
            if ((i = _write(h, p, n)) <= 0) {
                result = TSF_FALSE;
                break;
            }
            size -= i;
            p += i;
        }
        _close(h);
    }
    else {
        result = TSF_FALSE;
    }

    return result;
}
#endif

#ifdef TSF_SUPPORT_INPUT
TsfBool TSF_API TsfReadFile(const TSF_TCHAR *fileName, void *data, size_t size, size_t *fileLength)
{
    int h;
    long len;
    TsfBool result = TSF_TRUE;
    TsfByte *p = (TsfByte*)data;
    unsigned n;
    int i;

    if (data != NULL) memset(data, '\0', size);

#ifdef TSF_UNICODE
    h = _wopen(fileName, _O_RDONLY | _O_BINARY);
#else
    h = _open(fileName, _O_RDONLY | _O_BINARY);
#endif
    if (h != -1) {
        len = _filelength(h);
        if (len >= 0) {
            if (fileLength != NULL) *fileLength = (size_t)len;
            if ((size_t)len <= size) {
                while (len) {
                    n = (len > INT_MAX) ? INT_MAX : (unsigned)len;
                    if ((i = _read(h, p, n)) <= 0) {
                        result = TSF_FALSE;
                        break;
                    }
                    len -= i;
                    p += i;
                }
            }
            else {
                result = TSF_FALSE;
            }
        }
        else {
            result = TSF_FALSE;
        }
        _close(h);
    }
    else {
        result = TSF_FALSE;
    }

    return result;
}
#endif

#else

#include <io.h>
#include <stdio.h>

#ifdef TSF_SUPPORT_OUTPUT
TsfBool TSF_API TsfWriteFile(TsfObject *tsf, const TSF_TCHAR *fileName)
{
    FILE *fp;
    TsfBool result = TSF_TRUE;

    TsfStdAssert(tsf);
    if (tsf->Status == TSFS_ERROR) return TSF_FALSE;

#ifdef TSF_UNICODE
    fp = _wfopen(fileName, L"wb");
#else
    fp = fopen(fileName, "wb");
#endif
    if (fp) {
        if (fwrite(TsfDataBase(tsf), TsfDataSize(tsf), 1, fp) != 1) result = TSF_FALSE;
        fclose(fp);
    }
    else {
        result = TSF_FALSE;
    }

    return result;
}
#endif

#ifdef TSF_SUPPORT_INPUT
TsfBool TSF_API TsfReadFile(const TSF_TCHAR *fileName, void *data, size_t size, size_t *fileLength)
{
    FILE *fp;
    size_t len;
    TsfBool result = TSF_TRUE;

    if (data != NULL) memset(data, '\0', size);

#ifdef TSF_UNICODE
    fp = _wfopen(fileName, L"rb");
#else
    fp = fopen(fileName, "rb");
#endif
    if (fp) {
        len = _filelength(_fileno(fp));
        if (fileLength != NULL) *fileLength = len;
        if (len <= size) {
            if (fread(data, len, 1, fp) != 1) result = TSF_FALSE;
        }
        else {
            result = TSF_FALSE;
        }
        fclose(fp);
    }
    else {
        result = TSF_FALSE;
    }

    return result;
}
#endif

#endif

#ifdef TSF_SUPPORT_INPUT
TsfBool TSF_API TsfOpenFile(TsfObject *tsf, const TSF_TCHAR *fileName, void *data, size_t size)
{
    size_t len;

    if (!TsfReadFile(fileName, data, size, &len)) return TSF_FALSE;
    TsfOpen(tsf, data, len);

    return TSF_TRUE;
}
#endif

#endif
