/*
 * -----------------------------------------------------------------------------
 * Tagged Stream Format V1.1.19
 * Copyright (c) 2017-2019 Rudy Tellert Elektronik
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy 
 * of this software and associated documentation files (the "Software"), to deal 
 * in the Software without restriction, including without limitation the rights 
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 * SOFTWARE.
 * -----------------------------------------------------------------------------
 */

#ifndef __TSF_H
#define __TSF_H

/*- Switches for target platform ---------------------------------------------*/

/* #define TSF_DEBUG */ /* inserts debug code (warning: compile entire project if changed) */
#define TSF_SUPPORT_C99 /* compiler supports C99 */
#define TSF_SUPPORT_WCHAR_T /* compiler supports wchar_t */
#define TSF_SUPPORT_WIN32 /* compiler supports WIN32 */
#define TSF_SUPPORT_FILE /* compiler supports file I/O */
#define TSF_USE_LOW_LEVEL_FILE_ACCESS /* uses open() instead of fopen() */

#define TSF_SUPPORT_INPUT /* generate code for input */
#define TSF_SUPPORT_OUTPUT /* generate code for output */
#define TSF_SUPPORT_REV /* support for both little and big endian */
#define TSF_SUPPORT_REV_ACCESS /* support for access of the reverse property */
#define TSF_SUPPORT_REV_CHECK /* generate code for endian check */
#define TSF_SUPPORT_REV_FIELD /* support for additional reverse field (faster code) */
#define TSF_SUPPORT_NOP /* support for the NOP token */
#define TSF_SUPPORT_SIMPLE_EXT /* support for simple extension tokens */
#define TSF_SUPPORT_SIMPLE_ARRAY /* support for 1-dimensional data arrays */
#define TSF_SUPPORT_LONG_ID /* support for long ids */
#define TSF_SUPPORT_TAG_ORDER_CHECK /* enable ids sort order check */
#define TSF_SUPPORT_STRICT_PAR_CHECK /* enable strict parameter check */
#define TSF_SUPPORT_INT /* support for integer type */
#define TSF_SUPPORT_UINT /* support for unsigned integer type */
#define TSF_SUPPORT_FLOAT /* support for floating point format types */
#define TSF_SUPPORT_UUID /* support for UUID type */

#define TSF_LEVEL_COUNT 8 /* max. level of embedded collections */
/* #define TSF_WRITE_SIZE0 */ /* write objects of size 0 */
/* #define TSF_WRITE_VAL0 */ /* write objects of value 0 */

/*- Include ------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <limits.h>
#ifdef TSF_SUPPORT_C99
#  include <stdint.h>
#  include <stdbool.h>
#endif

/*- Basic definitions --------------------------------------------------------*/

#ifndef TSF_EXPORT
#  define TSF_EXPORT
#endif
#ifndef TSF_API
#  define TSF_API
#endif

#if defined(UNICODE) || defined(_UNICODE)
#  define TSF_UNICODE
#endif

#if defined(TSF_SUPPORT_SIMPLE_EXT) || defined(TSF_SUPPORT_SIMPLE_ARRAY)
#  define TSF_SUPPORT_EXT
#endif

#ifndef TSF_SUPPORT_REV
#ifdef TSF_SUPPORT_REV_ACCESS
#undef TSF_SUPPORT_REV_ACCESS
#endif
#ifdef TSF_SUPPORT_REV_CHECK
#undef TSF_SUPPORT_REV_CHECK
#endif
#ifdef TSF_SUPPORT_REV_FIELD
#undef TSF_SUPPORT_REV_FIELD
#endif
#endif

#if !defined(TSF_POINTER_MODIFIER)
#  define TSF_POINTER_MODIFIER const
#endif

#ifdef __cplusplus
#  define TSF_USE_EXTERN_C
#  define TSF_USE_INLINE
#endif

#ifndef TSF_SUPPORT_C99
#  define INT8_MAX   0x7f
#  define INT16_MAX  0x7fff
#  define INT32_MAX  0x7fffffff
#  define INT64_MAX  0x7fffffffffffffff
#  define INT8_MIN   (-0x7f-1)
#  define INT16_MIN  (-0x7fff-1)
#  define INT32_MIN  (-0x7fffffff-1)
#  define INT64_MIN  (-0x7fffffffffffffff-1)
#  define UINT8_MAX  0xffU
#  define UINT16_MAX 0xffffU
#  define UINT32_MAX 0xffffffffUL
#  define UINT64_MAX 0xffffffffffffffffULL
#endif

#if SHRT_MAX == INT8_MAX
#  define TSF_SHRT_BITS 8
#elif SHRT_MAX == INT16_MAX
#  define TSF_SHRT_BITS 16
#elif SHRT_MAX == INT32_MAX
#  define TSF_SHRT_BITS 32
#elif SHRT_MAX == INT64_MAX
#  define TSF_SHRT_BITS 64
#endif

#if INT_MAX == INT8_MAX
#  define TSF_INT_BITS 8
#elif INT_MAX == INT16_MAX
#  define TSF_INT_BITS 16
#elif INT_MAX == INT32_MAX
#  define TSF_INT_BITS 32
#elif INT_MAX == INT64_MAX
#  define TSF_INT_BITS 64
#endif

#if LONG_MAX == INT8_MAX
#  define TSF_LONG_BITS 8
#elif LONG_MAX == INT16_MAX
#  define TSF_LONG_BITS 16
#elif LONG_MAX == INT32_MAX
#  define TSF_LONG_BITS 32
#elif LONG_MAX == INT64_MAX
#  define TSF_LONG_BITS 64
#endif

#if LLONG_MAX == INT8_MAX
#  define TSF_LLONG_BITS 8
#elif LLONG_MAX == INT16_MAX
#  define TSF_LLONG_BITS 16
#elif LLONG_MAX == INT32_MAX
#  define TSF_LLONG_BITS 32
#elif LLONG_MAX == INT64_MAX
#  define TSF_LLONG_BITS 64
#endif

#if USHRT_MAX == UINT8_MAX
#  define TSF_USHRT_BITS 8
#elif USHRT_MAX == UINT16_MAX
#  define TSF_USHRT_BITS 16
#elif USHRT_MAX == UINT32_MAX
#  define TSF_USHRT_BITS 32
#elif USHRT_MAX == UINT64_MAX
#  define TSF_USHRT_BITS 64
#endif

#if UINT_MAX == UINT8_MAX
#  define TSF_UINT_BITS 8
#elif UINT_MAX == UINT16_MAX
#  define TSF_UINT_BITS 16
#elif UINT_MAX == UINT32_MAX
#  define TSF_UINT_BITS 32
#elif UINT_MAX == UINT64_MAX
#  define TSF_UINT_BITS 64
#endif

#if ULONG_MAX == UINT8_MAX
#  define TSF_ULONG_BITS 8
#elif ULONG_MAX == UINT16_MAX
#  define TSF_ULONG_BITS 16
#elif ULONG_MAX == UINT32_MAX
#  define TSF_ULONG_BITS 32
#elif ULONG_MAX == UINT64_MAX
#  define TSF_ULONG_BITS 64
#endif

#if ULLONG_MAX == UINT8_MAX
#  define TSF_ULLONG_BITS 8
#elif ULLONG_MAX == UINT16_MAX
#  define TSF_ULLONG_BITS 16
#elif ULLONG_MAX == UINT32_MAX
#  define TSF_ULLONG_BITS 32
#elif ULLONG_MAX == UINT64_MAX
#  define TSF_ULLONG_BITS 64
#endif

#ifndef SIZE_MAX
#  define SIZE_MAX UINT_MAX
#endif

#if SIZE_MAX == UINT8_MAX
#  define TSF_SIZE_BITS 8
#elif SIZE_MAX == UINT16_MAX
#  define TSF_SIZE_BITS 16
#elif SIZE_MAX == UINT32_MAX
#  define TSF_SIZE_BITS 32
#elif SIZE_MAX == UINT64_MAX
#  define TSF_SIZE_BITS 64
#endif

#ifdef TSF_SUPPORT_C99
typedef int8_t   TsfInt8;
typedef int16_t  TsfInt16;
typedef int32_t  TsfInt32;
typedef int64_t  TsfInt64;
typedef uint8_t  TsfUInt8;
typedef uint16_t TsfUInt16;
typedef uint32_t TsfUInt32;
typedef uint64_t TsfUInt64;
#else
typedef signed char        TsfInt8;
typedef signed short       TsfInt16;
typedef signed long        TsfInt32;
typedef signed long long   TsfInt64;
typedef unsigned char      TsfUInt8;
typedef unsigned short     TsfUInt16;
typedef unsigned long      TsfUInt32;
typedef unsigned long long TsfUInt64;
#endif

#ifdef TSF_UNICODE
#  define TSF_T(str) L ## str
#  define TSF_TEXT(str) L ## str
#  ifdef TSF_SUPPORT_WCHAR_T
typedef wchar_t TSF_TCHAR;
#  else
typedef TsfUInt16 TSF_TCHAR;
#  endif
#else
#  define TSF_T(str) str
#  define TSF_TEXT(str) str
typedef char TSF_TCHAR;
#endif
typedef TSF_TCHAR *TSF_TSTR;
typedef const TSF_TCHAR *TSF_CTSTR;

#ifdef TSF_DEBUG
#  define TSF_SUPPORT_REV_CHECK
#endif

/*----------------------------------------------------------------------------*/

#ifdef TSF_USE_EXTERN_C
extern "C" {
#endif

    typedef TsfUInt8  TsfByte;
    typedef TsfUInt16 TsfWord;
#ifdef TSF_SUPPORT_LONG_ID
    typedef size_t  TsfId;
#else
    typedef TsfByte TsfId;
#endif
    typedef TsfByte TsfLevel;
    typedef TsfByte TsfUuid[16];

#ifdef TSF_SUPPORT_C99
#define TSF_FALSE false
#define TSF_TRUE true
    typedef bool TsfBool;
#else
    enum { TSF_FALSE, TSF_TRUE };
    typedef TsfByte TsfBool;
#endif

    enum { TSFS_DATA, TSFS_END, TSFS_END_OF_STREAM, TSFS_ERROR };
    typedef TsfByte TsfStatusType;

    enum { TSFH_NONE, TSFH_NORMAL
#ifdef TSF_SUPPORT_REV
        , TSFH_REV
#ifdef TSF_SUPPORT_REV_CHECK
        , TSFH_LITTLE_ENDIAN, TSFH_BIG_ENDIAN 
#endif
#endif
    };
    typedef TsfByte TsfHeaderType;

    enum {
        TSFE_DEFAULT = 0x00, TSFE_NULL = 0x01, TSFE_REDEF = 0x02,
        TSFE_FIXED_ARRAY = 0x11, TSFE_VAR_ARRAY = 0x12
    };

    enum {
        TSFT_NONE, TSFT_END, TSFT_END_OF_STREAM, TSFT_ERROR,
        TSFT_END_OF_ARRAY, TSFT_END_OF_COLLECTION,
        TSFT_DATA,
        TSFT_COLLECTION, TSFT_ARRAY,
        TSFT_EXT
    };
    typedef TsfByte TsfTokenType;

    typedef struct s_TsfLevelData {
        size_t Count;
#ifdef TSF_SUPPORT_TAG_ORDER_CHECK
        TsfId  CurrId;
#endif
    } TsfLevelData;

    typedef struct s_TsfObject {
#ifdef TSF_DEBUG
        volatile TsfUInt32 StartPattern;
#endif

        /* layer 0 */
        size_t        TotalSize;

        TsfByte       TSF_POINTER_MODIFIER *Data; /* stream */
        size_t        RemainingSize; /* stream */
        /* layer 1 */
        size_t        DataSize;
        TsfHeaderType Header;
#ifdef TSF_SUPPORT_REV_FIELD
        TsfBool       Reverse;
#endif
        TsfStatusType Status;
#ifdef TSF_SUPPORT_SIMPLE_ARRAY
        size_t        DataCount;
#endif
#ifdef TSF_SUPPORT_EXT
        TsfByte       Ext;
#endif
        TsfLevel      Level;
        TsfLevelData  LevelData[TSF_LEVEL_COUNT];
        size_t        Count;
        TsfId         Id;

#ifdef TSF_DEBUG
        volatile TsfUInt32 EndPattern;
#endif
    } TsfObject;

    typedef struct s_TsfStruct {
        TsfByte Id;
        TsfByte Type;
        TsfWord Number;
    } TsfStruct;

/*- uuid ---------------------------------------------------------------------*/

#ifdef TSF_SUPPORT_UUID
    TSF_EXPORT void TSF_API TsfUuidFromData(TsfUuid uuid, const void *data);
    TSF_EXPORT void TSF_API TsfUuidFromString(TsfUuid uuid, const char *str);
#ifdef TSF_SUPPORT_WCHAR_T
    TSF_EXPORT void TSF_API TsfUuidFromWideString(TsfUuid uuid, const wchar_t *wstr);
#endif
    TSF_EXPORT void TSF_API TsfUuidFromNonce(TsfUuid uuid, TsfUInt64 nonce);
    TSF_EXPORT TsfBool TSF_API TsfUuidIsEqual(const TsfUuid uuid1, const TsfUuid uuid2);
    TSF_EXPORT TsfBool TSF_API TsfUuidIsGood(const TsfUuid uuid);
#define TsfUuidIsBad(uuid) (!(TsfUuidIsGood(uuid)))
    TSF_EXPORT void TSF_API TsfUuidEmpty(TsfUuid uuid);
#ifdef TSF_SUPPORT_WIN32
    TSF_EXPORT void TSF_API TsfUuidNew(TsfUuid uuid);
#endif
#endif

/*- tsf ----------------------------------------------------------------------*/

#define TsfGetStatus(tsf) (tsf->Status)
#define TsfIsGood(tsf) (tsf->Status != TSFS_ERROR)
#define TsfIsBad(tsf) (tsf->Status == TSFS_ERROR)
#define TsfGetHeaderType(tsf) (tsf->Header)
#define TsfGetId(tsf) (tsf->Id)
#define TsfGetCount(tsf) (tsf->Count)
#define TsfGetLevel(tsf) (tsf->Level)
#ifdef TSF_SUPPORT_EXT
#  define TsfGetExt(tsf) (tsf->Ext)
#else
#  define TsfGetExt(tsf) (0)
#endif
#define TsfGetData(tsf) ((const void *)tsf->Data)
#define TsfGetDataChars(tsf) ((const char *)tsf->Data)
#define TsfGetDataBytes(tsf) ((const TsfByte *)tsf->Data)
#define TsfGetDataSize(tsf) (tsf->DataSize)
#ifdef TSF_SUPPORT_SIMPLE_ARRAY
#  define TsfGetDataCount(tsf) (tsf->DataCount)
#endif
#ifdef TSF_SUPPORT_REV_CHECK
TSF_EXPORT TsfBool TSF_API TsfIsLittleEndian(void);
#endif
#ifdef TSF_DEBUG
    TSF_EXPORT void TSF_API TsfClose(TsfObject *tsf);
#else
#  define TsfClose(tsf)
#endif

/*- read (layer 1) -----------------------------------------------------------*/

#ifdef TSF_SUPPORT_INPUT
    TSF_EXPORT size_t TSF_API TsfGetPos(TsfObject *tsf);
    TSF_EXPORT TsfBool TSF_API TsfSetPos(TsfObject *tsf, size_t pos);
#ifdef TSF_SUPPORT_FILE
    TSF_EXPORT TsfBool TSF_API TsfReadFile(const TSF_TCHAR *fileName, void *data, size_t size, size_t *fileLength);
    TSF_EXPORT TsfBool TSF_API TsfOpenFile(TsfObject *tsf, const TSF_TCHAR *fileName, void *data, size_t size);
#else 
#  define TsfReadFile(fileName, data, size, fileLength) (TSF_FALSE)
#  define TsfOpenFile(tsf, fileName, data, size) (TSF_FALSE)
#endif
    TSF_EXPORT void TSF_API TsfOpen(TsfObject *tsf, const void *data, size_t size);
    TSF_EXPORT TsfTokenType TSF_API TsfReadToken(TsfObject *tsf);
#ifdef TSF_SUPPORT_SIMPLE_ARRAY
    TSF_EXPORT TsfBool TSF_API TsfSelectNextItem(TsfObject *tsf);
#endif

#ifdef TSF_SUPPORT_REV_ACCESS
    TSF_EXPORT TsfBool TSF_API TsfGetReverse(TsfObject *tsf);
    TSF_EXPORT void TSF_API TsfSetReverse(TsfObject *tsf, TsfBool isRev);
#else
#  define TsfGetReverse(tsf) (TSF_FALSE)
#  define TsfSetReverse(tsf, b)
#endif

#ifdef TSF_SUPPORT_REV_FIELD
#  define TsfIsReverse(tsf) (tsf->Reverse)
#else
#  define TsfIsReverse(tsf) ((tsf->Header == TSFH_REV) ? TSF_TRUE : TSF_FALSE)
#endif
    TSF_EXPORT TsfBool TSF_API TsfReadData(TsfObject *tsf, void *data, size_t size);
    TSF_EXPORT TsfBool TSF_API TsfReadObject(TsfObject *tsf, void *data, size_t size);
#define TsfSkipData(tsf) TsfReadData(tsf, NULL, tsf->DataSize)

#ifdef TSF_SUPPORT_REV
    TSF_EXPORT void TSF_API TsfSortData(TsfObject *tsf, void *data, size_t size);
    TSF_EXPORT TsfBool TSF_API TsfReadSortedData(TsfObject *tsf, void *data, size_t size);
    TSF_EXPORT TsfBool TSF_API TsfReadSortedObject(TsfObject *tsf, void *data, size_t size);
#else
#  define TsfSortData(tsf, data, n)
#  define TsfReadSortedData(tsf, data, n) TsfReadData(tsf, data, n)
#endif

#ifdef TSF_SUPPORT_INT
    TSF_EXPORT TsfBool TSF_API TsfReadInt8(TsfObject *tsf, TsfInt8 *data);
    TSF_EXPORT TsfBool TSF_API TsfReadInt16(TsfObject *tsf, TsfInt16 *data);
    TSF_EXPORT TsfBool TSF_API TsfReadInt32(TsfObject *tsf, TsfInt32 *data);
    TSF_EXPORT TsfBool TSF_API TsfReadInt64(TsfObject *tsf, TsfInt64 *data);
#endif
#ifdef TSF_SUPPORT_UINT
    TSF_EXPORT TsfBool TSF_API TsfReadUInt8(TsfObject *tsf, TsfUInt8 *data);
    TSF_EXPORT TsfBool TSF_API TsfReadUInt16(TsfObject *tsf, TsfUInt16 *data);
    TSF_EXPORT TsfBool TSF_API TsfReadUInt32(TsfObject *tsf, TsfUInt32 *data);
    TSF_EXPORT TsfBool TSF_API TsfReadUInt64(TsfObject *tsf, TsfUInt64 *data);
#endif
#ifdef TSF_SUPPORT_FLOAT
    TSF_EXPORT TsfBool TSF_API TsfReadFloat32(TsfObject *tsf, float *data);
    TSF_EXPORT TsfBool TSF_API TsfReadFloat64(TsfObject *tsf, double *data);
#endif
#ifdef TSF_SUPPORT_UUID
    TSF_EXPORT TsfBool TSF_API TsfReadUuid(TsfObject *tsf, TsfUuid *data);
#endif

/*- read (layer2 ) -----------------------------------------------------------*/

    TSF_EXPORT TsfBool TSF_API TsfSelectObject(TsfObject *tsf, TsfId id);
#define TsfSelect(tsf, id) TsfSelectObject(tsf, id)
    TSF_EXPORT size_t TSF_API TsfReadArraySize(TsfObject *tsf, TsfId id);
#define TsfSelectArray(tsf, id) (TsfReadArraySize(tsf, id) != 0)
    TSF_EXPORT TsfBool TSF_API TsfSkipCollectionOrArray(TsfObject *tsf, TsfBool skipCollectionOnly);
#define TsfSkipCollection(tsf) TsfSkipCollectionOrArray(tsf, TSF_TRUE)
#define TsfSkipArray(tsf) TsfSkipCollectionOrArray(tsf, TSF_FALSE)

#ifdef TSF_SUPPORT_FLOAT 
#  define TsfReadFloat(tsf, data) TsfReadFloat32(tsf, data)
#  define TsfReadDouble(tsf, data) TsfReadFloat64(tsf, data)
#endif

#ifdef TSF_SUPPORT_INT
#  define TsfReadChar(tsf, data) TsfReadInt8(tsf, (TsfInt8*)data)
#  if TSF_SHRT_BITS == 8
#    define TsfReadShort(tsf, data) TsfReadInt8(tsf, (TsfInt8*)data)
#  elif TSF_SHRT_BITS == 16
#    define TsfReadShort(tsf, data) TsfReadInt16(tsf, (TsfInt16*)data)
#  elif TSF_SHRT_BITS == 32
#    define TsfReadShort(tsf, data) TsfReadInt32(tsf, (TsfInt32*)data)
#  elif TSF_SHRT_BITS == 64
#    define TsfReadShort(tsf, data) TsfReadInt64(tsf, (TsfInt64*)data)
#  endif

#  if TSF_INT_BITS == 8
#    define TsfReadInt(tsf, data) TsfReadInt8(tsf, (TsfInt8*)data)
#  elif TSF_INT_BITS == 16
#    define TsfReadInt(tsf, data) TsfReadInt16(tsf, (TsfInt16*)data)
#  elif TSF_INT_BITS == 32
#    define TsfReadInt(tsf, data) TsfReadInt32(tsf, (TsfInt32*)data)
#  elif TSF_INT_BITS == 64
#    define TsfReadInt(tsf, data) TsfReadInt64(tsf, (TsfInt64*)data)
#  endif

#  if TSF_LONG_BITS == 8
#    define TsfReadLong(tsf, data) TsfReadInt8(tsf, (TsfInt8*)data)
#  elif TSF_LONG_BITS == 16
#    define TsfReadLong(tsf, data) TsfReadInt16(tsf, (TsfInt16*)data)
#  elif TSF_LONG_BITS == 32
#    define TsfReadLong(tsf, data) TsfReadInt32(tsf, (TsfInt32*)data)
#  elif TSF_LONG_BITS == 64
#    define TsfReadLong(tsf, data) TsfReadInt64(tsf, (TsfInt64*)data)
#  endif

#  if TSF_LLONG_BITS == 8
#    define TsfReadLLong(tsf, data) TsfReadInt8(tsf, (TsfInt8*)data)
#  elif TSF_LLONG_BITS == 16
#    define TsfReadLLong(tsf, data) TsfReadInt16(tsf, (TsfInt16*)data)
#  elif TSF_LLONG_BITS == 32
#    define TsfReadLLong(tsf, data) TsfReadInt32(tsf, (TsfInt32*)data)
#  elif TSF_LLONG_BITS == 64
#    define TsfReadLLong(tsf, data) TsfReadInt64(tsf, (TsfInt64*)data)
#  endif
#endif

#ifdef TSF_SUPPORT_UINT
#  define TsfReadUChar(tsf, data) TsfReadUInt8(tsf, (TsfUInt8*)data)
#  if TSF_USHRT_BITS == 8
#    define TsfReadUShort(tsf, data) TsfReadUInt8(tsf, (TsfUInt8*)data)
#  elif TSF_USHRT_BITS == 16
#    define TsfReadUShort(tsf, data) TsfReadUInt16(tsf, (TsfUInt16*)data)
#  elif TSF_USHRT_BITS == 32
#    define TsfReadUShort(tsf, data) TsfReadUInt32(tsf, (TsfUInt32*)data)
#  elif TSF_USHRT_BITS == 64
#    define TsfReadUShort(tsf, data) TsfReadUInt64(tsf, (TsfUInt64*)data)
#  endif

#  if TSF_UINT_BITS == 8
#    define TsfReadUInt(tsf, data) TsfReadUInt8(tsf, (TsfUInt8*)data)
#  elif TSF_UINT_BITS == 16
#    define TsfReadUInt(tsf, data) TsfReadUInt16(tsf, (TsfUInt16*)data)
#  elif TSF_UINT_BITS == 32
#    define TsfReadUInt(tsf, data) TsfReadUInt32(tsf, (TsfUInt32*)data)
#  elif TSF_UINT_BITS == 64
#    define TsfReadUInt(tsf, data) TsfReadUInt64(tsf, (TsfUInt64*)data)
#  endif

#  if TSF_ULONG_BITS == 8
#    define TsfReadULong(tsf, data) TsfReadUInt8(tsf, (TsfUInt8*)data)
#  elif TSF_ULONG_BITS == 16
#    define TsfReadULong(tsf, data) TsfReadUInt16(tsf, (TsfUInt16*)data)
#  elif TSF_ULONG_BITS == 32
#    define TsfReadULong(tsf, data) TsfReadUInt32(tsf, (TsfUInt32*)data)
#  elif TSF_ULONG_BITS == 64
#    define TsfReadULong(tsf, data) TsfReadUInt64(tsf, (TsfUInt64*)data)
#  endif

#  if TSF_ULLONG_BITS == 8
#    define TsfReadULLong(tsf, data) TsfReadUInt8(tsf, (TsfUInt8*)data)
#  elif TSF_ULLONG_BITS == 16
#    define TsfReadULLong(tsf, data) TsfReadUInt16(tsf, (TsfUInt16*)data)
#  elif TSF_ULLONG_BITS == 32
#    define TsfReadULLong(tsf, data) TsfReadUInt32(tsf, (TsfUInt32*)data)
#  elif TSF_ULLONG_BITS == 64
#    define TsfReadULLong(tsf, data) TsfReadUInt64(tsf, (TsfUInt64*)data)
#  endif

#  if TSF_SIZE_BITS == 8
#    define TsfReadSize(tsf, data) TsfReadUInt8(tsf, (TsfUInt8*)data)
#  elif TSF_SIZE_BITS == 16
#    define TsfReadSize(tsf, data) TsfReadUInt16(tsf, (TsfUInt16*)data)
#  elif TSF_SIZE_BITS == 32
#    define TsfReadSize(tsf, data) TsfReadUInt32(tsf, (TsfUInt32*)data)
#  elif TSF_SIZE_BITS == 64
#    define TsfReadSize(tsf, data) TsfReadUInt64(tsf, (TsfUInt64*)data)
#  endif
#endif

#endif

/*- tsfwrite -----------------------------------------------------------------*/

#ifdef TSF_SUPPORT_OUTPUT
#define TsfDataSize(tsf) (tsf->TotalSize - tsf->RemainingSize)
#define TsfDataBase(tsf) ((const TsfByte *)(tsf->Data - TsfDataSize(tsf)))
#ifdef TSF_SUPPORT_FILE
TSF_EXPORT TsfBool TSF_API TsfWriteFile(TsfObject *tsf, const TSF_TCHAR *fileName);
#else 
#  define TsfWriteFile(tsf, fileName) (TSF_FALSE)
#endif
    TSF_EXPORT void TSF_API TsfCreate(TsfObject *tsf, void *data, size_t availableSize, TsfHeaderType h, TsfBool writeHeader);
    TSF_EXPORT TsfBool TSF_API TsfWriteData(TsfObject *tsf, const void *data, size_t size);
#ifdef TSF_SUPPORT_REV
    TSF_EXPORT TsfBool TSF_API TsfWriteSortedData(TsfObject *tsf, const void *data, size_t size);
#else
#  define TsfWriteSortedData(tsf, data, size) TsfWriteData(tsf, data, size)
#endif

    TSF_EXPORT TsfBool TSF_API TsfWriteArray(TsfObject *tsf, TsfId id, size_t size);
    TSF_EXPORT TsfBool TSF_API TsfWriteEnd(TsfObject *tsf);

    TSF_EXPORT TsfBool TSF_API TsfWriteObject(TsfObject *tsf, TsfId id, const void *data, size_t size);
    TSF_EXPORT TsfBool TSF_API TsfWriteObject0(TsfObject *tsf, TsfId id, const void *data, size_t size);
    TSF_EXPORT TsfBool TSF_API TsfWriteSortedObject(TsfObject *tsf, TsfId id, const void *data, size_t size);
    TSF_EXPORT TsfBool TSF_API TsfWriteSortedObject0(TsfObject *tsf, TsfId id, const void *data, size_t size);

#ifdef TSF_SUPPORT_SIMPLE_ARRAY
    TSF_EXPORT TsfBool TSF_API TsfWriteFixedArray(TsfObject *tsf, TsfId id, size_t count, size_t itemSize);
#  define TsfWriteFixedArrayData(tsf, data) TsfWriteData(tsf, data, tsf->DataSize)
#  define TsfWriteFixedArraySortedData(tsf, data) TsfWriteSortedData(tsf, data, tsf->DataSize)
#  ifdef TSF_SUPPORT_INT
    TSF_EXPORT TsfBool TSF_API TsfWriteFixedArrayInt8(TsfObject *tsf, TsfInt8 data);
    TSF_EXPORT TsfBool TSF_API TsfWriteFixedArrayInt16(TsfObject *tsf, TsfInt16 data);
    TSF_EXPORT TsfBool TSF_API TsfWriteFixedArrayInt32(TsfObject *tsf, TsfInt32 data);
    TSF_EXPORT TsfBool TSF_API TsfWriteFixedArrayInt64(TsfObject *tsf, TsfInt64 data);
#  endif
#  ifdef TSF_SUPPORT_UINT
    TSF_EXPORT TsfBool TSF_API TsfWriteFixedArrayUInt8(TsfObject *tsf, TsfUInt8 data);
    TSF_EXPORT TsfBool TSF_API TsfWriteFixedArrayUInt16(TsfObject *tsf, TsfUInt16 data);
    TSF_EXPORT TsfBool TSF_API TsfWriteFixedArrayUInt32(TsfObject *tsf, TsfUInt32 data);
    TSF_EXPORT TsfBool TSF_API TsfWriteFixedArrayUInt64(TsfObject *tsf, TsfUInt64 data);
#  endif
#  ifdef TSF_SUPPORT_FLOAT
    TSF_EXPORT TsfBool TSF_API TsfWriteFixedArrayFloat32(TsfObject *tsf, float data);
    TSF_EXPORT TsfBool TSF_API TsfWriteFixedArrayFloat64(TsfObject *tsf, double data);
#  endif
#  ifdef TSF_SUPPORT_UUID
    TSF_EXPORT TsfBool TSF_API TsfWriteFixedArrayUuid(TsfObject *tsf, const TsfUuid data);
#  endif
    TSF_EXPORT TsfBool TSF_API TsfWriteVarArray(TsfObject *tsf, TsfId id, size_t count);
    TSF_EXPORT TsfBool TSF_API TsfWriteVarArrayData(TsfObject *tsf, const void *data, size_t size);
    TSF_EXPORT TsfBool TSF_API TsfWriteVarArraySortedData(TsfObject *tsf, const void *data, size_t size);
#  ifdef TSF_SUPPORT_INT
    TSF_EXPORT TsfBool TSF_API TsfWriteVarArrayInt8(TsfObject *tsf, TsfInt8 data);
    TSF_EXPORT TsfBool TSF_API TsfWriteVarArrayInt16(TsfObject *tsf, TsfInt16 data);
    TSF_EXPORT TsfBool TSF_API TsfWriteVarArrayInt32(TsfObject *tsf, TsfInt32 data);
    TSF_EXPORT TsfBool TSF_API TsfWriteVarArrayInt64(TsfObject *tsf, TsfInt64 data);
#  endif
#  ifdef TSF_SUPPORT_UINT
    TSF_EXPORT TsfBool TSF_API TsfWriteVarArrayUInt8(TsfObject *tsf, TsfUInt8 data);
    TSF_EXPORT TsfBool TSF_API TsfWriteVarArrayUInt16(TsfObject *tsf, TsfUInt16 data);
    TSF_EXPORT TsfBool TSF_API TsfWriteVarArrayUInt32(TsfObject *tsf, TsfUInt32 data);
    TSF_EXPORT TsfBool TSF_API TsfWriteVarArrayUInt64(TsfObject *tsf, TsfUInt64 data);
#  endif
#  ifdef TSF_SUPPORT_FLOAT
    TSF_EXPORT TsfBool TSF_API TsfWriteVarArrayFloat32(TsfObject *tsf, float data);
    TSF_EXPORT TsfBool TSF_API TsfWriteVarArrayFloat64(TsfObject *tsf, double data);
#  endif
#  ifdef TSF_SUPPORT_UUID
    TSF_EXPORT TsfBool TSF_API TsfWriteVarArrayUuid(TsfObject *tsf, const TsfUuid data);
#  endif
#endif

#ifdef TSF_SUPPORT_INT
    TSF_EXPORT TsfBool TSF_API TsfWriteInt8(TsfObject *tsf, TsfId id, TsfInt8 data);
    TSF_EXPORT TsfBool TSF_API TsfWriteInt16(TsfObject *tsf, TsfId id, TsfInt16 data);
    TSF_EXPORT TsfBool TSF_API TsfWriteInt32(TsfObject *tsf, TsfId id, TsfInt32 data);
    TSF_EXPORT TsfBool TSF_API TsfWriteInt64(TsfObject *tsf, TsfId id, TsfInt64 data);
#endif
#ifdef TSF_SUPPORT_UINT
    TSF_EXPORT TsfBool TSF_API TsfWriteUInt8(TsfObject *tsf, TsfId id, TsfUInt8 data);
    TSF_EXPORT TsfBool TSF_API TsfWriteUInt16(TsfObject *tsf, TsfId id, TsfUInt16 data);
    TSF_EXPORT TsfBool TSF_API TsfWriteUInt32(TsfObject *tsf, TsfId id, TsfUInt32 data);
    TSF_EXPORT TsfBool TSF_API TsfWriteUInt64(TsfObject *tsf, TsfId id, TsfUInt64 data);
#endif
#ifdef TSF_SUPPORT_FLOAT
    TSF_EXPORT TsfBool TSF_API TsfWriteFloat32(TsfObject *tsf, TsfId id, float data);
    TSF_EXPORT TsfBool TSF_API TsfWriteFloat64(TsfObject *tsf, TsfId id, double data);
#endif
#ifdef TSF_SUPPORT_UUID
    TSF_EXPORT TsfBool TSF_API TsfWriteUuid(TsfObject *tsf, TsfId id, const TsfUuid data);
#endif

#ifdef TSF_USE_EXTERN_C
}
#endif

#ifdef TSF_SUPPORT_FLOAT 
#  define TsfWriteFloat(tsf, id, data) TsfWriteFloat32(tsf, id, data)
#  define TsfWriteDouble(tsf, id, data) TsfWriteFloat64(tsf, id, data)
#endif

#ifdef TSF_SUPPORT_INT
#  define TsfWriteChar(tsf, id, data) TsfWriteInt8(tsf, id, data)
#  if TSF_SHRT_BITS == 8
#    define TsfWriteShort(tsf, id, data) TsfWriteInt8(tsf, id, data)
#  elif TSF_SHRT_BITS == 16
#    define TsfWriteShort(tsf, id, data) TsfWriteInt16(tsf, id, data)
#  elif TSF_SHRT_BITS == 32
#    define TsfWriteShort(tsf, id, data) TsfWriteInt32(tsf, id, data)
#  elif TSF_SHRT_BITS == 64
#    define TsfWriteShort(tsf, id, data) TsfWriteInt64(tsf, id, data)
#  endif

#  if TSF_INT_BITS == 8
#    define TsfWriteInt(tsf, id, data) TsfWriteInt8(tsf, id, data)
#  elif TSF_INT_BITS == 16
#    define TsfWriteInt(tsf, id, data) TsfWriteInt16(tsf, id, data)
#  elif TSF_INT_BITS == 32
#    define TsfWriteInt(tsf, id, data) TsfWriteInt32(tsf, id, data)
#  elif TSF_INT_BITS == 64
#    define TsfWriteInt(tsf, id, data) TsfWriteInt64(tsf, id, data)
#  endif

#  if TSF_LONG_BITS == 8
#    define TsfWriteLong(tsf, id, data) TsfWriteInt8(tsf, id, data)
#  elif TSF_LONG_BITS == 16
#    define TsfWriteLong(tsf, id, data) TsfWriteInt16(tsf, id, data)
#  elif TSF_LONG_BITS == 32
#    define TsfWriteLong(tsf, id, data) TsfWriteInt32(tsf, id, data)
#  elif TSF_LONG_BITS == 64
#    define TsfWriteLong(tsf, id, data) TsfWriteInt64(tsf, id, data)
#  endif

#  if TSF_LLONG_BITS == 8
#    define TsfWriteLLong(tsf, id, data) TsfWriteInt8(tsf, id, data)
#  elif TSF_LLONG_BITS == 16
#    define TsfWriteLLong(tsf, id, data) TsfWriteInt16(tsf, id, data)
#  elif TSF_LLONG_BITS == 32
#    define TsfWriteLLong(tsf, id, data) TsfWriteInt32(tsf, id, data)
#  elif TSF_LLONG_BITS == 64
#    define TsfWriteLLong(tsf, id, data) TsfWriteInt64(tsf, id, data)
#  endif
#endif

#ifdef TSF_SUPPORT_UINT
#  define TsfWriteUChar(tsf, id, data) TsfWriteUInt8(tsf, id, data)
#  if TSF_USHRT_BITS == 8
#    define TsfWriteUShort(tsf, id, data) TsfWriteUInt8(tsf, id, data)
#  elif TSF_USHRT_BITS == 16
#    define TsfWriteUShort(tsf, id, data) TsfWriteUInt16(tsf, id, data)
#  elif TSF_USHRT_BITS == 32
#    define TsfWriteUShort(tsf, id, data) TsfWriteUInt32(tsf, id, data)
#  elif TSF_USHRT_BITS == 64
#    define TsfWriteUShort(tsf, id, data) TsfWriteUInt64(tsf, id, data)
#  endif

#  if TSF_UINT_BITS == 8
#    define TsfWriteUInt(tsf, id, data) TsfWriteUInt8(tsf, id, data)
#  elif TSF_UINT_BITS == 16
#    define TsfWriteUInt(tsf, id, data) TsfWriteUInt16(tsf, id, data)
#elif TSF_UINT_BITS == 32
#  define TsfWriteUInt(tsf, id, data) TsfWriteUInt32(tsf, id, data)
#  elif TSF_UINT_BITS == 64
#    define TsfWriteUInt(tsf, id, data) TsfWriteUInt64(tsf, id, data)
#  endif

#  if TSF_ULONG_BITS == 8
#    define TsfWriteULong(tsf, id, data) TsfWriteUInt8(tsf, id, data)
#  elif TSF_ULONG_BITS == 16
#    define TsfWriteULong(tsf, id, data) TsfWriteUInt16(tsf, id, data)
#  elif TSF_ULONG_BITS == 32
#    define TsfWriteULong(tsf, id, data) TsfWriteUInt32(tsf, id, data)
#  elif TSF_ULONG_BITS == 64
#    define TsfWriteULong(tsf, id, data) TsfWriteUInt64(tsf, id, data)
#  endif

#  if TSF_ULLONG_BITS == 8
#    define TsfWriteULLong(tsf, id, data) TsfWriteUInt8(tsf, id, data)
#  elif TSF_ULLONG_BITS == 16
#    define TsfWriteULLong(tsf, id, data) TsfWriteUInt16(tsf, id, data)
#  elif TSF_ULLONG_BITS == 32
#    define TsfWriteULLong(tsf, id, data) TsfWriteUInt32(tsf, id, data)
#  elif TSF_ULLONG_BITS == 64
#    define TsfWriteULLong(tsf, id, data) TsfWriteUInt64(tsf, id, data)
#  endif

#  if TSF_SIZE_BITS == 8
#    define TsfWriteSize(tsf, id, data) TsfWriteUInt8(tsf, id, data)
#  elif TSF_SIZE_BITS == 16
#    define TsfWriteSize(tsf, id, data) TsfWriteUInt16(tsf, id, data)
#  elif TSF_SIZE_BITS == 32
#    define TsfWriteSize(tsf, id, data) TsfWriteUInt32(tsf, id, data)
#  elif TSF_SIZE_BITS == 64
#    define TsfWriteSize(tsf, id, data) TsfWriteUInt64(tsf, id, data)
#  endif
#endif

#ifdef TSF_SUPPORT_SIMPLE_ARRAY

#  ifdef TSF_SUPPORT_FLOAT 
#    define TsfWriteFixedArrayFloat(tsf, data) TsfWriteFixedArrayFloat32(tsf, data)
#    define TsfWriteFixedArrayDouble(tsf, data) TsfWriteFixedArrayFloat64(tsf, data)
#  endif

#  ifdef TSF_SUPPORT_INT
#    define TsfWriteFixedArrayChar(tsf, data) TsfWriteFixedArrayInt8(tsf, data)
#    if TSF_SHRT_BITS == 8
#      define TsfWriteFixedArrayShort(tsf, data) TsfWriteFixedArrayInt8(tsf, data)
#    elif TSF_SHRT_BITS == 16
#      define TsfWriteFixedArrayShort(tsf, data) TsfWriteFixedArrayInt16(tsf, data)
#    elif TSF_SHRT_BITS == 32
#      define TsfWriteFixedArrayShort(tsf, data) TsfWriteFixedArrayInt32(tsf, data)
#    elif TSF_SHRT_BITS == 64
#      define TsfWriteFixedArrayShort(tsf, data) TsfWriteFixedArrayInt64(tsf, data)
#    endif

#    if TSF_INT_BITS == 8
#      define TsfWriteFixedArrayInt(tsf, data) TsfWriteFixedArrayInt8(tsf, data)
#    elif TSF_INT_BITS == 16
#      define TsfWriteFixedArrayInt(tsf, data) TsfWriteFixedArrayInt16(tsf, data)
#    elif TSF_INT_BITS == 32
#      define TsfWriteFixedArrayInt(tsf, data) TsfWriteFixedArrayInt32(tsf, data)
#    elif TSF_INT_BITS == 64
#      define TsfWriteFixedArrayInt(tsf, data) TsfWriteFixedArrayInt64(tsf, data)
#    endif

#    if TSF_LONG_BITS == 8
#      define TsfWriteFixedArrayLong(tsf, data) TsfWriteFixedArrayInt8(tsf, data)
#    elif TSF_LONG_BITS == 16
#      define TsfWriteFixedArrayLong(tsf, data) TsfWriteFixedArrayInt16(tsf, data)
#    elif TSF_LONG_BITS == 32
#      define TsfWriteFixedArrayLong(tsf, data) TsfWriteFixedArrayInt32(tsf, data)
#    elif TSF_LONG_BITS == 64
#      define TsfWriteFixedArrayLong(tsf, data) TsfWriteFixedArrayInt64(tsf, data)
#    endif

#    if TSF_LLONG_BITS == 8
#      define TsfWriteFixedArrayLLong(tsf, data) TsfWriteFixedArrayInt8(tsf, data)
#    elif TSF_LLONG_BITS == 16
#      define TsfWriteFixedArrayLLong(tsf, data) TsfWriteFixedArrayInt16(tsf, data)
#    elif TSF_LLONG_BITS == 32
#      define TsfWriteFixedArrayLLong(tsf, data) TsfWriteFixedArrayInt32(tsf, data)
#    elif TSF_LLONG_BITS == 64
#      define TsfWriteFixedArrayLLong(tsf, data) TsfWriteFixedArrayInt64(tsf, data)
#    endif
#  endif

#  ifdef TSF_SUPPORT_UINT
#    define TsfWriteFixedArrayUChar(tsf, data) TsfWriteFixedArrayUInt8(tsf, data)
#    if TSF_USHRT_BITS == 8
#      define TsfWriteFixedArrayUShort(tsf, data) TsfWriteFixedArrayUInt8(tsf, data)
#    elif TSF_USHRT_BITS == 16
#      define TsfWriteFixedArrayUShort(tsf, data) TsfWriteFixedArrayUInt16(tsf, data)
#    elif TSF_USHRT_BITS == 32
#      define TsfWriteFixedArrayUShort(tsf, data) TsfWriteFixedArrayUInt32(tsf, data)
#    elif TSF_USHRT_BITS == 64
#      define TsfWriteFixedArrayUShort(tsf, data) TsfWriteFixedArrayUInt64(tsf, data)
#    endif

#    if TSF_UINT_BITS == 8
#      define TsfWriteFixedArrayUInt(tsf, data) TsfWriteFixedArrayUInt8(tsf, data)
#    elif TSF_UINT_BITS == 16
#      define TsfWriteFixedArrayUInt(tsf, data) TsfWriteFixedArrayUInt16(tsf, data)
#    elif TSF_UINT_BITS == 32
#      define TsfWriteFixedArrayUInt(tsf, data) TsfWriteFixedArrayUInt32(tsf, data)
#    elif TSF_UINT_BITS == 64
#      define TsfWriteFixedArrayUInt(tsf, data) TsfWriteFixedArrayUInt64(tsf, data)
#    endif

#    if TSF_ULONG_BITS == 8
#      define TsfWriteFixedArrayULong(tsf, data) TsfWriteFixedArrayUInt8(tsf, data)
#    elif TSF_ULONG_BITS == 16
#      define TsfWriteFixedArrayULong(tsf, data) TsfWriteFixedArrayUInt16(tsf, data)
#    elif TSF_ULONG_BITS == 32
#      define TsfWriteFixedArrayULong(tsf, data) TsfWriteFixedArrayUInt32(tsf, data)
#    elif TSF_ULONG_BITS == 64
#      define TsfWriteFixedArrayULong(tsf, data) TsfWriteFixedArrayUInt64(tsf, data)
#    endif

#    if TSF_ULLONG_BITS == 8
#      define TsfWriteFixedArrayULLong(tsf, data) TsfWriteFixedArrayUInt8(tsf, data)
#    elif TSF_ULLONG_BITS == 16
#      define TsfWriteFixedArrayULLong(tsf, data) TsfWriteFixedArrayUInt16(tsf, data)
#    elif TSF_ULLONG_BITS == 32
#      define TsfWriteFixedArrayULLong(tsf, data) TsfWriteFixedArrayUInt32(tsf, data)
#    elif TSF_ULLONG_BITS == 64
#      define TsfWriteFixedArrayULLong(tsf, data) TsfWriteFixedArrayUInt64(tsf, data)
#    endif

#    if TSF_SIZE_BITS == 8
#      define TsfWriteFixedArraySize(tsf, data) TsfWriteFixedArrayUInt8(tsf, data)
#    elif TSF_SIZE_BITS == 16
#      define TsfWriteFixedArraySize(tsf, data) TsfWriteFixedArrayUInt16(tsf, data)
#    elif TSF_SIZE_BITS == 32
#      define TsfWriteFixedArraySize(tsf, data) TsfWriteFixedArrayUInt32(tsf, data)
#    elif TSF_SIZE_BITS == 64
#      define TsfWriteFixedArraySize(tsf, data) TsfWriteFixedArrayUInt64(tsf, data)
#    endif
#  endif

#  ifdef TSF_SUPPORT_FLOAT 
#    define TsfWriteVarArrayFloat(tsf, data) TsfWriteVarArrayFloat32(tsf, data)
#    define TsfWriteVarArrayDouble(tsf, data) TsfWriteVarArrayFloat64(tsf, data)
#  endif

#  ifdef TSF_SUPPORT_INT
#    define TsfWriteVarArrayChar(tsf, data) TsfWriteVarArrayInt8(tsf, data)
#    if TSF_SHRT_BITS == 8
#      define TsfWriteVarArrayShort(tsf, data) TsfWriteVarArrayInt8(tsf, data)
#    elif TSF_SHRT_BITS == 16
#      define TsfWriteVarArrayShort(tsf, data) TsfWriteVarArrayInt16(tsf, data)
#    elif TSF_SHRT_BITS == 32
#      define TsfWriteVarArrayShort(tsf, data) TsfWriteVarArrayInt32(tsf, data)
#    elif TSF_SHRT_BITS == 64
#      define TsfWriteVarArrayShort(tsf, data) TsfWriteVarArrayInt64(tsf, data)
#    endif

#    if TSF_INT_BITS == 8
#      define TsfWriteVarArrayInt(tsf, data) TsfWriteVarArrayInt8(tsf, data)
#    elif TSF_INT_BITS == 16
#      define TsfWriteVarArrayInt(tsf, data) TsfWriteVarArrayInt16(tsf, data)
#    elif TSF_INT_BITS == 32
#      define TsfWriteVarArrayInt(tsf, data) TsfWriteVarArrayInt32(tsf, data)
#    elif TSF_INT_BITS == 64
#      define TsfWriteVarArrayInt(tsf, data) TsfWriteVarArrayInt64(tsf, data)
#    endif

#    if TSF_LONG_BITS == 8
#      define TsfWriteVarArrayLong(tsf, data) TsfWriteVarArrayInt8(tsf, data)
#    elif TSF_LONG_BITS == 16
#      define TsfWriteVarArrayLong(tsf, data) TsfWriteVarArrayInt16(tsf, data)
#    elif TSF_LONG_BITS == 32
#      define TsfWriteVarArrayLong(tsf, data) TsfWriteVarArrayInt32(tsf, data)
#    elif TSF_LONG_BITS == 64
#      define TsfWriteVarArrayLong(tsf, data) TsfWriteVarArrayInt64(tsf, data)
#    endif

#    if TSF_LLONG_BITS == 8
#      define TsfWriteVarArrayLLong(tsf, data) TsfWriteVarArrayInt8(tsf, data)
#    elif TSF_LLONG_BITS == 16
#      define TsfWriteVarArrayLLong(tsf, data) TsfWriteVarArrayInt16(tsf, data)
#    elif TSF_LLONG_BITS == 32
#      define TsfWriteVarArrayLLong(tsf, data) TsfWriteVarArrayInt32(tsf, data)
#    elif TSF_LLONG_BITS == 64
#      define TsfWriteVarArrayLLong(tsf, data) TsfWriteVarArrayInt64(tsf, data)
#    endif
#  endif

#  ifdef TSF_SUPPORT_UINT
#    define TsfWriteVarArrayUChar(tsf, data) TsfWriteVarArrayUInt8(tsf, data)
#    if TSF_USHRT_BITS == 8
#      define TsfWriteVarArrayUShort(tsf, data) TsfWriteVarArrayUInt8(tsf, data)
#    elif TSF_USHRT_BITS == 16
#      define TsfWriteVarArrayUShort(tsf, data) TsfWriteVarArrayUInt16(tsf, data)
#    elif TSF_USHRT_BITS == 32
#      define TsfWriteVarArrayUShort(tsf, data) TsfWriteVarArrayUInt32(tsf, data)
#    elif TSF_USHRT_BITS == 64
#      define TsfWriteVarArrayUShort(tsf, data) TsfWriteVarArrayUInt64(tsf, data)
#    endif

#    if TSF_UINT_BITS == 8
#      define TsfWriteVarArrayUInt(tsf, data) TsfWriteVarArrayUInt8(tsf, data)
#    elif TSF_UINT_BITS == 16
#      define TsfWriteVarArrayUInt(tsf, data) TsfWriteVarArrayUInt16(tsf, data)
#    elif TSF_UINT_BITS == 32
#      define TsfWriteVarArrayUInt(tsf, data) TsfWriteVarArrayUInt32(tsf, data)
#    elif TSF_UINT_BITS == 64
#      define TsfWriteVarArrayUInt(tsf, data) TsfWriteVarArrayUInt64(tsf, data)
#    endif

#    if TSF_ULONG_BITS == 8
#      define TsfWriteVarArrayULong(tsf, data) TsfWriteVarArrayUInt8(tsf, data)
#    elif TSF_ULONG_BITS == 16
#      define TsfWriteVarArrayULong(tsf, data) TsfWriteVarArrayUInt16(tsf, data)
#    elif TSF_ULONG_BITS == 32
#      define TsfWriteVarArrayULong(tsf, data) TsfWriteVarArrayUInt32(tsf, data)
#    elif TSF_ULONG_BITS == 64
#      define TsfWriteVarArrayULong(tsf, data) TsfWriteVarArrayUInt64(tsf, data)
#    endif

#    if TSF_ULLONG_BITS == 8
#      define TsfWriteVarArrayULLong(tsf, data) TsfWriteVarArrayUInt8(tsf, data)
#    elif TSF_ULLONG_BITS == 16
#      define TsfWriteVarArrayULLong(tsf, data) TsfWriteVarArrayUInt16(tsf, data)
#    elif TSF_ULLONG_BITS == 32
#      define TsfWriteVarArrayULLong(tsf, data) TsfWriteVarArrayUInt32(tsf, data)
#    elif TSF_ULLONG_BITS == 64
#      define TsfWriteVarArrayULLong(tsf, data) TsfWriteVarArrayUInt64(tsf, data)
#    endif

#    if TSF_SIZE_BITS == 8
#      define TsfWriteVarArraySize(tsf, data) TsfWriteVarArrayUInt8(tsf, data)
#    elif TSF_SIZE_BITS == 16
#      define TsfWriteVarArraySize(tsf, data) TsfWriteVarArrayUInt16(tsf, data)
#    elif TSF_SIZE_BITS == 32
#      define TsfWriteVarArraySize(tsf, data) TsfWriteVarArrayUInt32(tsf, data)
#    elif TSF_SIZE_BITS == 64
#      define TsfWriteVarArraySize(tsf, data) TsfWriteVarArrayUInt64(tsf, data)
#    endif
#  endif
#endif

#endif

#endif
