﻿/*----------------------------------------------------------------------------*/
/* Firmware Obfuscation V1.0.4                                                */
/* Written in 2019 by Rudy Tellert Elektronik                                 */
/*                                                                            */
/* To the extent possible under law, the author has dedicated all copyright   */
/* and related and neighboring rights to the software "firmware obfuscation"  */
/* and its documentation to the public domain. This software and its          */
/* documentation are distributed without any warranty.                        */
/*                                                                            */
/* See also                                                                   */
/* http://creativecommons.org/publicdomain/zero/1.0/                          */
/* ---------------------------------------------------------------------------*/

using System;
using System.Diagnostics;
using System.IO;
using System.Runtime.InteropServices;

namespace Tellert
{
    public struct Utc
    {
        private uint ticks; // seconds since Thursday, 1970-01-01 (representable until 2106-02-06)

        public uint Ticks { get { return ticks; } }

        public static readonly Utc Zero = new Utc(0);
        public static readonly Utc MinValue = new Utc(1);
        public static readonly Utc MaxValue = new Utc(uint.MaxValue);

        public bool IsMinValue { get { return ticks == 1; } }
        public bool IsMaxValue { get { return ticks == uint.MaxValue; } }
        public bool IsZero { get { return ticks == 0; } }

        public Utc(uint t)
        {
            ticks = t;
        }

        public Utc(DateTime dt)
        {
            Debug.Assert(dt >= MinValue.ToDateTime() && dt <= MaxValue.ToDateTime());

            // round towards next integer second
            long l = (dt.ToUniversalTime().Ticks - 621355967995000000) / 10000000;
            if (l < 1 && l > uint.MaxValue) ticks = 0;
            else ticks = (uint)l;
        }

        public Utc(byte[] arr)
        {
            ticks = (arr.Length == 4) ? BitConverter.ToUInt32(arr, 0) : 0;
        }

        public static Utc Now { get { return new Utc(DateTime.UtcNow); } }

        public DateTime ToDateTime()
        {
            return new DateTime((long)ticks * 10000000 + 621355968000000000, DateTimeKind.Utc);
        }

        public byte[] ToBytes()
        {
            return BitConverter.GetBytes(ticks);
        }

        public override string ToString()
        {
            if (IsZero) return string.Empty;
            DateTime dt = ToDateTime();
            return string.Format("{0:D4}-{1:D2}-{2:D2}T{3:D2}:{4:D2}:{5:D2}Z",
                dt.Year, dt.Month, dt.Day, dt.Hour, dt.Minute, dt.Second);
        }

        public static Utc GetNonce(string fileName, uint timeout = 5000)
        {
            uint nonce = Utc.Now.Ticks;
            uint t0 = (uint)Environment.TickCount;

            if (File.Exists(fileName))
            {
                for (; ; )
                {
                    try
                    {
                        using (FileStream stream = new FileStream(fileName, FileMode.Open, FileAccess.ReadWrite, FileShare.None))
                        {
                            byte[] arr = new byte[4];
                            if (stream.Read(arr, 0, 4) == 4)
                            {
                                uint fileNonce = BitConverter.ToUInt32(arr, 0);
                                if (nonce <= fileNonce) nonce = fileNonce + 1;
                            }
                            stream.Seek(0, SeekOrigin.Begin);
                            stream.Write(BitConverter.GetBytes(nonce), 0, 4);
                        }
                        break;
                    }
                    catch (IOException e)
                    {
                        if (Marshal.GetHRForException(e) != -2147024864) break;
                        if ((uint)Environment.TickCount - t0 > timeout) break;
                    }
                    catch
                    {
                        break;
                    }
                }
            }
            else
            {
                for (; ; )
                {
                    try
                    {
                        using (FileStream stream = new FileStream(fileName, FileMode.Create, FileAccess.ReadWrite, FileShare.None))
                        {
                            stream.Write(BitConverter.GetBytes(nonce), 0, 4);
                        }
                        break;
                    }
                    catch (IOException e)
                    {
                        if (Marshal.GetHRForException(e) != -2147024864) break;
                        if ((uint)Environment.TickCount - t0 > timeout) break;
                    }
                    catch
                    {
                        break;
                    }
                }
            }

            return new Utc(nonce);
        }
    }
}

