﻿/*----------------------------------------------------------------------------*/
/* Firmware Obfuscation V1.0.4                                                */
/* Written in 2019 by Rudy Tellert Elektronik                                 */
/*                                                                            */
/* To the extent possible under law, the author has dedicated all copyright   */
/* and related and neighboring rights to the software "firmware obfuscation"  */
/* and its documentation to the public domain. This software and its          */
/* documentation are distributed without any warranty.                        */
/*                                                                            */
/* See also                                                                   */
/* http://creativecommons.org/publicdomain/zero/1.0/                          */
/* ---------------------------------------------------------------------------*/

#define IMPLEMENT_HASH // Comment out this line if your own hash code should be used

using System;

namespace Tellert.Firmware
{
    public class Hash : IDisposable
    {
        #region hash functions

        public static bool UseHashInsteadOfMac { get { return false; } }
        public static bool IsDoubleHashingRequired { get { return true; } }
        public static bool UseMacWithPermutation { get { return false; } }

#if IMPLEMENT_HASH
        public static int HashSize { get { return 128; } }

        UInt128 state = 1;
        int count = 0;
        Permutation128 permutation = new Permutation128();
        public byte[] Permutation0;

        public void Init(byte[] key)
        {
            if (!UseMacWithPermutation) throw new ArgumentException();
            permutation.Init(key);
            state = 1;
            count = 0;
            Permutation0 = permutation.ForwardTransform(new byte[Permutation128.BlockLength]);
        }

        public void Init()
        {
            if (UseMacWithPermutation) throw new ArgumentException();
            state = 1;
        }

        public void TransformBlock(byte[] msg, int offset, int len)
        {
            for (int i = 0; i < len; i++) state += (state << 8) + (msg[offset + i] + 1);
            count += len;
        }

        public byte[] TransformFinalBlock(byte[] msg, int offset, int len)
        {
            byte[] block;

            TransformBlock(msg, offset, len);
            block = BitConverterEx.GetBytes(state);
            if (UseMacWithPermutation)
            {
                if (!BitConverter.IsLittleEndian) Array.Reverse(block);
                block = permutation.ForwardTransform(block);
                byte[] cntBlock = BitConverter.GetBytes(count);
                if (!BitConverter.IsLittleEndian) Array.Reverse(cntBlock);
                for (int i = 0; i < cntBlock.Length; i++) block[i] ^= cntBlock[i];
                block = permutation.ForwardTransform(block);
            }
            else
            {
                if (BitConverter.IsLittleEndian) Array.Reverse(block);
                for (int i = block.Length - 2; i >= 0; i--) block[i] ^= block[i + 1];
            }

            return block;
        }

        protected virtual void Dispose(bool disposing)
        {
        }

#else

        public static int HashSize { get { return 128; } }

        public void Init(byte[] key)
        {
            if (!UseMacWithPermutation) throw new ArgumentException();
            
            /* [insert code here] */
        }

        public void Init()
        {
            if (UseMacWithPermutation) throw new ArgumentException();

            /* [insert code here] */
        }

        public void TransformBlock(byte[] msg, int offset, int len)
        {
            /* [insert code here] */
        }

        public byte[] TransformFinalBlock(byte[] msg, int offset, int len)
        {
            /* [insert code here] */

            return new byte[HashLength];
        }

        protected virtual void Dispose(bool disposing)
        {
        }

#endif

        #endregion

        public static int HashLength { get { return HashSize / 8; } }
        public static int TagSize { get { return Math.Min(HashSize / 2, 128); } }
        public static int TagLength { get { return TagSize / 8; } }
        public static int NonceSize { get { return 64; } }
        public static int NonceLength { get { return NonceSize / 8; } }

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        ~Hash()
        {
            Dispose(false);
        }
    }

    public class Permutation128 : IDisposable
    {
#region 128-bit block permutation functions

        public static int DefaultKeySize { get { return 56; } }
        public static bool UsePermutation128 {  get { return true; } }
        public static bool UsePermutation128WithHash { get { return true; } }

        public static int BlockSize { get { return UsePermutation128WithHash ? Hash.HashSize: 128; } }

        byte[] key;

        public void Init(byte[] key)
        {
            if (key.Length > 7) throw new ArgumentException();
            
            /* [insert code here] */

            this.key = key;
        }

        public byte[] ForwardTransform(byte[] block)
        {
            if (UsePermutation128WithHash)
            {
                return Hash.UseHashInsteadOfMac ? Obfuscation.KeyedHash(key, block) : Obfuscation.Mac(key, block, null, 0, BlockSize);
            }
            else
            {
                /* [insert code here] */

                return block;
            }
        }

        protected virtual void Dispose(bool disposing)
        {
        }

#endregion

        public static int DefaultKeyLength { get { return DefaultKeySize / 8; } }
        public static int BlockLength { get { return BlockSize / 8; } }

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        ~Permutation128()
        {
            Dispose(false);
        }
    }

    public class Permutation64 : IDisposable
    {
#region 64-bit block permutation functions

        public static int DefaultKeySize { get { return 56; } }
        public static int BlockSize { get { return 64; } }

        byte[] key;

        public void Init(byte[] key)
        {
            if (key.Length > 7) throw new ArgumentException();
            this.key = key;
        }

        public byte[] ForwardTransform(byte[] block)
        {
            return block;
        }

        protected virtual void Dispose(bool disposing)
        {
        }

#endregion

        public static int BlockLength { get { return BlockSize / 8; } }

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        ~Permutation64()
        {
            Dispose(false);
        }
    }
}