﻿/*----------------------------------------------------------------------------*/
/* Firmware Obfuscation V1.0.4                                                */
/* Written in 2019 by Rudy Tellert Elektronik                                 */
/*                                                                            */
/* To the extent possible under law, the author has dedicated all copyright   */
/* and related and neighboring rights to the software "firmware obfuscation"  */
/* and its documentation to the public domain. This software and its          */
/* documentation are distributed without any warranty.                        */
/*                                                                            */
/* See also                                                                   */
/* http://creativecommons.org/publicdomain/zero/1.0/                          */
/* ---------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/* Settings                                                                   */
/*----------------------------------------------------------------------------*/

#define WRITE_OBFUSCATION_KEY

/*----------------------------------------------------------------------------*/
/* Program                                                                    */
/*----------------------------------------------------------------------------*/

using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.Serialization;
using Tellert;
using Tellert.Firmware;
using Tellert.Format;

namespace makeboot
{
    public static class Const
    {
        public const int TsfBlockSize = 1024; // max. TSF size for an obfuscated block (incl. nonce and DRM)
    }

    public static class StringHelper
    {
        public static byte[] ToBytes(this string str)
        {
            if (str == null) str = "";
            return System.Text.Encoding.Default.GetBytes(str);
        }

        public static void CopyTo(this string str, byte[] dest, int offset, int maxSize)
        {
            byte[] b = str.ToBytes();
            int size = b.Length;
            if (size > maxSize) size = maxSize;
            Array.Copy(b, 0, dest, offset, size);
        }

        public static void CopyTo(this uint u, byte[] dest, int offset, int size = 4)
        {
            byte[] b = BitConverter.GetBytes(u);
            Array.Copy(b, 0, dest, offset, size);
        }

        static byte HexNibble(char ch)
        {
            if (ch >= '0' && ch <= '9') return (byte)(ch - '0');
            if (ch >= 'A' && ch <= 'F') return (byte)(ch - 'A' + 10);
            if (ch >= 'a' && ch <= 'f') return (byte)(ch - 'a' + 10);
            return 0;
        }

        public static byte[] HexToBytes(this string str)
        {
            if (str == null) return null;
            
            int hexDigits = 0;
            for (int i = 0; i < str.Length; i++) if (str[i] > ' ') hexDigits++;

            int k = 0;
            byte[] b = new byte[hexDigits / 2];
            for (int i = 0; i < b.Length; i++) {
                while (str[k] <= ' ') k++;
                byte hi = HexNibble(str[k++]);
                while (str[k] <= ' ') k++;
                byte lo = HexNibble(str[k++]);
                b[i] = (byte)((hi << 4) + lo);
            }

            return b;
        }

        public static void Clear(this byte[] dest)
        {
            Array.Clear(dest, 0, dest.Length);
        }

        public static uint ToLittleEndian(this uint val)
        {
            if (!BitConverter.IsLittleEndian)
            {
                byte[] b = BitConverter.GetBytes(val);
                Array.Reverse(b);
                val = BitConverter.ToUInt32(b, 0);
            }
            return val;
        }
    }

    public struct Drm
    {
        public uint type;
        public uint min;
        public uint max;
        public uint xorMask;
        public uint reqFeatures;
    }

    public struct Area
    {
        public uint address;
        public byte[] data;
    }

    [DataContract(Namespace="http://www.tellert.de/2015/bootldr")]
    public class Config
    {
        [DataMember] public string strDesc, strName, strVersion;
        [DataMember] public uint id, features, versionMajor, versionMinor, versionRevision, build, config;
        [DataMember] public uint eraseBegin, eraseSize, programBegin, programSize, eraseDataBegin, eraseDataSize, programDataBegin, programDataSize;
        [DataMember] public uint nmiIsrAddress, dataInfoAddress, configInfoAddress;

        [DataMember] public uint romBegin, appInfoAddress, deviceIdCodeAddress;

        [DataMember] public uint manufacturerIdMin, manufacturerIdMax, hardwareIdMin, hardwareIdMax, hardwareVersionMin, hardwareVersionMax;
        [DataMember] public uint internalDeviceNumberMin, internalDeviceNumberMax, deviceNumberMin, deviceNumberMax, customerIdMin, customerIdMax;
        [DataMember] public uint dateMin, dateMax, hardwareFeaturesXorMask, hardwareFeaturesReq;

        [DataMember] public uint eraseRegions;

        [DataMember] public string orgBootLoaderHexFile, deviceBootLoaderHexFile, eraseTsfFile;

        [DataMember] public uint manufacturerId, hardwareId, hardwareFeatures, hardwareVersion, internalDeviceNumber, deviceNumber, customer;

        [DataMember] public string utcFile;
    
        [DataMember] public string strKey;

        [DataMember] public uint hwInfoAddress, appEntryAddress;
        [DataMember] public int maxTsfSize;

        [DataMember] public bool ownDrmBlock, tinyBootloader;
    }

    public class Key
    {
        public byte[] o, // obfuscation key
            a, // authentication key 
            n, // nonce key
            id, // flash ROM id code
            r1, // reserve device dependent key #1
            r2, // reserve device dependent key #2
            k1, // reserve device independent key #1
            k2; // reserve device independent key #2
        public uint token; // publicly visible device dependent 32-bit value

        public Key(byte[] k, int deviceNumber=0)
        {
            o = GetKey(k, 0, 0, 7);
            a = GetKey(k, 1, 0, 16);
            n = GetKey(k, 2, 0, 7);
            byte[] h = GetKey(k, 3, 0, 16); // temporary key for generating t without risking k
            byte[] t = GetKey(h, 3, deviceNumber, 4);
            if (!BitConverter.IsLittleEndian) Array.Reverse(t);
            token = BitConverter.ToUInt32(t, 0);
            id = GetKey(k, 4, deviceNumber, 16);
            h = GetKey(k, 5, 0, 16);
            r1 = GetKey(h, 5, deviceNumber, 16);
            h = GetKey(k, 6, 0, 16);
            r2 = GetKey(h, 6, deviceNumber, 16);
            k1 = GetKey(k, 7, 0, 16);
            k2 = GetKey(k, 8, 0, 16);
        }

        static byte[] GetKey(byte[] k, int type, int deviceNumber, int size)
        {
            byte[] result = new byte[size];

            return Obfuscation.Mac(k, null, null, type, size * 8, deviceNumber);
        }
    }
    
    class Program
    {
        public static Config cfg;

        static bool GetDrm(ref Drm drm, uint type, uint min, uint max)
        {
            if (min == 0 && max == 0) return false;
            drm.type = type; drm.min = min; drm.max = max;
            drm.xorMask = drm.reqFeatures = 0;
            return true;
        }

        static void CreateBootloader(string outName)
        {
            HexFile hexFile = new HexFile();
            hexFile.Read(cfg.orgBootLoaderHexFile);
            Utc utcNow = Utc.GetNonce(cfg.utcFile);

            byte[] km = cfg.strKey.HexToBytes();
            Key key = new Key(km, (int)cfg.deviceNumber);

            string info = string.Format("M{0:x8} H{1:x8} F{2:x8} V{3:x8} D{4:x8} I{5:x8} N{6:x8} C{7:x8} T{8:x8}", 
              cfg.manufacturerId, cfg.hardwareId, cfg.hardwareFeatures, cfg.hardwareVersion,
              utcNow.Ticks, cfg.internalDeviceNumber, cfg.deviceNumber, cfg.customer, key.token);

            byte[] hwInfo = new byte[256];
            cfg.appInfoAddress.CopyTo(hwInfo, 0, 4);
            info.CopyTo(hwInfo, 4, 91);
            cfg.manufacturerId.CopyTo(hwInfo, 96, 4);
            cfg.hardwareId.CopyTo(hwInfo, 100, 4);
            cfg.hardwareFeatures.CopyTo(hwInfo, 104, 4);
            cfg.hardwareVersion.CopyTo(hwInfo, 108, 4);
            utcNow.Ticks.CopyTo(hwInfo, 112, 4);
            cfg.internalDeviceNumber.CopyTo(hwInfo, 116, 4);
            cfg.deviceNumber.CopyTo(hwInfo, 120, 4);
            cfg.customer.CopyTo(hwInfo, 124, 4);
            key.token.CopyTo(hwInfo, 128, 4);
            key.o.CopyTo(hwInfo, 148);
            key.a.CopyTo(hwInfo, 164);
            key.k1.CopyTo(hwInfo, 180);
            key.k2.CopyTo(hwInfo, 196);
            key.r1.CopyTo(hwInfo, 212);
            key.r2.CopyTo(hwInfo, 228);

            hexFile.SetBytes(cfg.hwInfoAddress, hwInfo);

            hexFile.SetBytes(cfg.deviceIdCodeAddress, key.id);

            hexFile.Write(outName);
        }

        static void CreateErase(string outName)
        {
            Utc utc = new Utc(File.GetLastWriteTime(cfg.deviceBootLoaderHexFile));
            Utc utcNow = Utc.GetNonce(cfg.utcFile);
            byte[] km = cfg.strKey.HexToBytes();
            Key key = new Key(km, (int)cfg.deviceNumber);
            uint checksum;

            Area[] areaArr = new Area[1];
            areaArr[0].address = cfg.appInfoAddress;
            areaArr[0].data = "ERASE ALL".ToBytes();

            List<Drm> drmList = new List<Drm>();
            Drm drm = new Drm();
            if (GetDrm(ref drm, 0, cfg.manufacturerIdMin, cfg.manufacturerIdMax)) drmList.Add(drm);
            if (GetDrm(ref drm, 1, cfg.hardwareIdMin, cfg.hardwareIdMax)) drmList.Add(drm);
            if (GetDrm(ref drm, 2, cfg.hardwareVersionMin, cfg.hardwareVersionMax)) drmList.Add(drm);
            if (GetDrm(ref drm, 3, cfg.internalDeviceNumberMin, cfg.internalDeviceNumberMax)) drmList.Add(drm);
            if (GetDrm(ref drm, 4, cfg.deviceNumberMin, cfg.deviceNumberMax)) drmList.Add(drm);
            if (GetDrm(ref drm, 5, cfg.customerIdMin, cfg.customerIdMax)) drmList.Add(drm);
            if (GetDrm(ref drm, 6, cfg.dateMin, cfg.dateMax)) drmList.Add(drm);
            if (cfg.hardwareFeaturesReq != 0)
            {
                drm.type = 64;
                drm.min = drm.max = 0;
                drm.xorMask = cfg.hardwareFeaturesXorMask;
                drm.reqFeatures = cfg.hardwareFeaturesReq;
                drmList.Add(drm);
            }
            Drm[] drmArr = drmList.ToArray();

            // Create Tsf fragments
            byte[][] blocks = new byte[1][];
            Tsf tsf = new Tsf();
            if (!cfg.tinyBootloader)
            {
                // Initial fragment
                tsf.Create(Const.TsfBlockSize, Tsf.HeaderType.None);
                if (!BitConverter.IsLittleEndian) tsf.Reverse = true;
                if (drmArr.Length != 0)
                {
                    if (tsf.WriteArray(5, drmArr.Length))
                    {
                        for (int i = 0; i < drmArr.Length; i++)
                        {
                            if (drmArr[i].type != 0) tsf.Write(3, drmArr[i].type);
                            if (drmArr[i].min == drmArr[i].max)
                            {
                                if (drmArr[i].min != 0) tsf.Write(9, drmArr[i].min);
                            }
                            else
                            {
                                if (drmArr[i].min != 0) tsf.Write(5, drmArr[i].min);
                                if (drmArr[i].max != 0) tsf.Write(7, drmArr[i].max);
                            }
                            if (drmArr[i].xorMask != 0) tsf.Write(11, drmArr[i].xorMask);
                            if (drmArr[i].reqFeatures != 0) tsf.Write(13, drmArr[i].reqFeatures);
                            tsf.WriteEnd();
                        }
                    }
                }
                tsf.Write(7, 1); // cfg.eraseRegions
                if (tsf.WriteArray(15, 1))
                {
                    tsf.Write(17, areaArr[0].address);
                    tsf.Write0(19, areaArr[0].data, 0, areaArr[0].data.Length);
                    tsf.WriteEnd();
                }
                blocks[0] = tsf.ToBytes();
            }
            else
            {
                byte[] block0 = new byte[(drmArr.Length + 1) * 12];
                Array.Copy(BitConverter.GetBytes(0xf8139cbd.ToLittleEndian()), block0, 4);
                Array.Copy(BitConverter.GetBytes(((uint)1).ToLittleEndian()), 0, block0, 4, 4);
                Array.Copy(BitConverter.GetBytes(((uint)drmArr.Length).ToLittleEndian()), 0, block0, 8, 4);
                for (int i = 0; i < drmArr.Length; i++)
                {
                    if ((drmArr[i].type & 0x40) == 0)
                    {
                        Array.Copy(BitConverter.GetBytes(drmArr[i].type.ToLittleEndian()), 0, block0, 12 + i * 12, 4);
                        Array.Copy(BitConverter.GetBytes(drmArr[i].min.ToLittleEndian()), 0, block0, 16 + i * 12, 4);
                        Array.Copy(BitConverter.GetBytes(drmArr[i].max.ToLittleEndian()), 0, block0, 20 + i * 12, 4);
                    }
                    else
                    {
                        Array.Copy(BitConverter.GetBytes(drmArr[i].type.ToLittleEndian()), 0, block0, 12 + i * 12, 4);
                        Array.Copy(BitConverter.GetBytes(drmArr[i].xorMask.ToLittleEndian()), 0, block0, 16 + i * 12, 4);
                        Array.Copy(BitConverter.GetBytes(drmArr[i].reqFeatures.ToLittleEndian()), 0, block0, 20 + i * 12, 4);
                    }
                }
                blocks[0] = block0;
            }

            // Obfuscate blocks
            byte[] nonce = Obfuscation.CreateObfuscatedNonce(key.n, 0);
            byte[] tag;
            for (int i = 0; i < blocks.Length; i++)
            {
                blocks[i] = Obfuscation.Ctr(key.o, nonce, blocks[i], i);
                tag = Obfuscation.Mac(key.a, nonce, blocks[i], i, Hash.TagSize);
                byte[] tagBlock = new byte[blocks[i].Length + tag.Length];
                Array.Copy(tag, tagBlock, tag.Length);
                Array.Copy(blocks[i], 0, tagBlock, tag.Length, blocks[i].Length);
                blocks[i] = tagBlock;
            }

            // blocks[0] = nonce || blocks[0]
            byte[] tmp = new byte[nonce.Length + blocks[0].Length];
            Array.Copy(nonce, tmp, nonce.Length);
            Array.Copy(blocks[0], 0, tmp, nonce.Length, blocks[0].Length);
            blocks[0] = tmp;

            // Create 2nd DRM Tsf fragment
            tsf.Create(Const.TsfBlockSize, Tsf.HeaderType.None);
            if (!BitConverter.IsLittleEndian) tsf.Reverse = true;
            if (drmArr.Length != 0)
            {
                if (tsf.WriteArray(5, drmArr.Length))
                {
                    for (int i = 0; i < drmArr.Length; i++)
                    {
                        if (drmArr[i].type != 0) tsf.Write(3, drmArr[i].type);
                        if (drmArr[i].min == drmArr[i].max)
                        {
                            if (drmArr[i].min != 0) tsf.Write(9, drmArr[i].min);
                        }
                        else
                        {
                            if (drmArr[i].min != 0) tsf.Write(5, drmArr[i].min);
                            if (drmArr[i].max != 0) tsf.Write(7, drmArr[i].max);
                        }
                        if (drmArr[i].xorMask != 0) tsf.Write(11, drmArr[i].xorMask);
                        if (drmArr[i].reqFeatures != 0) tsf.Write(13, drmArr[i].reqFeatures);
                        tsf.WriteEnd();
                    }
                }
            }
            byte[] drmData = tsf.ToBytes();
            uint salt = utcNow.Ticks;
            Obfuscation.Add(drmData, salt);

            // Create output file
            tsf.Create(cfg.maxTsfSize, Tsf.HeaderType.Normal);
            if (tsf.WriteArray(1, 1))
            {
                tsf.Write(2, 134u);
                tsf.WriteEnd();
            }
            if (tsf.WriteArray(27, 1))
            {
                if (drmData.Length != 0)
                {
                    tsf.Write(3, salt);
                    tsf.Write0(5, drmData, 0, drmData.Length);
                }
#if WRITE_OBFUSCATION_KEY
                if (tsf.WriteArray(6, 1))
                {
                    tsf.Write0(3, key.o, 0, key.o.Length);
                    tsf.WriteEnd();
                }
#endif
                if (tsf.WriteArray(7, blocks.Length))
                {
                    for (int i = 0; i < blocks.Length; i++)
                    {
                        tsf.Write0(3, blocks[i], 0, blocks[i].Length);
                        tsf.WriteEnd();
                    }
                }
                tsf.WriteEnd();
            }
            checksum = Checksum.Compute(tsf.ToBytes(), 0, tsf.ToBytes().Length);
            if (checksum != 0) tsf.Write(30, checksum);
            tsf.WriteEnd();
            tsf.WriteFile(outName);
        }

        static void ConvertApplication(string motFileName, string outName)
        {
            HexFile hexFile = new HexFile();
            hexFile.Read(motFileName);
            Utc utc = new Utc(File.GetLastWriteTime(motFileName));
            Utc utcNow = Utc.GetNonce(cfg.utcFile);

            uint appEntry = BitConverter.ToUInt32(hexFile.GetBytes(cfg.appEntryAddress, 4), 0);
            uint version = (cfg.versionMajor << 24) + (cfg.versionMinor << 16) + cfg.versionRevision;
            string info = string.Format("A{0:x4} F{1:x4} V{2:x8} B{3:x4} C{4:x4} D{5:x8}",
                cfg.id, cfg.features, version, cfg.build, cfg.config, utc.Ticks);
            byte[] km = cfg.strKey.HexToBytes();
            Key key = new Key(km, (int)cfg.deviceNumber);

            byte[] appInfo = new byte[256];
            appInfo[0] = 0x7a;
            appInfo[1] = 0xe0;
            cfg.strDesc.CopyTo(appInfo, 2, 63);
            info.CopyTo(appInfo, 66, 43);
            cfg.strName.CopyTo(appInfo, 110, 20);
            cfg.strVersion.CopyTo(appInfo, 131, 20);
            cfg.id.CopyTo(appInfo, 152, 2);
            cfg.features.CopyTo(appInfo, 154, 2);
            version.CopyTo(appInfo, 156, 4);
            cfg.build.CopyTo(appInfo, 160, 2);
            cfg.config.CopyTo(appInfo, 162, 2);
            utc.Ticks.CopyTo(appInfo, 164, 4);
            cfg.eraseBegin.CopyTo(appInfo, 168);
            cfg.eraseSize.CopyTo(appInfo, 172);
            cfg.programBegin.CopyTo(appInfo, 176);
            cfg.programSize.CopyTo(appInfo, 180);
            cfg.eraseDataBegin.CopyTo(appInfo, 184);
            cfg.eraseDataSize.CopyTo(appInfo, 188);
            cfg.programDataBegin.CopyTo(appInfo, 192);
            cfg.programDataSize.CopyTo(appInfo, 196);
            appEntry.CopyTo(appInfo, 236);
            cfg.nmiIsrAddress.CopyTo(appInfo, 240);
            cfg.dataInfoAddress.CopyTo(appInfo, 244);
            cfg.configInfoAddress.CopyTo(appInfo, 248);
            uint checksum = Checksum.Compute(appInfo, 0, 252);
            checksum.CopyTo(appInfo, 252);
            hexFile.SetBytes(cfg.appInfoAddress, appInfo);

            List<Area> areaList = new List<Area>();
            int size = (int)((((cfg.appInfoAddress + 256 - cfg.romBegin) + 255) / 256) * 256);
            byte[] arr = hexFile.GetBytes(cfg.romBegin, size);
            for (int i = 0; i < arr.Length; i += 256)
            {
                if (!hexFile.IsDataAvailable((uint)(cfg.romBegin + i), 256)) continue;

                Area area = new Area();
                area.address = (uint)(cfg.romBegin + i);
                area.data = new byte[256];
                Array.Copy(arr, i, area.data, 0, 256);
                areaList.Add(area);
            }
            Area[] areaArr = areaList.ToArray();

            List<Drm> drmList = new List<Drm>();
            Drm drm = new Drm();
            if (GetDrm(ref drm, 0, cfg.manufacturerIdMin, cfg.manufacturerIdMax)) drmList.Add(drm);
            if (GetDrm(ref drm, 1, cfg.hardwareIdMin, cfg.hardwareIdMax)) drmList.Add(drm);
            if (GetDrm(ref drm, 2, cfg.hardwareVersionMin, cfg.hardwareVersionMax)) drmList.Add(drm);
            if (GetDrm(ref drm, 3, cfg.internalDeviceNumberMin, cfg.internalDeviceNumberMax)) drmList.Add(drm);
            if (GetDrm(ref drm, 4, cfg.deviceNumberMin, cfg.deviceNumberMax)) drmList.Add(drm);
            if (GetDrm(ref drm, 5, cfg.customerIdMin, cfg.customerIdMax)) drmList.Add(drm);
            if (GetDrm(ref drm, 6, cfg.dateMin, cfg.dateMax)) drmList.Add(drm);
            if (cfg.hardwareFeaturesReq != 0)
            {
                drm.type = 64; 
                drm.min = drm.max = 0;
                drm.xorMask = cfg.hardwareFeaturesXorMask;
                drm.reqFeatures = cfg.hardwareFeaturesReq;
                drmList.Add(drm);
            }
            Drm[] drmArr = drmList.ToArray();

            Tsf tsf = new Tsf();
            byte[][] blocks;
            // Create Tsf fragments
            if (!cfg.tinyBootloader)
            {
                if (!cfg.ownDrmBlock)
                {
                    blocks = new byte[areaArr.Length][];
                }
                else {
                    blocks = new byte[areaArr.Length + 1][];
                }
                // Initial fragment
                tsf.Create(Const.TsfBlockSize, Tsf.HeaderType.None);
                if (!BitConverter.IsLittleEndian) tsf.Reverse = true;
                if (drmArr.Length != 0)
                {
                    if (tsf.WriteArray(5, drmArr.Length))
                    {
                        for (int i = 0; i < drmArr.Length; i++)
                        {
                            if (drmArr[i].type != 0) tsf.Write(3, drmArr[i].type);
                            if (drmArr[i].min == drmArr[i].max)
                            {
                                if (drmArr[i].min != 0) tsf.Write(9, drmArr[i].min);
                            }
                            else
                            {
                                if (drmArr[i].min != 0) tsf.Write(5, drmArr[i].min);
                                if (drmArr[i].max != 0) tsf.Write(7, drmArr[i].max);
                            }
                            if (drmArr[i].xorMask != 0) tsf.Write(11, drmArr[i].xorMask);
                            if (drmArr[i].reqFeatures != 0) tsf.Write(13, drmArr[i].reqFeatures);
                            tsf.WriteEnd();
                        }
                    }
                }
                if (cfg.eraseRegions != 0) tsf.Write(7, cfg.eraseRegions);
                if (!cfg.ownDrmBlock)
                {
                    if (tsf.WriteArray(15, 1))
                    {
                        tsf.Write(17, areaArr[0].address);
                        tsf.Write0(19, areaArr[0].data, 0, 256);
                        tsf.WriteEnd();
                    }
                }
                blocks[0] = tsf.ToBytes();

                for (int i = cfg.ownDrmBlock ? 0 : 1; i < areaArr.Length; i++)
                {
                    tsf.Create(Const.TsfBlockSize, Tsf.HeaderType.None);
                    if (!BitConverter.IsLittleEndian) tsf.Reverse = true;
                    if (tsf.WriteArray(15, 1))
                    {
                        tsf.Write(17, areaArr[i].address);
                        tsf.Write0(19, areaArr[i].data, 0, 256);
                        tsf.WriteEnd();
                    }
                    blocks[cfg.ownDrmBlock ? i + 1 : i] = tsf.ToBytes();
                }
            }
            else
            {
                blocks = new byte[areaArr.Length + 1][];
                byte[] block0 = new byte[(drmArr.Length + 1) * 12];
                Array.Copy(BitConverter.GetBytes(0xf8139cbd.ToLittleEndian()), block0, 4);
                Array.Copy(BitConverter.GetBytes(cfg.eraseRegions.ToLittleEndian()), 0, block0, 4, 4);
                Array.Copy(BitConverter.GetBytes(((uint)drmArr.Length).ToLittleEndian()), 0, block0, 8, 4);
                for (int i = 0; i < drmArr.Length; i++)
                {
                    if ((drmArr[i].type & 0x40) == 0)
                    {
                        Array.Copy(BitConverter.GetBytes(drmArr[i].type.ToLittleEndian()), 0, block0, 12 + i * 12, 4);
                        Array.Copy(BitConverter.GetBytes(drmArr[i].min.ToLittleEndian()), 0, block0, 16 + i * 12, 4);
                        Array.Copy(BitConverter.GetBytes(drmArr[i].max.ToLittleEndian()), 0, block0, 20 + i * 12, 4);
                    }
                    else
                    {
                        Array.Copy(BitConverter.GetBytes(drmArr[i].type.ToLittleEndian()), 0, block0, 12 + i * 12, 4);
                        Array.Copy(BitConverter.GetBytes(drmArr[i].xorMask.ToLittleEndian()), 0, block0, 16 + i * 12, 4);
                        Array.Copy(BitConverter.GetBytes(drmArr[i].reqFeatures.ToLittleEndian()), 0, block0, 20 + i * 12, 4);
                    }
                }
                blocks[0] = block0;

                for (int i = 0; i < areaArr.Length; i++)
                {
                    byte[] b = new byte[4 + 4 + 256];
                    Array.Copy(BitConverter.GetBytes(areaArr[i].address.ToLittleEndian()), b, 4);
                    Array.Copy(BitConverter.GetBytes(0x100U.ToLittleEndian()), 0, b, 4, 4);
                    Array.Copy(areaArr[i].data, 0, b, 8, 256);
                    blocks[i + 1] = b;
                }
            }

            byte[] nonce = Obfuscation.CreateObfuscatedNonce(key.n, 0);
            byte[] tag;
            for (int i = 0; i < blocks.Length; i++)
            {
                blocks[i] = Obfuscation.Ctr(key.o, nonce, blocks[i], i);
                tag = Obfuscation.Mac(key.a, nonce, blocks[i], i, Hash.TagSize);
                byte[] tagBlock = new byte[blocks[i].Length + tag.Length];
                Array.Copy(tag, tagBlock, tag.Length);
                Array.Copy(blocks[i], 0, tagBlock, tag.Length, blocks[i].Length);
                blocks[i] = tagBlock;
            }

            // blocks[0] = nonce || blocks[0]
            byte[] tmp = new byte[nonce.Length + blocks[0].Length];
            Array.Copy(nonce, tmp, nonce.Length);
            Array.Copy(blocks[0], 0, tmp, nonce.Length, blocks[0].Length);
            blocks[0] = tmp;

            // Create 2nd DRM Tsf fragment
            tsf.Create(Const.TsfBlockSize, Tsf.HeaderType.None);
            if (!BitConverter.IsLittleEndian) tsf.Reverse = true;
            if (tsf.WriteArray(3, 1))
            {
                if (cfg.id != 0) tsf.Write(3, (ushort)cfg.id);
                if (cfg.features != 0) tsf.Write(5, (ushort)cfg.features);
                if (cfg.versionMajor != 0) tsf.Write(7, (byte)cfg.versionMajor);
                if (cfg.versionMinor != 0) tsf.Write(9, (byte)cfg.versionMinor);
                if (cfg.versionRevision != 0) tsf.Write(11, (ushort)cfg.versionRevision);
                if (cfg.build != 0) tsf.Write(13, (ushort)cfg.build);
                if (cfg.config != 0) tsf.Write(15, (ushort)cfg.config);
                if (utc.Ticks != 0) tsf.Write(17, utc.Ticks);
                if (cfg.strDesc.Length != 0) tsf.Write(25, cfg.strDesc.ToBytes(), 0, cfg.strDesc.Length);
                if (cfg.strName.Length != 0) tsf.Write(27, cfg.strName.ToBytes(), 0, cfg.strName.Length);
                if (cfg.strVersion.Length != 0) tsf.Write(29, cfg.strVersion.ToBytes(), 0, cfg.strVersion.Length);
                tsf.WriteEnd();
            }
            if (drmArr.Length != 0)
            {
                if (tsf.WriteArray(5, drmArr.Length))
                {
                    for (int i = 0; i < drmArr.Length; i++)
                    {
                        if (drmArr[i].type != 0) tsf.Write(3, drmArr[i].type);
                        if (drmArr[i].min == drmArr[i].max)
                        {
                            if (drmArr[i].min != 0) tsf.Write(9, drmArr[i].min);
                        }
                        else
                        {
                            if (drmArr[i].min != 0) tsf.Write(5, drmArr[i].min);
                            if (drmArr[i].max != 0) tsf.Write(7, drmArr[i].max);
                        }
                        if (drmArr[i].xorMask != 0) tsf.Write(11, drmArr[i].xorMask);
                        if (drmArr[i].reqFeatures != 0) tsf.Write(13, drmArr[i].reqFeatures);
                        tsf.WriteEnd();
                    }
                }
            }
            byte[] drmData = tsf.ToBytes();
            uint salt = utcNow.Ticks;
            Obfuscation.Add(drmData, salt);

            // Create output file
            tsf.Create(cfg.maxTsfSize, Tsf.HeaderType.Normal);
            if (tsf.WriteArray(1, 1)) {
                tsf.Write(2, 134u);
                tsf.WriteEnd();
            }
            if (tsf.WriteArray(27, 1))
            {
                if (drmData.Length != 0)
                {
                    tsf.Write(3, salt);
                    tsf.Write0(5, drmData, 0, drmData.Length);
                }
#if WRITE_OBFUSCATION_KEY
                if (tsf.WriteArray(6, 1))
                {
                    tsf.Write0(3, key.o, 0, key.o.Length);
                    tsf.WriteEnd();
                }
#endif
                if (tsf.WriteArray(7, blocks.Length))
                {
                    for (int i = 0; i < blocks.Length; i++)
                    {
                        tsf.Write0(3, blocks[i], 0, blocks[i].Length);
                        tsf.WriteEnd();
                    }
                }
                tsf.WriteEnd();
            }
            checksum = Checksum.Compute(tsf.ToBytes(), 0, tsf.ToBytes().Length);
            if (checksum != 0) tsf.Write(30, checksum);
            tsf.WriteEnd();
            tsf.WriteFile(outName);
        }

        static void Main(string[] args)
        {
            string uri = System.Reflection.Assembly.GetExecutingAssembly().CodeBase;
            string xmlFileName = System.IO.Path.GetFileNameWithoutExtension(new Uri(uri).LocalPath) + ".xml";

            var ds = new DataContractSerializer(typeof(Config));
            using (Stream s = File.OpenRead(xmlFileName)) cfg = (Config)ds.ReadObject(s);

            if (args.Length == 2)
            {
                ConvertApplication(args[0], args[1]);
            }
            else if (args.Length == 1)
            {
                cfg.deviceNumber = uint.Parse(args[0]);
                CreateBootloader(cfg.deviceBootLoaderHexFile);
            }
            else if (args.Length == 0)
            {
                CreateErase(cfg.eraseTsfFile);
            }
        }
    }
}
