/*----------------------------------------------------------------------------*/
/* Firmware Obfuscation V1.0.4                                                */
/* Written in 2019 by Rudy Tellert Elektronik                                 */
/*                                                                            */
/* To the extent possible under law, the author has dedicated all copyright   */
/* and related and neighboring rights to the software "firmware obfuscation"  */
/* and its documentation to the public domain. This software and its          */
/* documentation are distributed without any warranty.                        */
/*                                                                            */
/* See also                                                                   */
/* http://creativecommons.org/publicdomain/zero/1.0/                          */
/* ---------------------------------------------------------------------------*/

#include "bootldr.h"

#ifdef REUSE_RX_BUFFER
Byte rxBuffer[BASE64_SIZE+2]; /* +1 because of queue, +1 because of '\n' */
Bool rxBufferLocked = FALSE;
#define IsRxBufferLocked() rxBufferLocked
Queue rxQueue;
#else
static Byte rxBuffer[(BASE64_SIZE) + 2]; /* +1 because of queue, +1 because of '\n' */
static Queue rxQueue;
#define IsRxBufferLocked() 0
#endif

static Bool isActive = FALSE;

#define SER_INIT_INDEPENDENT() \
  isActive = FALSE; \
  QueueInit(&rxQueue, rxBuffer, sizeof(rxBuffer));

/*----------------------------------------------------------------------------*/
/* Hardware Dependent Functions                                               */
/*----------------------------------------------------------------------------*/

#include <windows.h>

static HANDLE h = INVALID_HANDLE_VALUE;

Bool SerInit(UInt baudRate)
{
    TCHAR devName[24];
    COMMTIMEOUTS timeouts;

    SER_INIT_INDEPENDENT();

    wsprintf(devName, TEXT("\\\\.\\COM%d"), COM_NO);
    h = CreateFile(devName, GENERIC_READ | GENERIC_WRITE, 0, NULL, OPEN_EXISTING, 0, NULL);
    if (h == INVALID_HANDLE_VALUE) return FALSE;

    if (!SerSetBaudrate(baudRate)) {
        SerClose();
        return FALSE;
    }

    timeouts.ReadIntervalTimeout = MAXDWORD;
    timeouts.ReadTotalTimeoutConstant = 0;
    timeouts.ReadTotalTimeoutMultiplier = 0;
    timeouts.WriteTotalTimeoutConstant = 0;
    timeouts.WriteTotalTimeoutMultiplier = 0;
    if (!SetCommTimeouts(h, &timeouts)) {
        SerClose();
        return FALSE;
    }

    return TRUE;
}

void SerClose(void)
{
    if (h != INVALID_HANDLE_VALUE) {
        CloseHandle(h);
        h = INVALID_HANDLE_VALUE;
    }
}

Bool SerSetBaudrate(UInt baudRate)
{
    DCB dcb = { 0 };

    if (h == INVALID_HANDLE_VALUE) return FALSE;

    dcb.DCBlength = sizeof(DCB);
    if (!GetCommState(h, &dcb)) {
        return FALSE;
    }

    dcb.BaudRate = baudRate;
    dcb.ByteSize = 8;
    dcb.Parity = 0;
    dcb.StopBits = 0;

    if (!SetCommState(h, &dcb)) {
        SerClose();
        return FALSE;
    }

    return TRUE;
}

Bool SerIsActive(void)
{
    DWORD err;
    static COMSTAT stat;

    if (h == INVALID_HANDLE_VALUE) return FALSE;
    if (isActive) return TRUE;

    if (!ClearCommError(h, &err, &stat)) return FALSE;

    if (err & (CE_BREAK | CE_FRAME | CE_OVERRUN | CE_RXOVER | CE_RXPARITY)) return isActive = TRUE;

    return isActive = stat.cbInQue != 0;
}

void SerWaitForTx(void)
{
    /* SerPut() already waits for the transmission */
}

Bool SerPutWithoutChecksum(Byte b)
{
    DWORD w;
    static COMSTAT stat;

    if (h == INVALID_HANDLE_VALUE) return FALSE;

    if (!WriteFile(h, &b, 1, &w, NULL)) return FALSE;
    if (w == 0) return FALSE;
    do {
        if (!ClearCommError(h, &w, &stat)) break;
    } while (stat.cbOutQue > 0);

    return TRUE;
}

static void SerFillRxBuffer(void)
{
    Byte b;
    DWORD r;

    if (IsRxBufferLocked()) return;

    if (h != INVALID_HANDLE_VALUE) {
        while (!QueueIsFull(&rxQueue) && ReadFile(h, &b, 1, &r, NULL)) {
            if (r == 0) break;
            QueuePut(&rxQueue, b);
        }
    }
}

/*----------------------------------------------------------------------------*/
/* Hardware Independent Functions                                             */
/*----------------------------------------------------------------------------*/

Bool SerGet(Byte *p)
{
    SerFillRxBuffer();
    return QueueGet(&rxQueue, p);
}

#ifdef SUPPORT_CHECKSUM
Bool SerPut(Byte b)
{
    static Byte chksum = 0;

    if (b != '\n') {
        chksum -= b;
        if (!SerPutWithoutChecksum(b)) return FALSE;
    }
    else {
        if (!SerPutWithoutChecksum('\t')) return FALSE;
        if (!SerPutWithoutChecksum(GetHexDigit(chksum >> 4))) return FALSE;
        if (!SerPutWithoutChecksum(GetHexDigit(chksum & 0x0f))) return FALSE;
        if (!SerPutWithoutChecksum('\n')) return FALSE;
        chksum = 0;
    }

    return TRUE;
}
#endif

Bool SerWrite(const char *str) {
    while (*str) {
        if (!SerPut(*str++)) return FALSE;
    }
    return TRUE;
}

Bool SerWriteOk(char ch)
{
    if (!SerPut('r')) return FALSE;
    if (!SerPut(ch)) return FALSE;
    return SerPut('\n');
}

Bool SerWriteOkLine(char ch, const char *str)
{
    if (!SerPut('r')) return FALSE;
    if (!SerPut(ch)) return FALSE;
    if (!SerWrite(str)) return FALSE;
    return SerPut('\n');
}

Bool SerWriteError(char ch)
{
    if (!SerPut('e')) return FALSE;
    if (!SerPut(ch)) return FALSE;
    return SerPut('\n');
}

Bool SerWriteErrorLine(char ch, const char *str)
{
    if (!SerPut('e')) return FALSE;
    if (!SerPut(ch)) return FALSE;
    if (!SerWrite(str)) return FALSE;
    return SerPut('\n');
}
