/*----------------------------------------------------------------------------*/
/* Firmware Obfuscation V1.0.4                                                */
/* Written in 2019 by Rudy Tellert Elektronik                                 */
/*                                                                            */
/* To the extent possible under law, the author has dedicated all copyright   */
/* and related and neighboring rights to the software "firmware obfuscation"  */
/* and its documentation to the public domain. This software and its          */
/* documentation are distributed without any warranty.                        */
/*                                                                            */
/* See also                                                                   */
/* http://creativecommons.org/publicdomain/zero/1.0/                          */
/* ---------------------------------------------------------------------------*/

#include "bootldr.h"

static Hash h;
static Byte tmp[8];

void ReverseArray(Byte *arr, size_t arrSize)
{
    size_t i;
#define last arrSize
    last--;
    for (i = 0; i < last; i++, last--)
    {
        Byte tmpByte = arr[i];
        arr[i] = arr[last];
        arr[last] = tmpByte;
    }
#undef last
}

#ifdef SUPPORT_BIG_ENDIAN
Bool IsLittleEndian(void)
{
    const UShort tmpUShort = 0xff00;
    return ((const Byte*)&tmpUShort)[0] == 0;
}
#endif

static Byte Num(UInt u, Byte *dest)
{
    if (u < 0xff) {
        *dest = (Byte)u;
        return 1;
    }
    else if (u < 0xffff) {
        dest[0] = 0xff;
        *(UShort *)(dest + 1) = (UShort)u;
#ifdef SUPPORT_BIG_ENDIAN
        if (!IsLittleEndian()) ReverseArray(dest + 1, 2);
#endif
        return 3;
    }
    else {
        dest[0] = 0xff;
        dest[1] = 0xff;
        dest[2] = 0xff;
        *(UInt *)(dest + 3) = u;
#ifdef SUPPORT_BIG_ENDIAN
        if (!IsLittleEndian()) ReverseArray(dest + 3, 4);
#endif
        return 7;
    }
}

static Bool GetMacs(UInt seg, const MacArgs *args, Byte *dest)
{
    const Byte nul = 0;
#ifdef USE_MAC_WITH_PERMUTATION
    static Byte permutation0[PERMUTATION128_BLOCK_LENGTH];
#endif

#ifdef USE_MAC_WITH_PERMUTATION
    if (!InitKeyedHash(&h, args->Key, args->KeySize)) return FALSE;
#else 
    InitHash(&h);
#endif
    UpdateHash(&h, tmp, Num(seg, tmp));
#ifndef USE_MAC_WITH_PERMUTATION
    UpdateHash(&h, tmp, Num((UInt)args->KeySize, tmp));
    UpdateHash(&h, args->Key, args->KeySize);
#else
    UpdateHash(&h, tmp, Num(PERMUTATION128_BLOCK_LENGTH, tmp));
    UpdateHash(&h, h.Permutation0, PERMUTATION128_BLOCK_LENGTH);
#endif
    UpdateHash(&h, tmp, Num((UInt)args->NonceSize, tmp));
    UpdateHash(&h, args->Nonce, args->NonceSize);
    UpdateHash(&h, tmp, Num(args->Type, tmp));
    if (args->NonceSize != 0) {
        UpdateHash(&h, tmp, Num((UInt)args->MsgSize, tmp));
        UpdateHash(&h, args->Msg, args->MsgSize);
    }
    else {
        UpdateHash(&h, tmp, Num(args->DevNo, tmp));
    }
    GetHash(&h, dest, HASH_LENGTH);
    
#ifdef DOUBLE_HASHING_REQUIRED
#ifdef USE_MAC_WITH_PERMUTATION
    if (!InitKeyedHash(&h, args->Key, args->KeySize)) return FALSE;
#else 
    InitHash(&h);
#endif
    UpdateHash(&h, &nul, 1);
    UpdateHash(&h, dest, HASH_LENGTH);
    GetHash(&h, dest, HASH_LENGTH);
#endif

    return TRUE;
}

size_t GetMac(const MacArgs *args, Byte *dest, size_t destSize)
{
    UInt seg = 1;
    size_t len = destSize;
    size_t n;
    static Byte macs[HASH_LENGTH];

    while (len) {
        if (!GetMacs(seg++, args, macs)) return 0;
        n = (len < HASH_LENGTH) ? len : HASH_LENGTH;
        memcpy(dest, macs, n);
        dest += n;
        len -= n;
    }

    return destSize;
}

#ifdef USE_HASH_INSTEAD_OF_MAC
size_t GetKeyedHash(const Byte *key, size_t keySize, const Byte *msg, size_t msgSize, Byte *dest, size_t destSize) 
{
#ifdef USE_KEYED_HASH_WITH_PERMUTATION128
    if (!InitKeyedHash(&h, key, keySize)) return 0;
#else 
    InitHash(&h);
    UpdateHash(&h, key, keySize);
#endif
    UpdateHash(&h, msg, msgSize);
    return GetHash(&h, dest, destSize);
}
#endif

#ifdef IMPLEMENT_CTR
Bool GetCtr(const Byte *key, size_t keySize, const Byte *nonce, size_t nonceSize, Byte *msg, size_t msgSize, UInt msgNo)
{
    size_t len = msgSize;
    UInt i = 0;
    unsigned n;
    unsigned k;
    static Byte block[PERMUTATION128_BLOCK_LENGTH];
    static Permutation128 p;
    
    if (!InitPermutation128(&p, key, keySize)) return FALSE;
    
    while (len) {
        memcpy(block, &i, 4);
#ifdef SUPPORT_BIG_ENDIAN
        if (!IsLittleEndian()) ReverseArray(block, 4);
#endif
        memcpy(block + 4, &msgNo, 4);
#ifdef SUPPORT_BIG_ENDIAN
        if (!IsLittleEndian()) ReverseArray(block + 4, 4);
#endif
        memcpy(block + 8, nonce, 8);
        if (ForwardPermutation128(&p, block, 16, block, PERMUTATION128_BLOCK_LENGTH) != PERMUTATION128_BLOCK_LENGTH) return FALSE;
        n = (unsigned)((len < PERMUTATION128_BLOCK_LENGTH) ? len : PERMUTATION128_BLOCK_LENGTH);
        for (k = 0; k < n; k++) msg[k] ^= block[k];
        len -= n;
        msg += n;
        i++;
    }

    return TRUE;
}
#endif   