﻿/*----------------------------------------------------------------------------*/
/* Firmware Obfuscation V1.0.4                                                */
/* Written in 2019 by Rudy Tellert Elektronik                                 */
/*                                                                            */
/* To the extent possible under law, the author has dedicated all copyright   */
/* and related and neighboring rights to the software "firmware obfuscation"  */
/* and its documentation to the public domain. This software and its          */
/* documentation are distributed without any warranty.                        */
/*                                                                            */
/* See also                                                                   */
/* http://creativecommons.org/publicdomain/zero/1.0/                          */
/* ---------------------------------------------------------------------------*/

using Microsoft.Win32;
using System;
using System.ComponentModel;
using System.IO;
using System.IO.Ports;
using System.Runtime.Serialization;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using Tellert.Bootloader;
using System.Threading;

namespace appldr
{
    public partial class AppldrForm : Form
    {
        Bootloader bl;
        Firmware fw;
        byte[][] core;
        byte[][][] config;
        string regKey;
        string port;
        string[] ports;
        string firmware;
        string[] firmwareArr;
        enum Status { Init, WaitingForDevice, WaitingForConfirmation, Programming, Success, Error };
        Status status;
        int startTime;
        Config cfg;

        public AppldrForm()
        {
            InitializeComponent();

            string uri = System.Reflection.Assembly.GetExecutingAssembly().CodeBase;
            string xmlFileName = System.IO.Path.GetFileNameWithoutExtension(new Uri(uri).LocalPath) + ".xml";
            var ds = new DataContractSerializer(typeof(Config));
            using (Stream s = File.OpenRead(xmlFileName)) cfg = (Config)ds.ReadObject(s);

            bl = new Bootloader();
            fw = new Firmware();
            firmwareArr = new string[1];
            firmwareArr[0] = cfg.firmwareName;
            status = Status.Init;

            if (cfg.title != null && cfg.title != "") Text = cfg.title;

            regKey = "HKEY_CURRENT_USER\\SOFTWARE\\Tellert\\APPLDR\\2.0\\" + cfg.regName;

            port = (string)Registry.GetValue(regKey, "Port", null);
            firmware = (string)Registry.GetValue(regKey, "Item", null);
            
            ports = SerialPort.GetPortNames();
            Array.Sort<string>(ports);
            foreach (string s in ports) portComboBox.Items.Add(s);
            for (int i = 0; i < ports.Length; i++)
            {
                if (ports[i] == port)
                {
                    portComboBox.SelectedIndex = i;
                    break;
                }
            }
            nextButton.Enabled = portComboBox.SelectedIndex >= 0;

            foreach (string s in firmwareArr) firmwareComboBox.Items.Add(s);
            for (int i = 0; i < firmwareArr.Length; i++)
            {
                if (firmwareArr[i] == firmware)
                {
                    firmwareComboBox.SelectedIndex = i;
                    break;
                }
            }
            if (firmwareComboBox.SelectedIndex < 0) firmwareComboBox.SelectedIndex = 0;
            if (firmwareArr.Length <= 1) firmwareComboBox.Enabled = false;

            if (nextButton.Enabled) nextButton.Focus();
        }

        private void closeButton_Click(object sender, EventArgs e)
        {
            Close();
        }

        private void timer_Tick(object sender, EventArgs e)
        {
            switch (status)
            {
                case Status.Init:
                    string[] ports2 = SerialPort.GetPortNames();
                    Array.Sort<string>(ports2);
                    bool modified = false;

                    if (ports.Length != ports2.Length) modified = true;
                    else
                    {
                        for (int i = 0; i < ports.Length; i++)
                        {
                            if (ports[i] != ports2[i])
                            {
                                modified = true;
                                break;
                            }
                        }
                    }

                    if (modified) {
                        string currentPort = null;
                        int i = portComboBox.SelectedIndex;
                        if (i >= 0) currentPort = ports[i];
                        portComboBox.Items.Clear();
                        ports = ports2;
                        foreach (string s in ports) portComboBox.Items.Add(s);
                        for (i = 0; i < ports.Length; i++)
                        {
                            if (ports[i] == currentPort)
                            {
                                portComboBox.SelectedIndex = i;
                                break;
                            }
                        }
                        nextButton.Enabled = portComboBox.SelectedIndex >= 0;
                    }
                    break;

                case Status.WaitingForDevice:
                    if (!bl.IsInit)
                    {
                        bl.ReadVersion();
                        int br = bl.GetMaxBaudRate();
                        if (cfg.maxBaudRate > 0 && br > cfg.maxBaudRate) br = cfg.maxBaudRate;
                        if (bl.ChangeBaudRate(br))
                        {
                            if (bl.ReadInfo())
                            {
                                if (!fw.IsCompatible())
                                {
                                    timer.Enabled = false;
                                    MessageBox.Show(Properties.Resources.ErrorIncompatibleDevice, Text, MessageBoxButtons.OK, MessageBoxIcon.Error);
                                    timer.Enabled = true;
                                    prevButton_Click(null, null);
                                    return;
                                }
                                status = Status.WaitingForConfirmation;
                                initLabel.Visible = false;
                                initLabel.Refresh();

                                bl.UpdateAppInfo();
                                string dn = Bootloader.Info.deviceNumber.ToString();
                                if (Bootloader.Info.deviceNumber == 0) dn = "";
                                else dn += " ";
                                if (Bootloader.appName.Length != 0) dn += "(" + Bootloader.appName + ")";
                                deviceLabel2.Text = Bootloader.Info.deviceNumber.ToString();
                                if (dn.Length != 0)
                                {
                                    deviceLabel2.Text = dn;
                                    deviceLabel1.Visible = true;
                                    deviceLabel2.Visible = true;
                                }
                                nextButton.Enabled = true;
                                nextButton.Focus();
                                if (cfg.startPage > 1) nextButton_Click(null, null);
                            }
                            else
                            {
                                prevButton_Click(null, null);
                            }
                        }
                        else
                        {
                            prevButton_Click(null, null);
                        }
                    }
                    break;
            }
        }

        private void portComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            nextButton.Enabled = true;
        }

        private string GetRemainingTime(int val, int max)
        {
            if (val <= 0) return null;
            int u = Environment.TickCount - startTime;
            int T = (max * u) / val;
            int t = (T >= u) ? (T - u) : 0;
            t = t / 1000 + ((t % 1000 != 0) ? 1 : 0);
            if (t == 0) return null;
            if (t < 60) return t.ToString() + " s";
            int hour = t / 3600;
            t %= 3600;
            int min = t / 60;
            int sec = t % 60;
            return (hour != 0) ? 
              hour.ToString() + ":" + min.ToString("D2") + ":" + sec.ToString("D2") : 
              min.ToString() + ":" + sec.ToString("D2");
       }

        private void nextButton_Click(object sender, EventArgs ea)
        {
            switch (status)
            {
                case Status.Init:
                    if (portComboBox.SelectedIndex >= 0)
                    {
                        if (!fw.Read(cfg.tsfFileName))
                        {
                            MessageBox.Show(fw.IsChecksumMismatch ? Properties.Resources.ErrorChecksum : Properties.Resources.ErrorIncompatibleFile, Text, MessageBoxButtons.OK, MessageBoxIcon.Error);
                            return;
                        }

                        SerialPort sp = new SerialPort();
                        try
                        {
                            port = ports[portComboBox.SelectedIndex];
                            sp.PortName = port;
                            sp.Open();
                        }
                        catch (Exception e)
                        {
                            MessageBox.Show(e.Message, Text, MessageBoxButtons.OK, MessageBoxIcon.Error);
                            return;
                        }
                        sp.Close();
                        Registry.SetValue(regKey, "Port", port);
                        if (firmwareArr.Length > 1) Registry.SetValue(regKey, "Item", firmwareArr[firmwareComboBox.SelectedIndex]);

                        initLabel.Visible = true;
                        bl.Init(port);
                        status = Status.WaitingForDevice;
                        nextButton.Enabled = false;
                        portComboBox.Enabled = false;
                        prevButton.Enabled = true;
                    }
                    break;
                case Status.WaitingForConfirmation:
                    nextButton.Enabled = false;
                    prevButton.Enabled = false;
                    timeLabel1.Visible = true;
                    timeLabel1.Refresh();
                    core = fw.GetCoreBlocks();
                    config = fw.GetConfigBlocks();
                    progressBar.Minimum = 0;
                    progressBar.Maximum = Firmware.GetBlockCount(core, config) - 1;
                    progressBar.Value = 0;
                    //timeLabel2.BackColor = System.Drawing.Color.Transparent;
                    progressLabel.Visible = true;
                    progressLabel.Refresh();
                    progressBar.Visible = true;
                    //bl.PrepareBlockCommand(Bootloader.Command.WriteObfuscatedBlock);
                    startTime = Environment.TickCount;
                    status = Status.Programming;
                    bw.RunWorkerAsync();
                    break;
            }
        }

        private void Form1_Shown(object sender, EventArgs e)
        {
            int? left, top;
            left = (int?)Registry.GetValue(regKey, "Left", null);
            top = (int?)Registry.GetValue(regKey, "Top", null);
            if (left != null && top != null)
            {
                Left = (int)left;
                Top = (int)top;
            }
            Opacity = 1;

            if (cfg.startPage != 0) nextButton_Click(null, null);
        }

        private void Form_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (status >= Status.WaitingForConfirmation && status <= Status.Programming) bl.RunApp();
            bl.Exit();
            bw.CancelAsync();
            //while (bw.IsBusy) ;

            Registry.SetValue(regKey, "Left", Left);
            Registry.SetValue(regKey, "Top", Top);
        }

        private void prevButton_Click(object sender, EventArgs e)
        {
            if (status >= Status.WaitingForConfirmation && status <= Status.Programming) bl.RunApp();
            bl.Exit();
            bw.CancelAsync();
            while (bw.IsBusy) ;

            prevButton.Enabled = false;
            nextButton.Enabled = true;
            nextButton.Focus();

            status = Status.Init;
            portComboBox.Enabled = true;
            deviceLabel1.Visible = false;
            deviceLabel2.Visible = false;
            timeLabel1.Visible = false;
            timeLabel2.Visible = false;
            progressLabel.Visible = false;
            progressBar.Visible = false;
            initLabel.Visible = false;
            errorLabel.Visible = false;
            successLabel.Visible = false;
        }
        
        private void bw_ProgressChanged(object sender, ProgressChangedEventArgs e)
        {
            int i = e.ProgressPercentage;
            if (i > progressBar.Maximum) i = progressBar.Maximum;
            progressBar.Value = i;
            if (i == 1) startTime = Environment.TickCount;
            progressBar.Refresh();
            string s = GetRemainingTime(i - 1, progressBar.Maximum - 1);
            if (s != null)
            {
                timeLabel2.Visible = true;
                timeLabel2.Text = s;
                timeLabel2.Refresh();
            }
        }

        private void bw_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            bl.Exit();

            timeLabel2.Visible = false;
            progressLabel.Visible = false;
            progressBar.Visible = false;
            deviceLabel1.Visible = false;
            deviceLabel2.Visible = false;
            timeLabel1.Visible = false;
            if (status == Status.Error)
            {
                errorLabel.Visible = true;
            }
            else
            {
                successLabel.Visible = true;
                if (cfg.autoClose) Close();
            }
            prevButton.Enabled = true;
        }
        
        private void bw_DoWork(object sender, DoWorkEventArgs e)
        {
            int i;
            int m = 0;

            for (i = 0; i < core.Length; i++)
            {
                if (bw.CancellationPending)
                {
                    break;
                }

                bw.ReportProgress(m++);

                if (!bl.WriteBlockCommand(Bootloader.Command.WriteObfuscatedBlock, core[i], 0, core[i].Length, i, cfg.timeoutT0, cfg.timeoutT1))
                {
                    status = Status.Error;
                    break;
                }
            }
            if (status != Status.Error && i == core.Length)
            {
                for (i = 0; i < config.Length; i++)
                {
                    bl.NewBlockSequence();
                    int k; 
                    for (k = 0; k < config[i].Length; k++)
                    {
                        if (bw.CancellationPending)
                        {
                            break;
                        }

                        bw.ReportProgress(m++);

                        if (!bl.WriteBlockCommand(Bootloader.Command.WriteNonObfuscatedBlock, config[i][k], 0, config[i][k].Length, k, cfg.timeoutT0, cfg.timeoutT1))
                        {
                            status = Status.Error;
                            break;
                        }
                    }
                    if (status == Status.Error) break;
                }
            }
            if (status != Status.Error) status = Status.Success;

            bl.RunApp();
        }
    }

    [DataContract(Namespace = "http://www.tellert.de/2015/appldr")]
    public class Config
    {
        [DataMember]
        public string port, tsfFileName, firmwareName, regName, title;

        [DataMember]
        public int startPage;

        [DataMember]
        public bool autoClose;

        [DataMember]
        public int timeoutT0;

        [DataMember]
        public int timeoutT1;

        [DataMember]
        public int maxBaudRate;
    }
}
