﻿/*----------------------------------------------------------------------------*/
/* Firmware Obfuscation V1.0.4                                                */
/* Written in 2019 by Rudy Tellert Elektronik                                 */
/*                                                                            */
/* To the extent possible under law, the author has dedicated all copyright   */
/* and related and neighboring rights to the software "firmware obfuscation"  */
/* and its documentation to the public domain. This software and its          */
/* documentation are distributed without any warranty.                        */
/*                                                                            */
/* See also                                                                   */
/* http://creativecommons.org/publicdomain/zero/1.0/                          */
/* ---------------------------------------------------------------------------*/

using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO.Ports;
using System.Text;
using System.Threading;
using Tellert.Format;

namespace Tellert.Bootloader
{
    public static class StringHelper
    {
        public static byte[] ToBytes(this string str)
        {
            if (str == null) str = "";
            return System.Text.Encoding.Default.GetBytes(str);
        }

        public static void CopyTo(this string str, byte[] dest, int offset, int maxSize)
        {
            byte[] b = str.ToBytes();
            int size = b.Length;
            if (size > maxSize) size = maxSize;
            Array.Copy(b, 0, dest, offset, size);
        }

        public static void CopyTo(this uint u, byte[] dest, int offset, int size = 4)
        {
            byte[] b = BitConverter.GetBytes(u);
            Array.Copy(b, 0, dest, offset, size);
        }

        static byte HexNibble(char ch)
        {
            if (ch >= '0' && ch <= '9') return (byte)(ch - '0');
            if (ch >= 'A' && ch <= 'F') return (byte)(ch - 'A' + 10);
            if (ch >= 'a' && ch <= 'f') return (byte)(ch - 'a' + 10);
            return 0;
        }

        public static int ChecksumHexNibble(char ch)
        {
            if (ch >= '0' && ch <= '9') return (int)(ch - '0');
            if (ch >= 'A' && ch <= 'F') return (int)(ch - 'A' + 10);
            if (ch >= 'a' && ch <= 'f') return (int)(ch - 'a' + 10);
            return -1;
        }

        public static byte[] HexToBytes(this string str)
        {
            if (str == null) return null;

            int hexDigits = 0;
            for (int i = 0; i < str.Length; i++) if (str[i] > ' ') hexDigits++;

            int k = 0;
            byte[] b = new byte[hexDigits / 2];
            for (int i = 0; i < b.Length; i++)
            {
                while (str[k] <= ' ') k++;
                byte hi = HexNibble(str[k++]);
                while (str[k] <= ' ') k++;
                byte lo = HexNibble(str[k++]);
                b[i] = (byte)((hi << 4) + lo);
            }

            return b;
        }

        public static void Clear(this byte[] dest)
        {
            Array.Clear(dest, 0, dest.Length);
        }

        public static uint ToLittleEndian(this uint val)
        {
            if (!BitConverter.IsLittleEndian)
            {
                byte[] b = BitConverter.GetBytes(val);
                Array.Reverse(b);
                val = BitConverter.ToUInt32(b, 0);
            }
            return val;
        }
    }

    public static class Checksum
    {
        public static uint Compute(byte[] data, int offset, int size)
        {
            uint state = 1;

            for (int i = offset; size-- != 0; i++)
            {
                state += (uint)((state << 8) + (data[i] + 1));
            }

            return state;
        }

        public static uint Compute(byte[] data)
        {
            return Compute(data, 0, data.Length);
        }
    }

    public static class Obfuscation
    {
        public static void Add(byte[] data, uint salt)
        {
            if (data == null || data.Length == 0) return;
            byte[] reference = (byte[])data.Clone();
            uint state = (uint)(3000001321 * salt + 9000000101);
            data[0] ^= (byte)state;
            for (int i = 1; i < data.Length; i++)
            {
                state = (uint)(3000001321 * (state + reference[i - 1]) + 9000000101);
                data[i] ^= (byte)(state / 2);
            }
        }

        public static void Remove(byte[] data, uint salt)
        {
            if (data == null || data.Length == 0) return;
            uint state = (uint)(3000001321 * salt + 9000000101);
            data[0] ^= (byte)state;
            for (int i = 1; i < data.Length; i++)
            {
                state = (uint)(3000001321 * (state + data[i - 1]) + 9000000101);
                data[i] ^= (byte)(state / 2);
            }
        }
    }

    class TimeoutCounter
    {
        bool started = false;
        int startTime, period;

        public TimeoutCounter(int p = 0)
        {
            period = p;
            Start();
        }
        public void Start()
        {
            started = true;
            startTime = Environment.TickCount;
        }
        public int Period
        {
            get { return period; }
            set { period = value; }
        }
        public void Cancel()
        {
            started = false;
        }
        public bool Signaled
        {
            get
            {
                if (started)
                {
                    if (Environment.TickCount - startTime < period) return false;
                    started = false;
                }
                return true;
            }
        }
    }

    public class Firmware
    {
        public struct Drm
        {
            public uint type;
            public uint min;
            public uint max;
            public uint xorMask;
            public uint reqFeatures;

            public bool DrmTestFailed(uint val) {
                return (min > 0 && val < min) || (max > 0 && val > max) ||
                    (((val ^ xorMask) & reqFeatures) != reqFeatures);
            }
        }

        public struct Range
        {
            public uint address;
            public uint length;
        }

        public struct Area
        {
            public uint address;
            public byte[] data;
        }

        public class Map
        {
            public Drm[] drm;
            public uint eraseRegions;
            public Range[] eraseRange;
            public Area[] memoryArea;

            public bool DrmTestFailed(uint type, uint val)
            {
                foreach (Drm d in drm)
                {
                    if (d.type == type) { 
                        if (d.DrmTestFailed(val)) return true;
                    }
                }
                return false;
            }
        }

        public class Info
        {
            public uint id;
            public uint features;
            public uint versionMajor;
            public uint versionMinor;
            public uint revision;
            public uint build;
            public uint config;
            public Utc date;
            public string desc;
            public string name;
            public string version;
        }

        public class Core
        {
            public Drm[] drm;
            public uint salt;
            public byte[] obfuscatedInfo;
            public byte[] obfuscationKey;
            public Info info;
            public byte[][] blocks;
            public uint configurationType;
            public uint configurationIndex;

            public bool DrmTestFailed(uint type, uint val)
            {
                foreach (Drm d in drm)
                {
                    if (d.type == type)
                    {
                        if (d.DrmTestFailed(val)) return true;
                    }
                }
                return false;
            }
        }

        public class Config
        {
            public Map[] sequence;
        }

        bool checksumMismatch;
        int selectedCore;
        int selectedConfig;
        public Core[] core;
        public Config[] config;
        public int fileSize;

        public int SelectedCore
        {
            get { return selectedCore; }
            set {
                if (core == null || value >= core.Length)
                {
                    selectedCore = selectedConfig = -1;
                    if (config != null && config.Length == 1) selectedConfig = 0;
                }
                else 
                {
                    selectedCore = value;
                    selectedConfig = -1;
                    // Find matching config...
                    if (config != null && config.Length == 1) selectedConfig = 0;
                }
            } 
        }

        public bool IsCompatible()
        {
            if (Bootloader.hwInfo.Length == 0) return false;

            selectedCore = selectedConfig = -1;
            if (core.Length > 0)
            {
                // Check firmware
                for (int i = 0; i < core.Length; i++) {
                    if (core[i].DrmTestFailed(0, Bootloader.Info.manufacturerId)) continue;
                    if (core[i].DrmTestFailed(1, Bootloader.Info.hardwareId)) continue;
                    if (core[i].DrmTestFailed(2, Bootloader.Info.hardwareVersion)) continue;
                    if (core[i].DrmTestFailed(3, Bootloader.Info.internalDeviceNumber)) continue;
                    if (core[i].DrmTestFailed(4, Bootloader.Info.deviceNumber)) continue;
                    if (core[i].DrmTestFailed(5, Bootloader.Info.customer)) continue;
                    if (core[i].DrmTestFailed(6, Bootloader.Info.manufacturingDate)) continue;
                    if (core[i].DrmTestFailed(64, Bootloader.Info.hardwareFeatures)) continue;
                    selectedCore = i;
                    break;
                }
            }
            if (config.Length > 0) {
                // Check config
                for (int i = 0; i < config.Length; i++)
                {
                    int j;
                    for (j = 0; j < config[i].sequence.Length; j++) {
                        if (config[i].sequence[j].DrmTestFailed(0, Bootloader.Info.manufacturerId)) break;
                        if (config[i].sequence[j].DrmTestFailed(1, Bootloader.Info.hardwareId)) break;
                        if (config[i].sequence[j].DrmTestFailed(2, Bootloader.Info.hardwareVersion)) break;
                        if (config[i].sequence[j].DrmTestFailed(3, Bootloader.Info.internalDeviceNumber)) break;
                        if (config[i].sequence[j].DrmTestFailed(4, Bootloader.Info.deviceNumber)) break;
                        if (config[i].sequence[j].DrmTestFailed(5, Bootloader.Info.customer)) break;
                        if (config[i].sequence[j].DrmTestFailed(6, Bootloader.Info.manufacturingDate)) break;
                        if (config[i].sequence[j].DrmTestFailed(64, Bootloader.Info.hardwareFeatures)) break;
                    }
                    if (j < config[i].sequence.Length) continue;
                    if (selectedCore < 0)
                    {
                        if (Bootloader.appInfo.Length == 0) return false;
                        for (j = 0; j < config[i].sequence.Length; j++)
                        {
                            if (config[i].sequence[j].DrmTestFailed(129, Bootloader.Info.appId)) break;
                            if (config[i].sequence[j].DrmTestFailed(130, Bootloader.Info.appVersion)) break;
                            if (config[i].sequence[j].DrmTestFailed(131, Bootloader.Info.appBuild)) break;
                            if (config[i].sequence[j].DrmTestFailed(132, Bootloader.Info.appConfig)) break;
                            if (config[i].sequence[j].DrmTestFailed(133, Bootloader.Info.appDate)) break;
                            if (config[i].sequence[j].DrmTestFailed(192, Bootloader.Info.appFeatures)) break;
                        }
                        if (j < config[i].sequence.Length) continue;
                    }
                    else {
                        if (!IsConfigCompatible(i)) continue;
                    }
                    selectedConfig = i;
                    break;
                }
            }

            return selectedCore >= 0 || selectedConfig >= 0;
        }

        public bool IsChecksumMismatch { get { return checksumMismatch; } }

        public bool IsConfigCompatible(int i)
        {
            int j;
            if (selectedCore < 0) return false;
            for (j = 0; j < config[i].sequence.Length; j++)
            {
                uint version = (core[selectedCore].info.versionMajor << 24) +
                    (core[selectedCore].info.versionMinor << 16) +
                    core[selectedCore].info.revision;
                if (config[i].sequence[j].DrmTestFailed(129, core[selectedCore].info.id)) break;
                if (config[i].sequence[j].DrmTestFailed(130, version)) break;
                if (config[i].sequence[j].DrmTestFailed(131, core[selectedCore].info.build)) break;
                if (config[i].sequence[j].DrmTestFailed(132, core[selectedCore].info.config)) break;
                if (config[i].sequence[j].DrmTestFailed(133, core[selectedCore].info.date.Ticks)) break;
                if (config[i].sequence[j].DrmTestFailed(192, core[selectedCore].info.features)) break;
            }
            return (j == config[i].sequence.Length);
        }

        public int SelectedConfig
        {
            get { return selectedConfig; }
            set { selectedConfig = value; }
        }

        public byte[][] GetCoreBlocks() { return (selectedCore >= 0) ? core[selectedCore].blocks : new byte[0][]; }

        byte[][][] GetConfigBlocksClassic()
        {
            byte[][][] blocks = new byte[config[selectedConfig].sequence.Length][][];
            for (int i = 0; i < blocks.Length; i++)
            {
                Map seq = config[selectedConfig].sequence[i];

                List<Area> areaList = new List<Area>();
                foreach (Area a in seq.memoryArea)
                {
                    if (a.data.Length == 0) continue;
                    HexFile hexFile = new HexFile();
                    hexFile.SetBytes(a.address, a.data);
                    uint size = (((hexFile.LastAddress - hexFile.FirstAddress + 1) + 255) / 256) * 256;
                    for (uint k = 0; k < size; k += 256)
                    {
                        if (!hexFile.IsDataAvailable((uint)(hexFile.FirstAddress + k), 256)) continue;

                        Area area = new Area();
                        area.address = hexFile.FirstAddress + k;
                        area.data = hexFile.GetBytes(hexFile.FirstAddress + k, 256);
                        areaList.Add(area);
                    }
                }
                Area[] areaArr = areaList.ToArray();

                Tsf tsf = new Tsf();
                tsf.Create(1024, Tsf.HeaderType.None);
                if (!BitConverter.IsLittleEndian) tsf.Reverse = true;

                Drm[] drmArr = seq.drm;
                if (tsf.WriteArray(5, drmArr.Length))
                {
                    for (int k = 0; k < drmArr.Length; k++)
                    {
                        if (drmArr[k].type != 0) tsf.Write(3, drmArr[k].type);
                        if (drmArr[k].min == drmArr[k].max)
                        {
                            if (drmArr[k].min != 0) tsf.Write(9, drmArr[k].min);
                        }
                        else
                        {
                            if (drmArr[k].min != 0) tsf.Write(5, drmArr[k].min);
                            if (drmArr[k].max != 0) tsf.Write(7, drmArr[k].max);
                        }
                        if (drmArr[k].xorMask != 0) tsf.Write(11, drmArr[k].xorMask);
                        if (drmArr[k].reqFeatures != 0) tsf.Write(13, drmArr[k].reqFeatures);
                        tsf.WriteEnd();
                    }
                }
                tsf.Write(7, seq.eraseRegions);

                blocks[i] = new byte[areaArr.Length + 1][];
                blocks[i][0] = tsf.ToBytes();

                for (int k = 0; k < areaArr.Length; k++)
                {
                    tsf.Create(1024, Tsf.HeaderType.None);
                    if (!BitConverter.IsLittleEndian) tsf.Reverse = true;
                    if (tsf.WriteArray(15, 1))
                    {
                        tsf.Write(17, areaArr[k].address);
                        tsf.Write0(19, areaArr[k].data, 0, 256);
                        tsf.WriteEnd();
                    }
                    blocks[i][k + 1] = tsf.ToBytes();
                }
            }

            return blocks;
        }

        byte[][][] GetConfigBlocksTiny()
        {
            byte[][][] blocks = new byte[config[selectedConfig].sequence.Length][][];

            for (int i = 0; i < blocks.Length; i++)
            {
                Map seq = config[selectedConfig].sequence[i];

                List<Area> areaList = new List<Area>();
                foreach (Area a in seq.memoryArea)
                {
                    if (a.data.Length == 0) continue;
                    HexFile hexFile = new HexFile();
                    hexFile.SetBytes(a.address, a.data);
                    uint size = (((hexFile.LastAddress - hexFile.FirstAddress + 1) + 255) / 256) * 256;
                    for (uint k = 0; k < size; k += 256)
                    {
                        if (!hexFile.IsDataAvailable((uint)(hexFile.FirstAddress + k), 256)) continue;

                        Area area = new Area();
                        area.address = hexFile.FirstAddress + k;
                        area.data = hexFile.GetBytes(hexFile.FirstAddress + k, 256);
                        areaList.Add(area);
                    }
                }
                Area[] areaArr = areaList.ToArray();

                Drm[] drmArr = seq.drm;
                byte[] block0 = new byte[(drmArr.Length + 1) * 12];
                Array.Copy(BitConverter.GetBytes(0xf8139cbd.ToLittleEndian()), block0, 4);
                Array.Copy(BitConverter.GetBytes(seq.eraseRegions.ToLittleEndian()), 0, block0, 4, 4);
                Array.Copy(BitConverter.GetBytes(((uint)drmArr.Length).ToLittleEndian()), 0, block0, 8, 4);
                for (int k = 0; k < drmArr.Length; k++)
                {
                    if ((drmArr[k].type & 0x40) == 0)
                    {
                        Array.Copy(BitConverter.GetBytes(drmArr[k].type.ToLittleEndian()), 0, block0, 12 + k * 12, 4);
                        Array.Copy(BitConverter.GetBytes(drmArr[k].min.ToLittleEndian()), 0, block0, 16 + k * 12, 4);
                        Array.Copy(BitConverter.GetBytes(drmArr[k].max.ToLittleEndian()), 0, block0, 20 + k * 12, 4);
                    }
                    else
                    {
                        Array.Copy(BitConverter.GetBytes(drmArr[k].type.ToLittleEndian()), 0, block0, 12 + k * 12, 4);
                        Array.Copy(BitConverter.GetBytes(drmArr[k].xorMask.ToLittleEndian()), 0, block0, 16 + k * 12, 4);
                        Array.Copy(BitConverter.GetBytes(drmArr[k].reqFeatures.ToLittleEndian()), 0, block0, 20 + k * 12, 4);
                    }
                }

                blocks[i] = new byte[areaArr.Length + 1][];
                blocks[i][0] = block0;
                
                for (int k = 0; k < areaArr.Length; k++)
                {
                    byte[] b = new byte[4 + 4 + 256];
                    Array.Copy(BitConverter.GetBytes(areaArr[k].address.ToLittleEndian()), b, 4);
                    Array.Copy(BitConverter.GetBytes(0x100U.ToLittleEndian()), 0, b, 4, 4);
                    Array.Copy(areaArr[k].data, 0, b, 8, 256);
                    blocks[i][k + 1] = b;
                }
            }

            return blocks;
        }

        public static int GetBlockCount(byte[][] core, byte[][][] config)
        {
            int c = core.Length;
            foreach (byte[][] arr in config) c += arr.Length;
            return c;
        }

        public byte[][][] GetConfigBlocks()
        {
            if (selectedConfig < 0) return new byte[0][][];

            switch (Bootloader.kind)
            {
                case Bootloader.Kind.Classic:
                    return GetConfigBlocksClassic();
                case Bootloader.Kind.Tiny:
                    return GetConfigBlocksTiny();
                default:
                    throw new Exception();
                    //return new byte[0][][];
            }
        }

        public bool AddConfig(Drm[] drm, uint eraseRegions, Area[] areas)
        {
            Config[] cfg = new Config[config != null ? config.Length + 1 : 1];
            int i;
            for (i = 0; i < cfg.Length - 1; i++) cfg[i] = config[i];
            cfg[i] = new Config();
            cfg[i].sequence = new Map[1];
            cfg[i].sequence[0] = new Map();
            Map map = cfg[i].sequence[0];
            map.drm = drm;
            map.eraseRegions = eraseRegions;
            map.eraseRange = new Range[0];
            map.memoryArea = areas;
            config = cfg;

            return true;
        }

        public void Init() 
        {
            selectedConfig = selectedCore = -1;
            core = null;
            config = null;
            fileSize = 0;
            checksumMismatch = false;
        }

        public bool Read(string fileName)
        {
            Tsf tsf = new Tsf();
            Drm[] drm = null;
            byte[][] blocks = null;
            Init();
            tsf.ReadFile(fileName);
            uint id = 0;
            if (tsf.ReadArraySize(1) != 0)
            {
                if (tsf.Select(2)) tsf.Read(ref id);
                tsf.SkipArray();
            }
            if (id != 0x86) return false;
            int nCore = tsf.ReadArraySize(27);
            core = new Core[nCore];
            for (int i = 0; i < nCore; i++)
            {
                uint salt = 0;
                byte[] drmData = null;
                byte[] obfuscationKey = null;
                byte[] obfuscatedInfo = null;
                if (tsf.Select(3)) tsf.Read(ref salt);
                if (tsf.Select(5))
                {
                    drmData = new byte[tsf.DataSize];
                    tsf.Read(drmData, 0, drmData.Length);
                }
                obfuscatedInfo = (byte[])drmData.Clone();
                if (salt != 0) Obfuscation.Remove(drmData, salt);
                Tsf tsf2 = new Tsf();
                int len = (drmData != null) ? drmData.Length : 0;
                tsf2.Open(drmData, 0, len);
                if (!BitConverter.IsLittleEndian) tsf2.Reverse = true;
                Info info = new Info();
                int nInfo = tsf2.ReadArraySize(3);
                if (nInfo != 0)
                {
                    byte[] bytes = null;
                    if (tsf2.Select(3)) tsf2.Read(ref info.id);
                    if (tsf2.Select(5)) tsf2.Read(ref info.features);
                    if (tsf2.Select(7)) tsf2.Read(ref info.versionMajor);
                    if (tsf2.Select(9)) tsf2.Read(ref info.versionMinor);
                    if (tsf2.Select(11)) tsf2.Read(ref info.revision);
                    if (tsf2.Select(13)) tsf2.Read(ref info.build);
                    if (tsf2.Select(15)) tsf2.Read(ref info.config);
                    if (tsf2.Select(17))
                    {
                        uint u = 0;
                        tsf2.Read(ref u);
                        info.date = new Utc(u);
                    }
                    if (tsf2.Select(25))
                    {
                        tsf2.Read(ref bytes);
                        info.desc = System.Text.Encoding.Default.GetString(bytes);
                    }
                    if (tsf2.Select(27))
                    {
                        tsf2.Read(ref bytes);
                        info.name = System.Text.Encoding.Default.GetString(bytes);
                    }
                    if (tsf2.Select(29))
                    {
                        tsf2.Read(ref bytes);
                        info.version = System.Text.Encoding.Default.GetString(bytes);
                    }
                    tsf2.SkipArray();
                }
                int nDrm = tsf2.ReadArraySize(5);
                drm = new Drm[nDrm];
                for (int k = 0; k < nDrm; k++)
                {
                    if (tsf2.Select(3)) tsf2.Read(ref drm[k].type);
                    if (tsf2.Select(5)) tsf2.Read(ref drm[k].min);
                    if (tsf2.Select(7)) tsf2.Read(ref drm[k].max);
                    if (tsf2.Select(9))
                    {
                        tsf2.Read(ref drm[k].min);
                        drm[k].max = drm[k].min;
                    }
                    if (tsf2.Select(11)) tsf2.Read(ref drm[k].xorMask);
                    if (tsf2.Select(13)) tsf2.Read(ref drm[k].reqFeatures);
                    tsf2.SkipCollection();
                }
                if (tsf.SelectArray(6))
                {
                    if (tsf.Select(3)) tsf.Read(ref obfuscationKey);
                    tsf.SkipArray();
                }
                int nBlocks = tsf.ReadArraySize(7);
                blocks = new byte[nBlocks][];
                for (int k = 0; k < nBlocks; k++)
                {
                    if (tsf.Select(3))
                    {
                        blocks[k] = new byte[tsf.DataSize];
                        tsf.Read(blocks[k], 0, tsf.DataSize);
                    }
                    tsf.SkipCollection();
                }
                uint configurationType = 0, configurationIndex = 0;
                if (tsf.Select(9)) tsf.Read(ref configurationType);
                if (tsf.Select(11)) tsf.Read(ref configurationIndex);
                tsf.SkipCollection();

                core[i] = new Core();
                core[i].salt = salt;
                core[i].obfuscationKey = obfuscationKey;
                core[i].obfuscatedInfo = obfuscatedInfo;
                core[i].drm = drm;
                core[i].info = info;
                core[i].blocks = blocks;
                core[i].configurationType = configurationType;
                core[i].configurationIndex = configurationIndex;
            }
            int nCfg = tsf.ReadArraySize(29);
            config = new Config[nCfg];
            for (int i = 0; i < nCfg; i++) {
                config[i] = new Config();
                int nSeq = tsf.ReadArraySize(25);
                config[i].sequence = new Map[nSeq];
                for (int j = 0; j < nSeq; j++) {
                    config[i].sequence[j] = new Map();
                    int nDrm = tsf.ReadArraySize(5);
                    config[i].sequence[j].drm = new Drm[nDrm];
                    for (int k = 0; k < nDrm; k++) {
                        config[i].sequence[j].drm[k] = new Drm();
                        if (tsf.Select(3)) tsf.Read(ref config[i].sequence[j].drm[k].type);
                        if (tsf.Select(5)) tsf.Read(ref config[i].sequence[j].drm[k].min);
                        if (tsf.Select(7)) tsf.Read(ref config[i].sequence[j].drm[k].max);
                        if (tsf.Select(9))
                        {
                            tsf.Read(ref config[i].sequence[j].drm[k].min);
                            config[i].sequence[j].drm[k].max = config[i].sequence[j].drm[k].min;
                        }
                        if (tsf.Select(11)) tsf.Read(ref config[i].sequence[j].drm[k].xorMask);
                        if (tsf.Select(13)) tsf.Read(ref config[i].sequence[j].drm[k].reqFeatures);
                        tsf.SkipCollection();
                    }
                    if (tsf.Select(7)) tsf.Read(ref config[i].sequence[j].eraseRegions);
                    int nErase = tsf.ReadArraySize(9);
                    config[i].sequence[j].eraseRange = new Range[nErase];
                    for (int k = 0; k < nErase; k++) {
                        config[i].sequence[j].eraseRange[k] = new Range();
                        if (tsf.Select(3)) tsf.Read(ref config[i].sequence[j].eraseRange[k].address);
                        if (tsf.Select(5)) tsf.Read(ref config[i].sequence[j].eraseRange[k].length);
                        tsf.SkipCollection();
                    }
                    int nMemory = tsf.ReadArraySize(15);
                    config[i].sequence[j].memoryArea = new Area[nMemory];
                    for (int k = 0; k < nMemory; k++) {
                        config[i].sequence[j].memoryArea[k] = new Area();
                        if (tsf.Select(17)) tsf.Read(ref config[i].sequence[j].memoryArea[k].address);
                        if (tsf.Select(19)) tsf.Read(ref config[i].sequence[j].memoryArea[k].data);
                        tsf.SkipCollection();                        
                    }
                    tsf.SkipCollection();
                }
                tsf.SkipCollection();
            }
            uint checksum = 0;
            bool checksumFound = tsf.Select(30);
            byte[] body = tsf.ToBytes();
            if (checksumFound) tsf.Read(ref checksum);
            if (checksum != Checksum.Compute(body, 0, body.Length - 1))
            {
                checksumMismatch = true;
                return false;
            }
            fileSize = tsf.Pos + 1;

            if (core.Length == 1) SelectedCore = 0;

            return true;
        }

        public bool Write(string fileName, int maxFileSizeToBeExpected = 0)
        {
            Tsf tsf = new Tsf();
            if (maxFileSizeToBeExpected == 0) maxFileSizeToBeExpected = fileSize;
            tsf.Create(maxFileSizeToBeExpected);
            if (tsf.WriteArray(1, 1))
            {
                tsf.Write(2, 0x86U);
                tsf.WriteEnd();
            }
            if (tsf.WriteArray(27, core.Length))
            {
                for (int i = 0; i < core.Length; i++)
                {
                    tsf.Write(3, core[i].salt);
                    tsf.Write0(5, core[i].obfuscatedInfo);
                    if (core[i].obfuscationKey != null && core[i].obfuscationKey.Length > 0)
                    {
                        if (tsf.WriteArray(6, 1))
                        {
                            tsf.Write0(3, core[i].obfuscationKey);
                            tsf.WriteEnd();
                        }
                    }
                    if (tsf.WriteArray(7, core[i].blocks.Length)) {
                        for (int k = 0; k < core[i].blocks.Length; k++)
                        {
                            tsf.Write0(3, core[i].blocks[k]);
                            tsf.WriteEnd();
                        }
                    }
                    tsf.Write(9, core[i].configurationType);
                    tsf.Write(11, core[i].configurationIndex);
                    tsf.WriteEnd();
                }
            }
            if (tsf.WriteArray(29, config.Length))
            {
                for (int i = 0; i < config.Length; i++)
                {
                    if (tsf.WriteArray(25, config[i].sequence.Length)) {
                        for (int j = 0; j < config[i].sequence.Length; j++) {
                            Map seq = config[i].sequence[j];
                            if (tsf.WriteArray(5, seq.drm.Length)) {
                                for (int k = 0; k < seq.drm.Length; k++) {
                                    tsf.Write(3, seq.drm[k].type);
                                    if (seq.drm[k].min != seq.drm[k].max) {
                                        tsf.Write(5, seq.drm[k].min);
                                        tsf.Write(7, seq.drm[k].max);
                                    }
                                    else {
                                        tsf.Write(9, seq.drm[k].min);
                                    }
                                    tsf.Write(11, seq.drm[k].xorMask);
                                    tsf.Write(13, seq.drm[k].reqFeatures);
                                    tsf.WriteEnd();
                                }
                            }
                            tsf.Write(7, seq.eraseRegions);
                            if (tsf.WriteArray(9, seq.eraseRange.Length)) {
                                for (int k = 0; k < seq.eraseRange.Length; k++) {
                                    tsf.Write(3, seq.eraseRange[k].address);
                                    tsf.Write(5, seq.eraseRange[k].length);
                                    tsf.WriteEnd();
                                }
                            }
                            if (tsf.WriteArray(15, seq.memoryArea.Length)) {
                                for (int k = 0; k < seq.memoryArea.Length; k++) {
                                    tsf.Write(17, seq.memoryArea[k].address);
                                    tsf.Write0(19, seq.memoryArea[k].data);
                                    tsf.WriteEnd();
                                }
                            }
                            tsf.WriteEnd();
                        }
                    }
                    tsf.WriteEnd();
                }
            }
            uint checksum = Checksum.Compute(tsf.ToBytes(), 0, tsf.ToBytes().Length);
            if (checksum != 0) tsf.Write(30, checksum);
            tsf.WriteEnd();
            fileSize = tsf.Pos + 1;
            return tsf.TryWriteFile(fileName);
        }
    }

    public class Bootloader : IDisposable
    {
        public static int delay = 0;
        static SerialPort serialPort = new SerialPort();
        static Thread thread = null;
        static volatile bool threadIsAlive = false;
        enum Status { Init0, Init1, Init2, Idle, IdleButError, Busy };
        static volatile Status status = Status.Init0;
        static volatile Command command = Command.None;
        static volatile string resultString = null;

        public static string version;
        public static string hwInfo;
        public static string appName;
        public static string appInfo;
        public static string configInfo;
        public static string dataInfo;
        public enum Kind {
            Unknown, Classic, Tiny
        };
        public static Kind kind = Kind.Unknown;
        public static uint maxSupportedBaudRate = 230400;

        public enum Command {
            None, Cancel, Pause,
            CreateNewSession = 'a',
            GetBootloaderInfo = 'b',
            GetHardwareInfo = 'c',
            GetAppName = 'd',
            GetAppInfo = 'f',
            GetConfigInfo = 'h',
            GetDataInfo = 'h',
            PrepareBlockModeForComparison = 'i',
            PrepareBlockModeForWrite = 'j',
            CompareObfuscatedBlock = 'k',
            WriteObfuscatedBlock = 'l',
            CompareNonObfuscatedBlock = 'm',
            WriteNonObfuscatedBlock = 'n',
            GetBlockCounter = 'o',
            ChangeBaudrate = 'q',
            PrepareBaudrate9600 = 's',
            PrepareBaudrate57600 = 't',
            PrepareBaudrate115200 = 'u',
            PrepareBaudrate230400 = 'v',
            UpdateAppInfo = 'x',
            RunApp = 'y',
            PrepareToRunApp = 'z'
        }

        public static class Info
        {
            public static uint manufacturerId;
            public static uint hardwareId;
            public static uint hardwareFeatures;
            public static uint hardwareVersion;
            public static uint manufacturingDate;
            public static uint internalDeviceNumber;
            public static uint deviceNumber;
            public static uint customer;
            public static uint deviceToken;
            public static uint appId;
            public static uint appFeatures;
            public static uint appVersion;
            public static uint appBuild;
            public static uint appConfig;
            public static uint appDate;

            static uint Parse(string info, char ch, uint def = 0)
            {
                int i, k;
                uint result = def;

                for (i = 0; i < info.Length; i++) if (info[i] == ch) break;
                for (k = i + 1; k < info.Length; k++) if (info[k] == ' ') break;
                try { result = uint.Parse(info.Substring(i + 1, k - i - 1), NumberStyles.HexNumber); }
                catch { }

                return result;
            }

            static public void Init()
            {
                manufacturerId = Parse(Bootloader.hwInfo, 'M');
                hardwareId = Parse(Bootloader.hwInfo, 'H');
                hardwareFeatures = Parse(Bootloader.hwInfo, 'F');
                hardwareVersion = Parse(Bootloader.hwInfo, 'V');
                manufacturingDate = Parse(Bootloader.hwInfo, 'D');
                internalDeviceNumber = Parse(Bootloader.hwInfo, 'I');
                deviceNumber = Parse(Bootloader.hwInfo, 'N');
                customer = Parse(Bootloader.hwInfo, 'C');
                deviceToken = Parse(Bootloader.hwInfo, 'T');
                appId = Parse(Bootloader.appInfo, 'A');
                appFeatures = Parse(Bootloader.appInfo, 'F');
                appVersion = Parse(Bootloader.appInfo, 'V');
                appBuild = Parse(Bootloader.appInfo, 'B');
                appConfig = Parse(Bootloader.appInfo, 'C');
                appDate = Parse(Bootloader.appInfo, 'D');
            }
        }

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            Exit();
        }

        ~Bootloader()
        {
            Dispose(false);
        }

        public static bool BaudrateChanged { get { return serialPort.BaudRate != 9600;  } }

        public static void ComThread(object bootloaderObject)
        {
            Bootloader bl = (Bootloader)bootloaderObject;
            int idx;
            int ch;
            uint resultLength = 0;
            byte[] key = new byte[] { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, // UART synchronisation
                                      0xed, 0xf1, 0xfe, 0xf2, 0xf5, 0xeb, 0xfc, 0xf3, 0xf6, 0xf8, // bootloader key to open the communication
                                      0xfd, 0xf9, 0xef, 0xfb, 0xfa, 0xec, 0xf0, 0xf4, 0xf7, 0xee,  };
            byte[] requestPrompt = new byte[1] { 0xff };
            byte[] commandBuff = new byte[1];
            StringBuilder resultBuff = new StringBuilder();
            TimeoutCounter delayTimer = new TimeoutCounter(delay);
            bool initDelayTimer;
            byte chksum = 0;
            int chksumPos = 0;

            serialPort.DiscardOutBuffer();
            serialPort.DiscardInBuffer(); 
            idx = 0;
            initDelayTimer = true;
            chksum = 0;
            chksumPos = 0;
            command = Command.None;
            status = Status.Init0;

            // Step 1: Opening sequence
            while (threadIsAlive)
            {
                if (serialPort.BytesToWrite < 100)
                {
                    try
                    {
                        serialPort.Write(key, idx, 1);
                        if (++idx == key.Length) idx = 0;
                    }
                    catch { }
                }
                if (serialPort.BytesToRead > 0)
                {
                    ch = '\0';
                    try { ch = serialPort.ReadByte(); }
                    catch { }
                    if (ch == '>')
                    {
                        serialPort.DiscardOutBuffer();
                        serialPort.DiscardInBuffer();
                        if (initDelayTimer)
                        {
                            initDelayTimer = false;
                            delayTimer.Start();
                        }
                        if (delayTimer.Signaled)
                        {
                            serialPort.Write("a");
                            status = Status.Init2;
                            resultLength = 0;
                            break;
                        }
                    }
                    else
                    {
                        idx = 0;
                        serialPort.DiscardOutBuffer();
                        serialPort.DiscardInBuffer();
                    }
                }
                if (status == Status.Init0) status = Status.Init1;
            }
            while (threadIsAlive)
            {
                ch = '\0';
                if (serialPort.BytesToRead > 0)
                {
                    try { ch = serialPort.ReadByte(); }
                    catch { }
                    if (ch == '>')
                    {
                        serialPort.DiscardOutBuffer();
                        serialPort.DiscardInBuffer();
                        serialPort.Write("a");
                        resultLength = 0;
                    }
                    else if (ch == 'e')
                    {
                        serialPort.DiscardOutBuffer();
                        serialPort.DiscardInBuffer();
                        serialPort.Write(requestPrompt, 0, 1);
                    }
                    else
                    {
                        switch(resultLength) {
                            case 0:
                                if (ch == 'r') resultLength++;
                                break;
                            case 1:
                                if (ch == 'a') resultLength++; else resultLength = 0;
                                break;
                            case 2:
                                if (ch == '\n') resultLength = 6;
                                else if (ch == '\t') resultLength++; else resultLength = 0;
                                break;
                            case 3:
                                if (ch == '2') resultLength++; else resultLength = 0;
                                break;
                            case 4:
                                if (ch == 'D') resultLength++; else resultLength = 0;
                                break;
                            case 5:
                                if (ch == '\n') resultLength++; else resultLength = 0;
                                break;
                            default:
                                resultLength = 0;
                                break;
                        }
                    }
                }
                if (resultLength == 6)
                {
                    Thread.Sleep(200);
                    serialPort.DiscardOutBuffer();
                    serialPort.DiscardInBuffer();
                    status = Status.Idle;
                    break;
                }
            }

            // Step 2: Command loop
            while (threadIsAlive)
            {
                // Step 2.1: Await new command
                while (threadIsAlive)
                {
                    if (command == Command.Cancel)
                    {
                        resultString = null;
                        status = Status.Idle;
                        command = Command.None;
                    }
                    else if (command == Command.Pause)
                    {
                        status = Status.Busy;
                        while (threadIsAlive && command == Command.Pause) ;
                    }
                    else if (command != Command.None)
                    {
                        serialPort.DiscardOutBuffer();
                        serialPort.DiscardInBuffer();
                        commandBuff[0] = (byte)command;
                        serialPort.Write(commandBuff, 0, 1);
                        status = Status.Busy;
                        resultString = null;
                        resultBuff.Clear();
                        break;
                    }
                    else if (serialPort.BytesToRead > 0)
                    {
                        try { serialPort.ReadByte(); }
                        catch { }
                    }
                }

                // Step 2.2: Await result
                while (threadIsAlive)
                {
                    if (command == Command.Cancel)
                    {
                        status = Status.Idle;
                        command = Command.None;
                        chksum = 0;
                        chksumPos = 0;
                        break;
                    }
                    else if (command != Command.None)
                    {
                        // Await result string
                        if (serialPort.BytesToRead > 0)
                        {
                            ch = '\0';
                            try { ch = serialPort.ReadByte(); }
                            catch { }
                            if (chksumPos > 0 && ch != '\n')
                            {
                                if (chksumPos == 1) {
                                    chksumPos = 2;
                                    int nibble = -1;
                                    if (ch >= 0) nibble = StringHelper.ChecksumHexNibble((char)ch);
                                    if (nibble != chksum >> 4) chksumPos = 4;
                                }
                                else if (chksumPos == 2)
                                {
                                    chksumPos = 3;
                                    int nibble = -1;
                                    if (ch >= 0) nibble = StringHelper.ChecksumHexNibble((char)ch);
                                    if (nibble != (int)(chksum & 0x0f)) chksumPos = 4;
                                }
                                else
                                {
                                    chksumPos = 4;
                                }
                            }
                            else if (ch == '\t') {
                                chksumPos = 1;
                            }
                            else if (ch >= ' ' && ch <= 'z')
                            {
                                if (ch != '>')
                                {
                                    chksum -= (byte)ch;
                                    resultBuff.Append((char)ch);
                                    if (resultBuff.Length == 254)
                                    {
                                        resultString = resultBuff.ToString();
                                        status = Status.Idle;
                                        Thread.Sleep(200);
                                        serialPort.DiscardOutBuffer();
                                        serialPort.DiscardInBuffer();
                                    }
                                }
                            }
                            else if (ch == '\n')
                            {
                                if (resultBuff.Length >= 2)
                                {
                                    if (resultBuff[1] == (char)command)
                                    {
                                        resultBuff.Remove(1, 1);
                                        resultString = resultBuff.ToString();
                                        status = (chksumPos == 0 || chksumPos == 3) ? Status.Idle : Status.IdleButError;
                                    }
                                    else
                                    {
                                        resultString = null;
                                        status = Status.IdleButError;
                                    }
                                }
                                else
                                {
                                    resultString = null;
                                    status = Status.IdleButError;
                                }
                                chksum = 0;
                                chksumPos = 0;
                            }
                        }
                        if (status == Status.Idle || status == Status.IdleButError)
                        {
                            command = Command.None;
                            break;
                        }
                    }
                }
            }

            resultString = null;
            command = Command.None;
        }

        public void Init(string portName)
        {
            if (thread != null) return;

            version = hwInfo = appName = appInfo = configInfo = dataInfo = null;
            kind = Kind.Unknown;
            maxSupportedBaudRate = 230400;

            serialPort.PortName = portName;
            serialPort.BaudRate = 9600;
            serialPort.DataBits = 8;
            serialPort.Parity = Parity.None;
            serialPort.StopBits = StopBits.One;
            serialPort.Handshake = Handshake.None;
            serialPort.ReadTimeout = 500;
            serialPort.WriteTimeout = 500;
            serialPort.Open();
            
            thread = new Thread(ComThread);
            threadIsAlive = true;
            thread.Start(this);
        }

        public void Exit()
        {
            if (threadIsAlive)
            {
                threadIsAlive = false;
                while (thread.IsAlive) ;
                thread = null;
                serialPort.Close();
                version = hwInfo = appInfo = appName = appInfo = configInfo = dataInfo = null;
            }
        }

        public bool IsInit
        {
            get { return status < Status.Idle; }
        }

        public bool IsIdle
        {
            get { return status == Status.Idle || status == Status.IdleButError; }
        }

        bool RequestCommand(Command cmd, string expectedString = null, int timeout = 500)
        {
            TimeoutCounter timeoutCounter = new TimeoutCounter(timeout);

            resultString = null;
            for (int i = 0; i < 3; i++) {
                command = cmd;
                timeoutCounter.Start();
                while (command != Command.None) if (timeoutCounter.Signaled) command = Command.Cancel;
                if (status == Status.IdleButError) resultString = null;
                if (resultString != null)
                {
                    if (resultString.Length > 0)
                    {
                        if (resultString[0] == 'r')
                        {
                            resultString = resultString.Substring(1);
                            if (expectedString == null || resultString == expectedString) return true;
                        }
                    }
                }
            }
            return false;
        }

        bool ReadInfo(Command cmd)
        {
            resultString = null;
            TimeoutCounter Timeout = new TimeoutCounter(500);

            if (!RequestCommand(cmd)) return false;
            string resultA = resultString;

            if (!RequestCommand(cmd)) return false;
            string resultB = resultString;

            if (resultA == resultB) return true;

            if (!RequestCommand(cmd)) return false;
            if (resultString != resultA && resultString != resultB) resultString = null;

            return true;
        }

        public bool ReadVersion()
        {
            ReadInfo(Command.GetBootloaderInfo);
            version = resultString ?? "";
            if (string.IsNullOrEmpty(version)) return false;
            if (version.Contains("TBL 3S"))
            {
                kind = Kind.Classic;
            }
            else if (version.Contains("TBL 3T"))
            {
                kind = Kind.Tiny;
            }
            else
            {
                kind = Kind.Unknown;
            }
            switch (version[version.Length - 1]) { 
                case 'D':
                    maxSupportedBaudRate = 230400;
                    break;
                case 'C':
                    maxSupportedBaudRate = 115200;
                    break;
                case 'B':
                    maxSupportedBaudRate = 57600;
                    break;
                default:
                    maxSupportedBaudRate = 9600;
                    break;
            }
            return true;
        }

        public bool ReadInfo()
        {
            if (string.IsNullOrEmpty(version)) ReadVersion();
            if (!string.IsNullOrEmpty(version))
            {
                ReadInfo(Command.GetHardwareInfo);
                hwInfo = resultString ?? "";
                ReadInfo(Command.GetAppName);
                appName = resultString ?? "";
                ReadInfo(Command.GetAppInfo);
                appInfo = resultString ?? "";
                ReadInfo(Command.GetConfigInfo);
                configInfo = resultString ?? "";
                ReadInfo(Command.GetDataInfo);
                dataInfo = resultString ?? "";
            }
            else
            {
                hwInfo = appName = appInfo = configInfo = dataInfo = "";
                return false;
            }
            Bootloader.Info.Init();
            return true;
        }

        public bool CheckBaudRate(int baudRate)
        {
            TimeoutCounter Timeout = new TimeoutCounter(500);

            if (maxSupportedBaudRate < baudRate) return false;

            command = Command.Pause;
            while (status != Status.Busy) ;

            int prevBaudRate = serialPort.BaudRate;
            try
            {
                serialPort.BaudRate = baudRate;
            }
            catch (System.IO.IOException)
            {
                command = Command.Cancel;
                while (!IsIdle) ;
                return false;
            }
            serialPort.BaudRate = prevBaudRate;


            command = Command.Cancel;
            while (!IsIdle) ;

            return true;
        }

        public int GetMaxBaudRate()
        {
            if (CheckBaudRate(230400)) return 230400;
            if (CheckBaudRate(115200)) return 115200;
            if (CheckBaudRate(57600)) return 57600;
            return 9600;
        }

        public bool ChangeBaudRate(int baudRate)
        {
            bool result = false;

            char chBaudRate = 'r';
            if (baudRate == 230400)
            {
                if (baudRate == serialPort.BaudRate) return true;
                chBaudRate = (char)Command.PrepareBaudrate230400;
                if (baudRate <= maxSupportedBaudRate && RequestCommand((Command)chBaudRate)) result = true; else baudRate = 115200;
            }
            if (baudRate == 115200)
            {
                if (baudRate == serialPort.BaudRate) return true;
                chBaudRate = (char)Command.PrepareBaudrate115200;
                if (baudRate <= maxSupportedBaudRate && RequestCommand((Command)chBaudRate)) result = true; else baudRate = 57600;
            }
            if (baudRate == 57600)
            {
                if (baudRate == serialPort.BaudRate) return true;
                chBaudRate = (char)Command.PrepareBaudrate57600;
                if (baudRate <= maxSupportedBaudRate && RequestCommand((Command)chBaudRate)) result = true; else baudRate = 9600;
            }
            if (baudRate == 9600) {
                if (baudRate == serialPort.BaudRate) return true;
                chBaudRate = (char)Command.PrepareBaudrate9600;
                if (baudRate <= maxSupportedBaudRate && RequestCommand((Command)chBaudRate)) result = true;
            }
            if (!result) return false;

            if (!RequestCommand(Command.ChangeBaudrate, chBaudRate.ToString()))
            {
                return false;
            }

            command = Command.Pause;
            while (status != Status.Busy) ;

            serialPort.BaudRate = baudRate;

            command = Command.Cancel;
            while (!IsIdle) ;

            Thread.Sleep(200);

            return true;
        }

        public bool NewBlockSequence()
        {
            return RequestCommand(Command.CreateNewSession);
        }

        public bool UpdateAppInfo()
        {
            return RequestCommand(Command.UpdateAppInfo);
        }

        public bool RunApp()
        {
            if (!RequestCommand(Command.PrepareToRunApp)) return false;
            if (!RequestCommand(Command.RunApp)) return false;

            return true;
        }

        string GetBase64BlockWithChecksum(byte[] data, int offset, int size)
        {
            byte[] b = new byte[4 + size];
            uint chkSum = Checksum.Compute(data, offset, size);
            Array.Copy(BitConverter.GetBytes(chkSum), b, 4);
            if (!BitConverter.IsLittleEndian) Array.Reverse(b, 0, 4);
            Array.Copy(data, offset, b, 4, size);
            return GetBase64Block(b, 0, b.Length);
        }

        string GetBase64Block(byte[] data, int offset, int size)
        {
            byte[] b = new byte[size];
            Array.Copy(data, offset, b, 0, size);

            int base64Size = ((size + 2) / 3) * 4;
            char[] base64 = new char[base64Size];
            int i, j;
            for (i = j = 0; j < base64Size; i += 3, j += 4)
            {
                byte data0 = b[i];
                byte data1 = (i + 1 < size) ? b[i + 1] : (byte)0;
                byte data2 = (i + 2 < size) ? b[i + 2] : (byte)0;
                base64[j] = (char)(0x20 + (data0 >> 2));
                base64[j + 1] = (char)(0x20 + (((data0 << 4) | (data1 >> 4)) & 0x3f));
                base64[j + 2] = (char)(0x20 + (((data1 << 2) | (data2 >> 6)) & 0x3f));
                base64[j + 3] = (char)(0x20 + (data2 & 0x3f));
                if (i + 1 >= size) base64[j + 2] = '\x60';
                if (i + 2 >= size) base64[j + 3] = '\x60';
            }

            return new string(base64);
        }

        public bool WriteBlockCommand(Command cmd, byte[] data, int offset, int size, int no, int timeoutT0 = 0, int timeoutT1 = 0)
        // A block write command is separated into three steps: <PrepareCmd> <Base64Block> <Cmd>
        {
            if (timeoutT0 == 0) timeoutT0 = 5000;
            if (timeoutT1 == 0) timeoutT1 = 1000;
            int timeout = (no == 0) ? timeoutT0 : timeoutT1;
            TimeoutCounter timeoutCounter = new TimeoutCounter(timeout);

            bool writeAccess = (cmd == Command.WriteObfuscatedBlock || cmd == Command.WriteNonObfuscatedBlock);
            bool chkRequired = (cmd == Command.CompareNonObfuscatedBlock || cmd == Command.WriteNonObfuscatedBlock);
            char prepareBlockCmd = writeAccess ? (char)Command.PrepareBlockModeForWrite : (char)Command.PrepareBlockModeForComparison;

            for (int k = 0; k < 3; k++)
            {
                if (no == 0)
                {
                    if (!RequestCommand((Command)prepareBlockCmd)) continue;
                }

                if (!chkRequired) 
                {
                    serialPort.Write(GetBase64Block(data, offset, size));
                }
                else 
                {
                    serialPort.Write(GetBase64BlockWithChecksum(data, offset, size));
                }
                while (serialPort.BytesToWrite != 0) ;

                command = cmd;
                timeoutCounter.Start();
                while (command != Command.None) if (timeoutCounter.Signaled) { command = Command.Cancel; return false; }
                if (resultString == null) return false;
                if (resultString != "r") {
                    if (!RequestCommand(Command.GetBlockCounter)) return false;
                    uint u;
                    if (uint.TryParse(resultString.Substring(1), NumberStyles.HexNumber, CultureInfo.InvariantCulture, out u))
                    {
                        if (u == no) {
                            if (RequestCommand((Command)prepareBlockCmd)) continue;
                        }
                        if (u == no + 1)
                        {
                            if (RequestCommand((Command)prepareBlockCmd)) return true;
                        }
                    }
                    return false;
                }
                return true;
            }
            return false;
        }
    }
}
