﻿/*----------------------------------------------------------------------------*/
/* Firmware Obfuscation V1.0.4                                                */
/* Written in 2019 by Rudy Tellert Elektronik                                 */
/*                                                                            */
/* To the extent possible under law, the author has dedicated all copyright   */
/* and related and neighboring rights to the software "firmware obfuscation"  */
/* and its documentation to the public domain. This software and its          */
/* documentation are distributed without any warranty.                        */
/*                                                                            */
/* See also                                                                   */
/* http://creativecommons.org/publicdomain/zero/1.0/                          */
/* ---------------------------------------------------------------------------*/

using System;

namespace Tellert
{
    public struct UInt128
    {
        public ulong Low;
        public ulong High;

        public static UInt128 MinValue = new UInt128(0);

        public UInt128(UInt128 value)
        {
            Low = value.Low;
            High = value.High;
        }

        public UInt128(ulong high, ulong low)
        {
            this.Low = low;
            this.High = high;
        }

        public static implicit operator UInt128(ulong value)
        {
            return new UInt128(0, value);
        }

        public byte[] ToByteArray()
        {
            byte[] arr = new byte[16];

            if (BitConverter.IsLittleEndian)
            {
                Array.Copy(BitConverter.GetBytes(Low), 0, arr, 0, 8);
                Array.Copy(BitConverter.GetBytes(High), 0, arr, 8, 8);
            }
            else
            {
                Array.Copy(BitConverter.GetBytes(High), 0, arr, 0, 8);
                Array.Copy(BitConverter.GetBytes(Low), 0, arr, 8, 8);
            }

            return arr;
        }

        public void FromByteArray(byte[] value, int startIndex)
        {
            if (BitConverter.IsLittleEndian)
            {
                Low = BitConverter.ToUInt64(value, startIndex + 0);
                High = BitConverter.ToUInt64(value, startIndex + 8);
            }
            else
            {
                High = BitConverter.ToUInt64(value, startIndex + 0);
                Low = BitConverter.ToUInt64(value, startIndex + 8);
            }
        }

        public static UInt128 operator - (UInt128 value)
        {
            UInt128 tmp = ~value;
            tmp++;
            return tmp;
        }

        public static UInt128 operator - (UInt128 left, UInt128 right)
        {
            return left + (-right);
        }

        public static UInt128 operator + (UInt128 left, byte right)
        {
            UInt128 tmp = new UInt128(left);

            ulong prev = tmp.Low;
            tmp.Low += right;
            if (tmp.Low < prev) {
                tmp.High++;
            }

            return tmp;
        }

        public static UInt128 operator + (UInt128 left, int right)
        {
            UInt128 tmp = new UInt128(left);

            if (right < 0) throw new ArgumentException();

            ulong prev = tmp.Low;
            tmp.Low += (uint)right;
            if (tmp.Low < prev)
            {
                tmp.High++;
            }

            return tmp;
        }

        public static UInt128 operator + (UInt128 left, UInt128 right)
        {
            UInt128 tmp = new UInt128(left);

            ulong prev = tmp.Low;
            tmp.Low += right.Low;
            if (tmp.Low < prev)
            {
                tmp.High++;
            }
            tmp.High += right.High;

            return tmp;
        }

        public static UInt128 operator ++(UInt128 value)
        {
            UInt128 tmp = new UInt128(value);

            if (++tmp.Low == 0) ++tmp.High;

            return tmp;
        }

        public static UInt128 operator --(UInt128 value)
        {
            UInt128 tmp = new UInt128(value);

            if (--tmp.Low == 0xffffffffffffffff) --tmp.High;

            return tmp;
        }

        public static UInt128 operator << (UInt128 left, int right)
        {
            UInt128 tmp;

            if (right == 0) return left;
            if (right < 0) return left >> -right;

            if (right < 64)
            {
                tmp.Low = left.Low << right;
                tmp.High = left.High << right;
                tmp.High |= left.Low >> (64 - right);
            }
            else if (right <= 128) {
                tmp.Low = 0;
                tmp.High = left.Low << (right - 64);
            }
            else throw new ArgumentException();

            return tmp;
        }

        public static UInt128 operator >> (UInt128 left, int right)
        {
            UInt128 tmp;

            if (right == 0) return left;
            if (right < 0) return left << -right;

            if (right < 64)
            {
                tmp.High = left.High >> right;
                tmp.Low = left.Low >> right;
                tmp.Low |= left.High << (64 - right);
            }
            else if (right <= 128)
            {
                tmp.High = 0;
                tmp.Low = left.High >> (right - 64);
            }
            else throw new ArgumentException();

            return tmp;
        }

        public static UInt128 operator ^ (UInt128 left, UInt128 right)
        {
            UInt128 tmp = new UInt128(left);

            tmp.Low ^= right.Low;
            tmp.High ^= right.High;

            return tmp;
        }

        public static UInt128 operator & (UInt128 left, UInt128 right)
        {
            UInt128 tmp = new UInt128(left);

            tmp.Low &= right.Low;
            tmp.High &= right.High;

            return tmp;
        }

        public static UInt128 operator | (UInt128 left, UInt128 right)
        {
            UInt128 tmp = new UInt128(left);

            tmp.Low |= right.Low;
            tmp.High |= right.High;

            return tmp;
        }

        public static UInt128 operator ~ (UInt128 value)
        {
            UInt128 tmp = new UInt128(value);

            tmp.Low = ~tmp.Low;
            tmp.High = ~tmp.High;

            return tmp;
        }

        public static bool operator == (UInt128 left, UInt128 right)
        {
            return (left.Low == right.Low) && (left.High == right.High);
        }

        public static bool operator !=(UInt128 left, UInt128 right)
        {
            return (left.Low != right.Low) || (left.High != right.High);
        }

        public override bool Equals(object value)
        {
            UInt128 u = (UInt128)value;
            return Low.Equals(u.Low) && High.Equals(u.High);
        }

        public override int GetHashCode()
        {
            return Low.GetHashCode() ^ High.GetHashCode();
        }
    }

    public static partial class BitConverterEx
    {
        public static byte[] GetBytes(UInt128 value)
        {
            return value.ToByteArray();
        }

        public static UInt128 ToUInt128(byte[] value, int startIndex)
        {
            UInt128 u = new UInt128();

            u.FromByteArray(value, startIndex);

            return u;
        }

        public static byte[] GetBytes(ulong value)
        {
            return BitConverter.GetBytes(value);
        }

        public static ulong ToUInt64(byte[] value, int startIndex)
        {
            return BitConverter.ToUInt64(value, startIndex);
        }

        public static byte[] GetBytes(uint value)
        {
            return BitConverter.GetBytes(value);
        }

        public static uint ToUInt32(byte[] value, int startIndex)
        {
            return BitConverter.ToUInt32(value, startIndex);
        }
    }
}