﻿/*----------------------------------------------------------------------------*/
/* Firmware Obfuscation V1.0.4                                                */
/* Written in 2019 by Rudy Tellert Elektronik                                 */
/*                                                                            */
/* To the extent possible under law, the author has dedicated all copyright   */
/* and related and neighboring rights to the software "firmware obfuscation"  */
/* and its documentation to the public domain. This software and its          */
/* documentation are distributed without any warranty.                        */
/*                                                                            */
/* See also                                                                   */
/* http://creativecommons.org/publicdomain/zero/1.0/                          */
/* ---------------------------------------------------------------------------*/

using System;
using System.Collections.Generic;
using System.IO;

namespace Tellert.Format
{
    class HexFile
    {

        #region Data
        // essential data
        List<Item> items = new List<Item>();
        // additional S-Record data
        List<Item> header = new List<Item>();
        List<Item> end = new List<Item>();
        // additional Intel hex file data
        List<StartItem> startAddress = new List<StartItem>();
        // additional S-Record format
        int maxDataLength;
        RecType endType;
        // FileType: After Intel, only Intel sub types are allowed
        public enum FileType { Undefined, None, Motorola, Intel, IntelSegment, IntelLinear };
        FileType fileType = FileType.None;
        uint currAddress = 0;
        uint entryPoint = 0;

        // static data
        static byte[] EmptyByteArray = new byte[0];
        static RecType[] recTypeTab = { RecType.S0, RecType.S1, RecType.S2, RecType.S3, RecType.Unknown, RecType.S5,
           RecType.Unknown, RecType.S7, RecType.S8, RecType.S9 };
        static RecType[] recTypeTabIntel = { RecType.I0, RecType.I1, RecType.I2, RecType.I3, RecType.I4, RecType.I5 };
        const int MinDataLength = 0x10;
        const int MaxDataLength = 0x20;

        enum RecType
        {
            Unknown = 0,
            S0 = TypeHeader | Size2,
            S1 = TypeData | Size2, S2 = TypeData | Size3, S3 = TypeData | Size4,
            S5 = TypeRecCount | Size2,
            S7 = TypeEnd | Size4, S8 = TypeEnd | Size3, S9 = TypeEnd | Size2,
            
            I0 = TypeData | Size2 | Intel,
            I1 = TypeEnd | Size2 | Intel,
            I2 = TypeIntelAddr | Size2 | Intel,
            I3 = TypeIntelStart | Size2 | Intel,
            I4 = TypeIntelAddr | Size4 | Intel,
            I5 = TypeIntelStart | Size4 | Intel,

            // :2 addressSize = { 2, 3, 4, undefined }
            Size2 = 0, Size3 = 1, Size4 = 2,
            SizeMask = Size2 | Size3 | Size4,
            // :3 type = { unknown, header, data, nRec, end }
            TypeHeader = 4, TypeData = 8, TypeRecCount = 12, TypeEnd = 16, TypeIntelAddr = 32, TypeIntelStart = 64,
            TypeMask = TypeHeader | TypeData | TypeRecCount | TypeEnd | TypeIntelAddr | TypeIntelStart,
            // Intel
            Intel = 128,
        }
        #endregion

        #region class Item / CompareItems / StartItem
        class Item
        {
            public uint Address;
            public byte[] Data = HexFile.EmptyByteArray;

            public uint LastAddress
            {
                get 
                { 
                    if (Data.Length == 0) throw new InvalidDataException();
                    return Address + (uint)Data.Length - 1; 
                }
            }

            public Item(uint address, byte[] data)
            {
                Address = address;
                Data = data;
            }

            public static bool IsRangeValid(uint address, int count)
            {
                if (address == 0) return true;
                uint maxCount = ~address + 1; // = 0x100000000 - address (for address > 0)
                return maxCount >= (uint)count;
            }

            public static void CorrectRange(uint address, ref int count)
            {
                if (address != 0)
                {
                    uint maxCount = ~address + 1; // = 0x100000000 - address (for address > 0)
                    if (maxCount < (uint)count) count = (int)maxCount;
                }
            }

            public bool Contains(uint address)
            {
                if (Data.Length == 0) return false;
                return address >= Address && address <= LastAddress;
            }

            public bool ContainsAny(uint address, int count)
            {
                if (Data.Length == 0 || count == 0) return false;
                CorrectRange(address, ref count);
                uint maxAddress = Math.Max(Address, address);
                uint minLastAddress = Math.Min(LastAddress, address + (uint)count - 1);
                if (maxAddress > minLastAddress) return false;
                return maxAddress >= Address && minLastAddress <= LastAddress;
            }

            public bool ContainsAny(Item item)
            {
                return ContainsAny(item.Address, item.Data.Length);
            }

            static System.Text.ASCIIEncoding enc = null;
            public override string ToString()
            {
                if (Data.Length == 0) return string.Format("{0:x8}", Address);

                byte[] data = new byte[Math.Min(Data.Length, 0x80)];
                for (int i = 0; i < data.Length; i++) {
                    data[i] = (Data[i] >= (byte)32 && Data[i] < (byte)128 && Data[i] != (byte)34) ? 
                        Data[i] : (byte)46;
                }

                if (enc == null) enc = new System.Text.ASCIIEncoding();
                return string.Format("{0:x8}-{1:x8}", Address, LastAddress) + 
                    ": \"" + enc.GetString(data) + (data.Length < Data.Length ? "..." : "") + "\"";
            }
        }
        static int CompareItems(Item x, Item y)
        {
            int result = x.Address.CompareTo(y.Address);
            if (result == 0) result = x.Data.Length.CompareTo(y.Data.Length);
            if (result == 0) for (int i = 0; i < x.Data.Length; i++) if ((result = x.Data[i].CompareTo(y.Data[i])) != 0) break;
            return result;
        }

        class StartItem
        {
            public RecType RecType;
            public uint StartAddress;

            public StartItem(RecType recType, uint startAddress)
            {
                RecType = recType;
                StartAddress = startAddress;
            }
        }
        #endregion

        #region Clear / Normalize / Compress / Expand
        public void Clear()
        {
            items.Clear();
            header.Clear();
            end.Clear();
            endType = RecType.Unknown;
            maxDataLength = 0;
            currAddress = 0;
            entryPoint = 0;
            fileType = FileType.None;
        }

        bool Normalize()
        {
            items.Sort(CompareItems);

            // check for multiple item definitions
            for (int i = items.Count - 1; --i >= 0; )
            {
                if (items[i].ContainsAny(items[i + 1])) return false;
            }

            Compress();

            return true;
        }

        public void Compress()
        {
            int[] mergeableItems = new int[items.Count];
            int i, k, length;

            for (i = 0; i < items.Count; )
            {
                length = items[i].Data.Length;
                for (k = i + 1; k < items.Count; k++)
                {
                    if (items[k-1].LastAddress + 1 != items[k].Address || 
                      (uint)length + (uint)items[k].Data.Length > (uint)int.MaxValue) break;
                    length += items[k].Data.Length;
                }
                int count = k - i;
                mergeableItems[i] = count;
                i = k;
            }

            for (i = items.Count - 1; --i >= 0; )
            {
                if (mergeableItems[i] > 1) {
                    // merge
                    length = 0;
                    for (k = mergeableItems[i]; --k >= 0; ) length += items[i+k].Data.Length;
                    byte[] mergedData = new byte[length];
                    length = 0;
                    for (k = 0; k < mergeableItems[i]; k++)
                    {
                        Array.Copy(items[i + k].Data, 0, mergedData, length, items[i + k].Data.Length);
                        length += items[i + k].Data.Length;
                    }
                    items[i].Data = mergedData;
                    items.RemoveRange(i + 1, mergeableItems[i]-1);
                }
            }
        }

        public bool IsExpansionRequired(int maxCount = 0)
        {
            if (maxCount == 0) maxCount = maxDataLength;
            maxCount = Math.Min(Math.Max(maxCount, MinDataLength), MaxDataLength);

            for (int i = items.Count; --i >= 0;)
            {
                if (items[i].Data.Length <= maxCount) continue;
                return true;
            }

            return false;
        }

        public void Expand(int maxCount = 0)
        {
            if (maxCount == 0) maxCount = maxDataLength;
            maxCount = Math.Min(Math.Max(maxCount, MinDataLength), MaxDataLength);

            // split overlong items
            for (int i = items.Count; --i >= 0; )
            {
                if (items[i].Data.Length <= maxCount) continue;

                int additionalItems = items[i].Data.Length / maxCount - 1;
                if (items[i].Data.Length % maxCount != 0) additionalItems++;

                Item[] newItems = new Item[additionalItems];
                uint address = items[i].Address + (uint)maxCount;
                int remainingBytes = items[i].Data.Length - maxCount;
                int idx = maxCount;
                for (int j = 0; j < additionalItems; j++)
                {
                    int length = (remainingBytes > maxCount) ? maxCount : remainingBytes;

                    byte[] bytes = new byte[length];
                    Array.Copy(items[i].Data, idx, bytes, 0, length);
                    newItems[j] = new Item(address, bytes);

                    idx += length;
                    address += (uint)length;
                    remainingBytes -= length;
                }
                byte[] tmpBytes = new byte[maxCount];
                Array.Copy(items[i].Data, tmpBytes, maxCount);
                items[i].Data = tmpBytes;
                items.InsertRange(i + 1, newItems);
            }

        }

        public bool IsExpansionIntelRequired()
        {
            for (int i = items.Count; --i > 0;)
            {
                if (items[i].Data.Length == 0) continue;
                if (((items[i].Address & 0xffff0000) ^ ((items[i].Address + items[i].Data.Length - 1)) & 0xffff0000) == 0) continue;
                return true;
            }

            return false;
        }

        public void ExpandIntel(int maxCount = 0)
        {
            // split whenever the high word of an address changes
            for (int i = items.Count; --i >= 0;)
            {
                if (items[i].Data.Length == 0) continue;
                if (((items[i].Address & 0xffff0000) ^ ((items[i].Address + items[i].Data.Length - 1)) & 0xffff0000) == 0) continue;

                int additionalItems = 0;
                for (int k = 1; k < items[i].Data.Length; k++) {
                    if ((((items[i].Address + k - 1) & 0xffff0000) ^ ((items[i].Address+k) & 0xffff0000)) != 0) {
                        additionalItems++;
                    }
                }

                int length0 = 1;
                for (; length0 < items[i].Data.Length; length0++) {
                    if ((((items[i].Address + length0 - 1) & 0xffff0000) ^ ((items[i].Address + length0) & 0xffff0000)) != 0)
                    {
                        break;
                    }
                }

                Item[] newItems = new Item[additionalItems];
                uint nextAddress = items[i].Address + (uint)length0;
                int remainingBytes = items[i].Data.Length - length0;
                int idx = length0;
                for (int j = 0; j < additionalItems; j++)
                {
                    int length = 1;
                    for (; idx+length < items[i].Data.Length; length++) {
                        if ((((items[i].Address + idx + length - 1) & 0xffff0000) ^ ((items[i].Address + idx + length) & 0xffff0000)) != 0)
                        {
                            break;
                        }
                    }
                    if (idx + length >= items[i].Data.Length) {
                        length = items[i].Data.Length - idx;
                    }

                    byte[] bytes = new byte[length];
                    Array.Copy(items[i].Data, idx, bytes, 0, length);
                    newItems[j] = new Item(nextAddress, bytes);

                    idx += length;
                    nextAddress += (uint)length;
                    remainingBytes -= length;
                }
                byte[] tmpBytes = new byte[length0];
                Array.Copy(items[i].Data, tmpBytes, length0);
                items[i].Data = tmpBytes;
                items.InsertRange(i + 1, newItems);
            }
        }
        #endregion

        #region Read
        public bool Read(string fileName)
        {
            bool result = false;
            using (StreamReader reader = new StreamReader(fileName))
            {
                result = Read(reader);
            }

            return result;
        }

        public bool Read(StreamReader reader) 
        {
            Clear();

            string line;
            while ((line = reader.ReadLine()) != null)
            {
                line = line.Trim();
                if (line.Length == 0) continue;
                    
                var parse = ParseRecord(line);
                if (fileType == FileType.None)
                {
                    fileType = (parse.Item1 & RecType.Intel) != 0 ? FileType.Intel : FileType.Motorola;
                }
                if ((parse.Item1 & RecType.Intel) != 0)
                {
                    if (parse.Item1 == RecType.I2 || parse.Item1 == RecType.I3)
                    {
                        fileType = FileType.IntelSegment;
                    }
                    else if (parse.Item1 == RecType.I4 || parse.Item1 == RecType.I5)
                    {
                        fileType = FileType.IntelLinear;
                    }
                }
                switch (parse.Item1 & RecType.TypeMask)
                {
                    case RecType.TypeHeader:
                        header.Add(new Item(parse.Item2, parse.Item3));
                        break;
                    case RecType.TypeData:
                        if (parse.Item3.Length == 0) continue; // discard empty items
                        if (!Item.IsRangeValid(parse.Item2, parse.Item3.Length)) {
                            Clear();
                            return false;
                        }
                        items.Add(new Item(parse.Item2, parse.Item3));
                        if (parse.Item3.Length > maxDataLength) maxDataLength = parse.Item3.Length;
                        break;
                    case RecType.TypeRecCount:
                        // ignore rec count item
                        break;
                    case RecType.TypeEnd:
                        endType = parse.Item1;
                        end.Add(new Item(parse.Item2, parse.Item3));
                        if (fileType == FileType.Motorola)
                        {
                            EntryPoint = parse.Item2;
                        }
                        else
                        {
                            if (startAddress.Count == 0)
                            {
                                EntryPoint = parse.Item2;
                            }
                        }
                        if (!Normalize())
                        {
                            Clear();
                            return false;
                        }
                        return true;
                    case RecType.TypeIntelAddr:
                        break;
                    case RecType.TypeIntelStart:
                        startAddress.Add(new StartItem(parse.Item1, parse.Item2));
                        EntryPoint = parse.Item2;
                        break;
                    default:
                        Clear();
                        return false;
                }
            }
            if (!Normalize())
            {
                Clear();
                return false;
            }

            return true;
        }

        Tuple<RecType, uint, byte[]> ParseRecord(string line)
        {
            RecType recType = RecType.Unknown;
            uint address = 0;
            byte[] data = EmptyByteArray;

            if ((line.Length & 1) == 0 && line.Substring(0, 1) == "S" && line.Length >= 5 * 2)
            {
                // MOTOROLA
                byte[] bytes = new byte[line.Length / 2];
                byte checkSum = 0;
                bytes[0] = Convert.ToByte(line.Substring(1, 1), 16);
                for (int i = 1; i < bytes.Length; i++) checkSum += bytes[i] = Convert.ToByte(line.Substring(i * 2, 2), 16);
                if (checkSum != 255 || bytes[1] != bytes.Length - 2) bytes = EmptyByteArray;
                else if (bytes[0] < recTypeTab.Length) recType = recTypeTab[bytes[0]];

                if (recType != RecType.Unknown)
                {
                    switch (recType & RecType.SizeMask)
                    {
                        case RecType.Size2:
                            address = bytes[2];
                            address <<= 8; address += bytes[3];
                            data = new byte[bytes.Length - 5];
                            Array.Copy(bytes, 4, data, 0, data.Length);
                            break;
                        case RecType.Size3:
                            address = bytes[2];
                            address <<= 8; address += bytes[3];
                            address <<= 8; address += bytes[4];
                            data = new byte[bytes.Length - 6];
                            Array.Copy(bytes, 5, data, 0, data.Length);
                            break;
                        case RecType.Size4:
                            address = bytes[2];
                            address <<= 8; address += bytes[3];
                            address <<= 8; address += bytes[4];
                            address <<= 8; address += bytes[5];
                            data = new byte[bytes.Length - 7];
                            Array.Copy(bytes, 6, data, 0, data.Length);
                            break;
                    }
                }
            }
            else if ((line.Length & 1) == 1 && line.Substring(0, 1) == ":" && line.Length >= 4 * 2 + 1) 
            {
                // INTEL
                byte[] bytes = new byte[line.Length / 2];
                byte checkSum = 0;
                for (int i = 0; i < bytes.Length; i++) checkSum += bytes[i] = Convert.ToByte(line.Substring(i*2+1, 2), 16);
                if (checkSum != 0 || bytes[0] != bytes.Length-5) bytes = EmptyByteArray;
                else if (bytes[3] < recTypeTabIntel.Length) recType = recTypeTabIntel[bytes[3]];

                if (recType != RecType.Unknown)
                {
                    switch (recType)
                    {
                        case RecType.I0:
                            address = bytes[1];
                            address <<= 8; address += bytes[2];
                            address += currAddress;
                            data = new byte[bytes.Length - 5];
                            Array.Copy(bytes, 4, data, 0, data.Length);
                            break;

                        case RecType.I1:
                            address = bytes[1];
                            address <<= 8; address += bytes[2];
                            break;

                        case RecType.I2:
                            currAddress = bytes[4];
                            currAddress <<= 8; currAddress += bytes[5];
                            currAddress <<= 4;
                            break;

                        case RecType.I3:
                            address = bytes[4];
                            address <<= 8; address += bytes[5];
                            address <<= 8; address += bytes[6];
                            address <<= 8; address += bytes[7];
                            break;

                        case RecType.I4:
                            currAddress = bytes[4];
                            currAddress <<= 8; currAddress += bytes[5];
                            currAddress <<= 16;
                            break;

                        case RecType.I5:
                            address = bytes[4];
                            address <<= 8; address += bytes[5];
                            address <<= 8; address += bytes[6];
                            address <<= 8; address += bytes[7];
                            break;
                    }
                }
            }

            return Tuple.Create(recType, address, data);
        }
        #endregion

        #region Write
        public bool Write(string fileName, FileType type = FileType.Undefined)
        {
            bool result = false;
            StreamWriter writer = new StreamWriter(fileName);
            try
            {
                result = Write(writer, type);
            }
            finally
            {
                if (writer != null)
                {
                    writer.Dispose();
                    if (!result) File.Delete(fileName);
                }
            }

            return result;
        }

        public bool Write(StreamWriter writer, FileType type = FileType.Undefined)
        {
            RecType dataType;

            if (type == FileType.Undefined) type = fileType;
            if (type == FileType.None) type = (startAddress.Count != 0) ? FileType.Intel : FileType.Motorola;

            if (type >= FileType.Intel) return WriteIntel(writer, type);

            if (IsExpansionRequired(MaxDataLength)) Expand();

            uint maxAddress = (items.Count != 0) ? items[items.Count-1].LastAddress : 0;
            if (maxAddress < EntryPoint) maxAddress = EntryPoint;
            if (maxAddress <= 0xffff)
            {
                dataType = RecType.S1;
            }
            else if (maxAddress <= 0xffffff)
            {
                dataType = RecType.S2;
            }
            else
            {
                dataType = RecType.S3;
            }

            foreach (Item item in header) WriteItemStringMotorola(writer, RecType.S0, item);
            foreach (Item item in items) WriteItemStringMotorola(writer, dataType, item);
            //foreach (Item item in end) WriteItemStringMotorola(writer, endType, item);
            if (end.Count >= 0) {
                switch (dataType)
                {
                    case RecType.S1:
                        WriteItemStringMotorola(writer, RecType.S9, new Item(EntryPoint, EmptyByteArray));
                        break;
                    case RecType.S2:
                        WriteItemStringMotorola(writer, RecType.S8, new Item(EntryPoint, EmptyByteArray));
                        break;
                    case RecType.S3:
                        WriteItemStringMotorola(writer, RecType.S7, new Item(EntryPoint, EmptyByteArray));
                        break;
                }
            }

            return true;
        }

        void WriteItemStringMotorola(StreamWriter writer, RecType recType, Item item)
        {
            string result = string.Empty;

            byte[] bytes = EmptyByteArray;
            switch (recType & RecType.SizeMask)
            {
                case RecType.Size2:
                    // T-N-A-A-D-...-C
                    bytes = new byte[item.Data.Length + 5];
                    bytes[2] = (byte)(item.Address >> 8);
                    bytes[3] = (byte)item.Address;
                    Array.Copy(item.Data, 0, bytes, 4, item.Data.Length);
                    break;
                case RecType.Size3:
                    // T-N-A-A-A-D-...-C
                    bytes = new byte[item.Data.Length + 6];
                    bytes[2] = (byte)(item.Address >> 16);
                    bytes[3] = (byte)(item.Address >> 8);
                    bytes[4] = (byte)item.Address;
                    Array.Copy(item.Data, 0, bytes, 5, item.Data.Length);
                    break;
                case RecType.Size4:
                    // T-N-A-A-A-A-D-...-C
                    bytes = new byte[item.Data.Length + 7];
                    bytes[2] = (byte)(item.Address >> 24);
                    bytes[3] = (byte)(item.Address >> 16);
                    bytes[4] = (byte)(item.Address >> 8);
                    bytes[5] = (byte)item.Address;
                    Array.Copy(item.Data, 0, bytes, 6, item.Data.Length);
                    break;
                default: return;
            }
            bytes[1] = (byte)(bytes.Length - 2);

            byte checkSum = bytes[1];
            for (int i = 2; i < bytes.Length - 1; i++) checkSum += bytes[i];
            bytes[bytes.Length - 1] = (byte)~checkSum;

            switch (recType)
            {
                case RecType.S0: bytes[0] = 0; break;
                case RecType.S1: bytes[0] = 1; break;
                case RecType.S2: bytes[0] = 2; break;
                case RecType.S3: bytes[0] = 3; break;
                case RecType.S5: bytes[0] = 5; break;
                case RecType.S7: bytes[0] = 7; break;
                case RecType.S8: bytes[0] = 8; break;
                case RecType.S9: bytes[0] = 9; break;
                default: return;
            }

            result = string.Format("S{0:X1}", bytes[0]);
            for (int i = 1; i < bytes.Length; i++)
            {
                result += string.Format("{0:X2}", bytes[i]);
            }

            writer.WriteLine(result);
        }

        bool WriteIntel(StreamWriter writer, FileType type)
        {
            RecType dataType = RecType.Unknown;

            currAddress = 0;
            if (IsExpansionIntelRequired()) ExpandIntel();
            if (IsExpansionRequired(MaxDataLength)) Expand();

            switch (type)
            {
                case FileType.IntelSegment:
                    dataType = RecType.I2;
                    break;
                case FileType.IntelLinear:
                    dataType = RecType.I4;
                    break;
            }

            if (dataType == RecType.Unknown)
            {
                uint maxAddress = (items.Count != 0) ? items[items.Count - 1].LastAddress : 0;
                if (maxAddress < EntryPoint) maxAddress = EntryPoint;
                if (maxAddress <= 0xffff)
                {
                    dataType = RecType.I0;
                }
                else if (maxAddress <= 0xfffff)
                {
                    dataType = RecType.I2;
                }
                else
                {
                    dataType = RecType.I4;
                }
            }
            foreach (Item item in items) WriteItemString(writer, dataType, item);
            foreach (StartItem item in startAddress) WriteItemStringStart(writer, item);
            if (startAddress.Count == 0)
            {
                switch (dataType)
                {
                    case RecType.I2:
                        if (EntryPoint != 0) WriteItemStringStart(writer, new StartItem(RecType.I3, EntryPoint));
                        break;
                    case RecType.I4:
                        if (EntryPoint != 0) WriteItemStringStart(writer, new StartItem(RecType.I5, EntryPoint));
                        break;
                }
            }
            //foreach (Item item in end) WriteItemString(writer, endType, item);
            if (end.Count >= 0)
            {
                switch (dataType)
                {
                    case RecType.I0:
                        WriteItemString(writer, RecType.I1, new Item(EntryPoint, EmptyByteArray));
                        break;
                    case RecType.I2:
                        WriteItemString(writer, RecType.I1, new Item(0, EmptyByteArray));
                        break;
                    case RecType.I4:
                        WriteItemString(writer, RecType.I1, new Item(0, EmptyByteArray));
                        break;
                }
            }

            return true;
        }

        void WriteItemString(StreamWriter writer, RecType recType, Item item)
        {
            string result = string.Empty;

            byte[] bytes = new byte[(1 + 2 + 1 + 1) + item.Data.Length];
            bytes[0] = (byte)item.Data.Length;
            bytes[1] = (byte)((item.Address >> 8) & 0xff);
            bytes[2] = (byte)((item.Address) & 0xff);
            Array.Copy(item.Data, 0, bytes, 4, item.Data.Length);
            switch (recType)
            {
                case RecType.I0:
                    break;
                case RecType.I1:
                    bytes[3] = 1;
                    break;
                case RecType.I2:
                    if (((item.Address ^ currAddress) & 0xffff0000) != 0)
                    {
                        currAddress = item.Address;
                        byte[] data = new byte[2];
                        ushort addr = (ushort)(((currAddress & 0xffff0000) >> 4) & 0xffff);
                        data[0] = (byte)(addr >> 8);
                        data[1] = (byte)(addr & 0xff);
                        WriteItemString(writer, RecType.I3, new Item(0, data));
                    }
                    break;
                case RecType.I3:
                    bytes[3] = 2;
                    break;
                case RecType.I4:
                    if (((item.Address ^ currAddress) & 0xffff0000) != 0)
                    {
                        currAddress = item.Address;
                        byte[] data = new byte[2];
                        ushort addr = (ushort)(((currAddress & 0xffff0000) >> 16) & 0xffff);
                        data[0] = (byte)(addr >> 8);
                        data[1] = (byte)(addr & 0xff);
                        WriteItemString(writer, RecType.I5, new Item(0, data));
                    }
                    break;
                case RecType.I5:
                    bytes[3] = 4;
                    break;
                default: return;
            }

            byte checkSum = 0;
            for (int i = 0; i < bytes.Length - 1; i++) checkSum += bytes[i];
            bytes[bytes.Length - 1] = (byte)((checkSum ^ 0xff) + 1);

            result = ":";
            for (int i = 0; i < bytes.Length; i++)
            {
                result += string.Format("{0:X2}", bytes[i]);
            }

            writer.WriteLine(result);
        }

        void WriteItemStringStart(StreamWriter writer, StartItem item)
        {
            string result = string.Empty;

            byte[] bytes = new byte[1 + 2 + 1 + 4 + 1];
            switch (item.RecType)
            {
                case RecType.I3:
                    bytes[3] = 3;
                    break;
                case RecType.I5:
                    bytes[3] = 5;
                    break;
                default: return;
            }
            bytes[0] = 4;
            bytes[4] = (byte)((item.StartAddress >> 24) & 0xff);
            bytes[5] = (byte)((item.StartAddress >> 16) & 0xff);
            bytes[6] = (byte)((item.StartAddress >>  8) & 0xff);
            bytes[7] = (byte)((item.StartAddress      ) & 0xff);

            byte checkSum = 0;
            for (int i = 0; i < bytes.Length - 1; i++) checkSum += bytes[i];
            bytes[bytes.Length - 1] = (byte)((checkSum ^ 0xff) + 1);

            result = ":";
            for (int i = 0; i < bytes.Length; i++)
            {
                result += string.Format("{0:X2}", bytes[i]);
            }

            writer.WriteLine(result);
        }
        #endregion

        #region IsDataAvailable / GetBytes / SetBytes / Remove / RemoveUntil / RemoveFrom / RemoveFillByteArea
        public bool IsDataAvailable(uint address, int count)
        {
            if (count < 0) return false;
            if (!Item.IsRangeValid(address, count)) return false;

            byte[] result = new byte[count];
            if (items.Count == 0) return false;

            int idx = 0;
            for (int i = 0; i < count; i++)
            {
                for (; idx < items.Count - 1; idx++)
                {
                    if (items[idx + 1].Address > address) break;
                }

                if (items[idx].Contains(address))
                {
                    return true;
                }

                address++;
            }

            return false;
        }

        public byte[] GetBytes(uint address, int count, byte fillByte = 0xff)
        {
            if (count < 0) return null;
            if (!Item.IsRangeValid(address, count)) return null;

            byte[] result = new byte[count];
            if (!GetBytes(address, result, fillByte)) return null;

            return result;
        }

        public bool GetBytes(uint address, byte[] data, byte fillByte = 0xff)
        {
            if (!Item.IsRangeValid(address, data.Length)) return false;

            int idx = 0;
            for (int i = 0; i < data.Length; i++)
            {
                for (; idx < items.Count - 1; idx++)
                {
                    if (items[idx + 1].Address > address) break;
                }

                if (items[idx].Contains(address))
                {
                    data[i] = items[idx].Data[address - items[idx].Address];
                }
                else
                {
                    data[i] = fillByte;
                }

                address++;
            }

            return true;
        }

        public bool SetBytes(uint address, byte[] data)
        {
            if (!Item.IsRangeValid(address, data.Length)) return false;
            if (items.Count == 0)
            {
                items.Add(new Item(address, data));
                return true;
            }

            int idx = 0;
            bool compressable = false;
            for (int i = 0; i < data.Length; i++)
            {
                for (; idx < items.Count - 1; idx++)
                {
                    if (items[idx + 1].Address > address) break;
                }
                if (items[idx].Contains(address))
                {
                    items[idx].Data[address - items[idx].Address] = data[i];
                } else {
                    byte[] b = new byte[1];
                    b[0] = data[i];
                    if (items[idx].Address < address) idx++;
                    items.Insert(idx, new Item(address, b));
                    compressable = true;
                }

                address++;
            }
            if (compressable) Compress();

            return true; 
        }

        public bool Remove(uint address, int count)
        {
            if (count < 0) return false;
            Item.CorrectRange(address, ref count);

            if (count == 0) return true;
            uint lastAddress = address + (uint)count - 1;
            for (int i = items.Count; --i >= 0; ) {
                if (!items[i].ContainsAny(address, count)) continue;

                if (address <= items[i].Address)
                {
                    int removeLength = (int)(lastAddress + 1 - items[i].Address);
                    if (removeLength >= items[i].Data.Length)
                    {
                        // remove entire item
                        items.RemoveAt(i);
                    }
                    else
                    {
                        // trim first part
                        byte[] data = new byte[items[i].Data.Length - removeLength];
                        Array.Copy(items[i].Data, removeLength, data, 0, data.Length);
                        items[i].Data = data;
                        items[i].Address += (uint)removeLength;
                    }
                } else {
                    if (lastAddress >= items[i].LastAddress)
                    {
                        // trim last part
                        int newLength = (int)(address - items[i].Address);
                        byte[] data = new byte[newLength];
                        Array.Copy(items[i].Data, data, data.Length);
                        items[i].Data = data;
                    }
                    else
                    {
                        // remove inner part
                        int newEndLength = (int)(items[i].LastAddress - lastAddress);
                        byte[] data = new byte[newEndLength];
                        Array.Copy(items[i].Data, items[i].Data.Length - data.Length, data, 0, data.Length);
                        Item item = new Item(items[i].Address + (uint)items[i].Data.Length - (uint)data.Length, data);
                        items.Insert(i + 1, item);
                        int newLength = (int)(address - items[i].Address);
                        data = new byte[newLength];
                        Array.Copy(items[i].Data, data, data.Length);
                        items[i].Data = data;
                    }
                }
            }

            return true;
        }

        public bool RemoveUntil(uint address)
        {
            uint totalRange = address;
            address = 0;

            while (totalRange != 0) {
                int range = (totalRange <= int.MaxValue) ? (int)totalRange : int.MaxValue;
                if (!Remove(address, range)) return false;
                totalRange -= (uint)range;
                address += (uint)range;
            }

            return true;
        }

        public bool RemoveFrom(uint address)
        {
            if (address == 0) {
                items.Clear();
                return true;
            }

            uint totalRange = ~address + 1; // = 0x100000000 - address (for address > 0)
            while (totalRange != 0)
            {
                int range = (totalRange <= int.MaxValue) ? (int)totalRange : int.MaxValue;
                if (!Remove(address, range)) return false;
                totalRange -= (uint)range;
                address += (uint)range;
            }

            return true;
        }

        bool RemoveFillByteArea(byte fillByte, int count)
        {
            foreach (Item item in items)
            {
                if (item.Data.Length >= count)
                {
                    int c = 0;
                    for (int i = 0; i < item.Data.Length; i++)
                    {
                        if (item.Data[i] != fillByte) c = 0;
                        else
                        {
                            c++;
                            if (c >= count)
                            {
                                if (i >= item.Data.Length - 1 || item.Data[i + 1] != fillByte)
                                {
                                    Remove(item.Address + (uint)i - (uint)c + 1, c);
                                    return true;
                                }
                            }
                        }
                    }
                }
            }
            return false;
        }

        public void RemoveFillByteAreas(byte fillByte, int count)
        {
            if (count != 0)
            {
                while (RemoveFillByteArea(fillByte, count)) ;
            }
        }
        #endregion

        #region Properties
        public uint EntryPoint { get { return entryPoint; } set { entryPoint = value; } }

        public uint FirstAddress { get { return (items.Count > 0) ? items[0].Address : 0; } }

        public uint LastAddress { get { return (items.Count > 0) ? items[items.Count - 1].LastAddress : 0; } }

        public int Size { get { return (items.Count > 0) ? (int)(LastAddress - FirstAddress + 1) : 0; } }

        public int ByteCount {  get
            {
                int c = 0;
                foreach (Item i in items)
                {
                    c += i.Data.Length;
                }
                return c;
            }
        }
        #endregion

        public override string ToString()
        {
            return (items.Count == 0) ? string.Empty :
                string.Format("{0:x8}-{1:x8}", FirstAddress, LastAddress);
        }
    }
}