/*----------------------------------------------------------------------------*/
/* Firmware Obfuscation V1.0.4                                                */
/* Written in 2019 by Rudy Tellert Elektronik                                 */
/*                                                                            */
/* To the extent possible under law, the author has dedicated all copyright   */
/* and related and neighboring rights to the software "firmware obfuscation"  */
/* and its documentation to the public domain. This software and its          */
/* documentation are distributed without any warranty.                        */
/*                                                                            */
/* See also                                                                   */
/* http://creativecommons.org/publicdomain/zero/1.0/                          */
/* ---------------------------------------------------------------------------*/

#include "bootldr.h"

#if defined(USE_MAC_WITH_PERMUTATION) && defined(IMPLEMENT_PERMUTATION128_WITH_HASH)
#error Cannot use both USE_MAC_WITH_PERMUTATION and IMPLEMENT_PERMUTATION128_WITH_HASH
#endif

/*----------------------------------------------------------------------------*/
/* Hash Function                                                              */
/*----------------------------------------------------------------------------*/

#ifdef IMPLEMENT_HASH

#define LO 0
#define HI 1

#ifdef USE_MAC_WITH_PERMUTATION
Bool InitKeyedHash(Hash *hash, const Byte *key, size_t keySize)
{
    hash->State[LO] = 1;
    hash->State[HI] = 0;
    hash->Count = 0;
    if (!InitPermutation128(&hash->Permutation, key, keySize)) return FALSE;
    zeromem(hash->Permutation0, PERMUTATION128_BLOCK_LENGTH);
    ForwardPermutation128(&hash->Permutation, hash->Permutation0, PERMUTATION128_BLOCK_LENGTH, hash->Permutation0, PERMUTATION128_BLOCK_LENGTH);

    return TRUE;
}
#else
void InitHash(Hash *hash)
{
    hash->State[LO] = 1;
    hash->State[HI] = 0;
}
#endif

static void HashCopyFrom(Hash *dest, const Hash *src)
{
    dest->State[LO] = src->State[LO];
    dest->State[HI] = src->State[HI];
}

static void HashShiftLeft8(Hash *hash)
{
    hash->State[HI] = (hash->State[HI] << 8) | (hash->State[LO] >> (64-8));
    hash->State[LO] <<= 8;
}

static void HashAdd(Hash *dest, Hash *right)
{
    ULong tmp = dest->State[LO];
    dest->State[LO] += right->State[LO];
    if (dest->State[LO] < tmp) dest->State[HI]++;
    dest->State[HI] += right->State[HI];
}

static void HashAddUShort(Hash *dest, UShort right)
{
    ULong tmp = dest->State[LO];
    dest->State[LO] += right;
    if (dest->State[LO] < tmp) dest->State[HI]++;
}

#ifdef USE_MAC_WITH_PERMUTATION
static void HashToLittleEndianByteArray(const Hash *hash, Byte *arr)
{
#if HASH_LENGTH != 16
#error Invalid hash length!
#endif
#ifdef SUPPORT_BIG_ENDIAN
    if (IsLittleEndian()) {
#endif
        memcpy(arr, &hash->State[LO], 16);
#ifdef SUPPORT_BIG_ENDIAN
    }
    else {
        memcpy(arr, &hash->State[LO], 8);
        memcpy(arr + 8, &hash->State[HI], 8);
        ReverseArray(arr, 16);
    }
#endif
}
#else
static void HashToBigEndianByteArray(const Hash *hash, Byte *arr)
{
#if HASH_LENGTH != 16
#error Invalid hash length!
#endif
#ifdef SUPPORT_BIG_ENDIAN
    if (IsLittleEndian()) {
#endif
        memcpy(arr, &hash->State[LO], 16);
        ReverseArray(arr, 16);
#ifdef SUPPORT_BIG_ENDIAN
    }
    else {
        memcpy(arr, &hash->State[HI], 8);
        memcpy(arr + 8, &hash->State[LO], 8);
    }
#endif
}
#endif

void UpdateHash(Hash *hash, const Byte *src, size_t srcSize)
{
#ifdef USE_MAC_WITH_PERMUTATION
    hash->Count += srcSize;
#endif
    while (srcSize--) {
        Hash tmp;
        HashCopyFrom(&tmp, hash);
        HashShiftLeft8(&tmp);
        HashAdd(hash, &tmp);
        HashAddUShort(hash, ((UShort)(*src++)) + 1);
    }
}

size_t GetHash(Hash *hash, Byte *dest, size_t destSize)
{
#ifdef USE_MAC_WITH_PERMUTATION
    size_t count;
#else
    unsigned i;
#endif

    if (destSize < HASH_LENGTH) return 0;

#ifdef USE_MAC_WITH_PERMUTATION
    HashToLittleEndianByteArray(hash, dest);
    ForwardPermutation128(&hash->Permutation, dest, HASH_LENGTH, dest, HASH_LENGTH);
#ifdef SUPPORT_BIG_ENDIAN
    if (IsLittleEndian()) {
#endif
        *(size_t*)dest ^= hash->Count;
#ifdef SUPPORT_BIG_ENDIAN
    }
    else {
        count = hash->Count;
        ReverseArray((Byte*)&count, sizeof(size_t));
        *(size_t*)dest ^= count;
    }
#endif
    ForwardPermutation128(&hash->Permutation, dest, HASH_LENGTH, dest, HASH_LENGTH);
#else
    HashToBigEndianByteArray(hash, dest);
    for (i = HASH_LENGTH-1; i--; ) dest[i] ^= dest[i + 1];
#endif

    return HASH_LENGTH;
}

#else

void InitHash(Hash *hash)
{
    /* [insert code here] */
}

void UpdateHash(Hash *hash, const Byte *src, size_t srcSize)
{
    /* [insert code here] */
}

size_t GetHash(Hash *hash, Byte *dest, size_t destSize)
{
    /* [insert code here] */

    if (destSize < HASH_LENGTH) return 0;

    zeromem(dest, HASH_LENGTH);

    return HASH_LENGTH;
}

#endif

/*----------------------------------------------------------------------------*/
/* 128-bit Block Permutation (by Using a Hash Function)                       */
/*----------------------------------------------------------------------------*/

#ifdef IMPLEMENT_PERMUTATION128

#ifdef IMPLEMENT_PERMUTATION128_WITH_HASH

Bool InitPermutation128(Permutation128 *p, const Byte *key, size_t keySize) 
{
    if (keySize > sizeof(p->Key)) return FALSE;
    memcpy(p->Key, key, keySize);
    memset(&p->Args, '\0', sizeof(MacArgs));
    p->Args.Key = p->Key;
    p->Args.KeySize = keySize;
    return TRUE;
}

size_t ForwardPermutation128(Permutation128 *p, const Byte *in, size_t inSize, Byte *out, size_t outSize)
{
#ifndef USE_HASH_INSTEAD_OF_MAC
    p->Args.Nonce = (Byte*)in;
    p->Args.NonceSize = inSize;
    return GetMac(&p->Args, out, outSize);
#else
    return GetKeyedHash(p->Args.Key, p->Args.KeySize, in, inSize, out, outSize);
#endif
}

#else

void InitPermutation128KeySchedule(void)
{
    /* A RAM optimized version is to create the key schedule in ROM and copy it like hwInfo into the bootloader hex file */

    /* [insert code here] */
}

Bool InitPermutation128(Permutation128 *p, const Byte *key, size_t keySize)
{
    if (key == hwInfo->SigningKey) {
        /* take the precomputed key schedule of the signing key [insert code here] */
    }
    else {
        /* take the precomputed key schedule of the obfuscation key [insert code here] */
    }

    return TRUE;
}

size_t ForwardPermutation128(Permutation128 *p, const Byte *in, size_t inSize, Byte *out, size_t outSize)
{
    /* [insert code here] */

    return PERMUTATION128_BLOCK_LENGTH;
}

#endif

#else

/*----------------------------------------------------------------------------*/
/* Obfuscation is NOT Used                                                    */
/*----------------------------------------------------------------------------*/

size_t ForwardPermutation128(Permutation128 *p, const Byte *in, size_t inSize, Byte *out, size_t outSize)
{
    if (outSize < inSize) return 0;

    if (in != out) memcpy(out, in, inSize);

    return inSize;
}

#endif