/*----------------------------------------------------------------------------*/
/* Firmware Obfuscation V1.0.4                                                */
/* Written in 2019 by Rudy Tellert Elektronik                                 */
/*                                                                            */
/* To the extent possible under law, the author has dedicated all copyright   */
/* and related and neighboring rights to the software "firmware obfuscation"  */
/* and its documentation to the public domain. This software and its          */
/* documentation are distributed without any warranty.                        */
/*                                                                            */
/* See also                                                                   */
/* http://creativecommons.org/publicdomain/zero/1.0/                          */
/* ---------------------------------------------------------------------------*/

#include "bootldr.h"

#ifdef _MSC_VER
#pragma warning( disable : 4312 )
#endif

#ifdef SUPPORT_BIG_ENDIAN
static Bool reverse;
#endif

#ifndef USE_TINY_BOOTLOADER

Bool GetMap(const void *data, size_t dataSize, void *mapData, size_t mapSize, Bool obfuscated,
    const application_info_t *appInfo, unsigned blockNumber)
{
    mem_block_t mbObject, *mb = &mbObject;
    map_t *map;
    TsfObject tsfObject, *tsf;
    unsigned typeVal;
    drm_item_t *drm1 = NULL;
    UInt prevAddr = 0;
    UInt minPrg = (UInt)hwInfo.AppInfo;
    UInt maxPrg = minPrg + 0x100;
    unsigned i, n;

    MemBlockInit(mb, mapData, mapSize);
    map = (map_t *)MemBlockAlloc(mb, 1, sizeof(map_t));
    if (map == 0) return FALSE;

    if (blockNumber == 0) reverse = FALSE;
    
    tsf = &tsfObject;
    TsfOpen(tsf, data, dataSize);
#ifdef SUPPORT_BIG_ENDIAN
    if (!IsLittleEndian()) {
        tsf->Header = TSFH_REV;
#ifdef TSF_SUPPORT_REV_FIELD
        tsf->Reverse = TSF_TRUE;
#endif
    }
#endif
    if (TsfSelect(tsf, 2)) {
        UInt signature;
        if (TsfReadUInt32(tsf, &signature)) {
            if (signature != 0x0b7c1f9a) {
#ifdef SUPPORT_BIG_ENDIAN
                if (signature != 0x9a1f7c0b) return FALSE;
                reverse = TRUE;
#else
                return FALSE;
#endif
            }
        }
        else {
            return FALSE;
        }
    }

#ifdef SUPPORT_BIG_ENDIAN
    if (reverse) {
        TsfSetReverse(tsf, !TsfGetReverse(tsf));
    }
#endif

    if (blockNumber == 0) {
        if ((n = (unsigned)TsfReadArraySize(tsf, 5)) != 0) {
            drm_item_t *drm = map->DrmItems = (drm_item_t*)MemBlockAlloc(mb, n, sizeof(drm_item_t));
            if (drm == 0) return FALSE;
            map->DrmCount = n;

            for (i = 0; i < n; i++) {
                UInt minMax = 0;
                if (TsfSelectObject(tsf, 3)) if (!TsfReadUInt32(tsf, &drm->Type)) return FALSE;
                if (TsfSelectObject(tsf, 5)) if (!TsfReadUInt32(tsf, &drm->Min)) return FALSE;
                if (TsfSelectObject(tsf, 7)) if (!TsfReadUInt32(tsf, &drm->Max)) return FALSE;
                if (TsfSelectObject(tsf, 9)) if (!TsfReadUInt32(tsf, &minMax)) return FALSE;
                if (TsfSelectObject(tsf, 11)) if (!TsfReadUInt32(tsf, &drm->XorMask)) return FALSE;
                if (TsfSelectObject(tsf, 13)) if (!TsfReadUInt32(tsf, &drm->ReqFeatures)) return FALSE;
                TsfSkipCollection(tsf);
                if (minMax) {
                    if (drm->Min || drm->Max) return FALSE;
                    drm->Min = drm->Max = minMax;
                }
                drm++;
            }
        }
        if (TsfSelectObject(tsf, 7)) if (!TsfReadUInt8(tsf, &map->EraseRegions)) return FALSE;

        if ((n = (unsigned)TsfReadArraySize(tsf, 9)) != 0) {
            erase_item_t *eraseItem = map->EraseItems = (erase_item_t*)MemBlockAlloc(mb, n, sizeof(erase_item_t));
            if (eraseItem == 0) return FALSE;
            map->EraseCount = n;

            for (i = 0; i < n; i++) {
                if (TsfSelectObject(tsf, 3)) if (!TsfReadUInt32(tsf, &eraseItem->Address)) return FALSE;
                if (TsfSelectObject(tsf, 5)) if (!TsfReadUInt32(tsf, &eraseItem->Length)) return FALSE;
                TsfSkipCollection(tsf);
                eraseItem++;
            }
        }
    }
    if ((n = (unsigned)TsfReadArraySize(tsf, 15)) != 0) {
        memory_item_t *memoryItem = map->MemoryItems = (memory_item_t*)MemBlockAlloc(mb, n, sizeof(memory_item_t));
        if (memoryItem == 0) return FALSE;
        map->MemoryCount = n;

        for (i = 0; i < n; i++) {
            if (TsfSelectObject(tsf, 17)) if (!TsfReadUInt32(tsf, &memoryItem->Address)) return FALSE;
            if (TsfSelectObject(tsf, 19)) {
                memoryItem->DataItems = (char *)MemBlockAlloc(mb, 1, tsf->DataSize);
                if (memoryItem->DataItems == 0) return FALSE;
                memoryItem->DataCount = tsf->DataSize;
                if (!TsfReadData(tsf, memoryItem->DataItems, tsf->DataSize)) return FALSE;
            }
            TsfSkipCollection(tsf);
            memoryItem++;
        }
    }

    /* test DRM */
    for (i = 0; i < map->DrmCount; i++) {
        drm_item_t *drm = map->DrmItems + i;
        if (drm->Type == 1) drm1 = drm;
        if (drm->Type > 6 && drm->Type != 64) {
            if (obfuscated) return FALSE;
            if ((drm->Type < 129 || drm->Type > 133) && drm->Type != 192) return FALSE;
        }
        if (drm->Min && drm->Max && drm->Min > drm->Max) return FALSE;
        if ((drm->Type & 64) == 0) {
            if (drm->Min == 0 && drm->Max == 0) return FALSE;
        }
        if (drm->Type >= 128) {
            if (appInfo == NULL) return FALSE;
        }

        typeVal = 0;
        switch (drm->Type) {
        case HWINFO_MANUFACTURER_ID: typeVal = hwInfo.ManufacturerId; break;
        case HWINFO_HARDWARE_ID: typeVal = hwInfo.HardwareId; break;
        case HWINFO_HARDWARE_VERSION: typeVal = hwInfo.HardwareVersion; break;
        case HWINFO_INTERNAL_DEVICE_NUMBER: typeVal = hwInfo.InternalDeviceNumber; break;
        case HWINFO_DEVICE_NUMBER: typeVal = hwInfo.DeviceNumber; break;
        case HWINFO_CUSTOMER: typeVal = hwInfo.Customer; break;
        case HWINFO_MANUFACTURING_DATE: typeVal = hwInfo.ManufacturingDate; break;
        case HWINFO_HARDWARE_FEATURES: typeVal = hwInfo.HardwareFeatures; break;
        case APPINFO_ID: typeVal = appInfo->Id; break;
        case APPINFO_VERSION: typeVal = appInfo->Version; break;
        case APPINFO_BUILD: typeVal = appInfo->Build; break;
        case APPINFO_CONFIG: typeVal = appInfo->Config; break;
        case APPINFO_DATE: typeVal = appInfo->Date; break;
        case APPINFO_FEATURES: typeVal = appInfo->Features; break;
        }

        if ((drm->Min && typeVal < drm->Min) ||
            (drm->Max && typeVal > drm->Max) ||
            (((typeVal ^ drm->XorMask) & drm->ReqFeatures) != drm->ReqFeatures)) return FALSE;
    }
    if (blockNumber == 0 && drm1 == NULL) return FALSE;

    for (i = 0; i < map->EraseCount; i++) {
        erase_item_t *ei = map->EraseItems + i;
        if (ei->Length == 0) continue;
        if (ei->Address < prevAddr) return FALSE;
        if (ei->Address + ei->Length <= ei->Address) return FALSE;
        if (!IsAreaProgrammable((const void *)ei->Address, ei->Length)) return FALSE;
        prevAddr = ei->Address + ei->Length;
    }
    prevAddr = 0;
    for (i = 0; i < map->MemoryCount; i++) {
        memory_item_t *mi = map->MemoryItems + i;
        if (mi->DataCount == 0) continue;
        if (mi->Address < prevAddr) return FALSE;
        if (mi->Address + mi->DataCount <= mi->Address) return FALSE;
        if (!IsAreaProgrammable((const void *)mi->Address, mi->DataCount)) return FALSE;
        prevAddr = mi->Address + (UInt)mi->DataCount;
    }

    /* search for appinfo data */
    for (i = 0; i < map->MemoryCount; i++) {
        memory_item_t *mi = map->MemoryItems + i;
        if (mi->DataCount == 0) continue;
        UInt end = mi->Address + (UInt)mi->DataCount;
        if (end <= minPrg || mi->Address >= maxPrg) continue;
        map->LastBlock = 1;
        break;
    }

    if (obfuscated) {
        if (blockNumber != 0 || map->MemoryCount != 1 || map->MemoryItems[0].DataCount != 9 ||
          memcmp(map->MemoryItems[0].DataItems, "ERASE ALL", 9) != 0 || map->LastBlock == 0) {
            map->EraseRegions &= ~ERASE_REGIONS_ALL;
        }
    }
    else {
        /* restrict unobfuscated access */
        unsigned i;
        UInt minData = 0;
        UInt maxData = 0;
        minPrg = 0;
        maxPrg = 0;

        map->EraseRegions &= ERASE_REGIONS_CFG + ERASE_REGIONS_CFG_DATA;

        if (appInfo) {
            minPrg = appInfo->EraseBegin;
            maxPrg = minPrg + appInfo->EraseSize;
            minData = appInfo->EraseDataBegin;
            maxData = minData + appInfo->EraseDataSize;
        }
        for (i = 0; i < map->EraseCount; i++) {
            erase_item_t *ei = map->EraseItems + i;
            UInt end = ei->Address + (UInt)ei->Length;
            if (ei->Length == 0) continue;
            if ((ei->Address < minPrg || end > maxPrg) &&
                (ei->Address < minData || end > maxData)) {
                /* remove region */
                ei->Length = 0;
            }
        }

        minPrg = maxPrg = minData = maxData = 0;
        if (appInfo) {
            minPrg = appInfo->ProgramBegin;
            maxPrg = minPrg + appInfo->ProgramSize;
            minData = appInfo->ProgramDataBegin;
            maxData = minData + appInfo->ProgramDataSize;
        }
        for (i = 0; i < map->MemoryCount; i++) {
            memory_item_t *mi = map->MemoryItems + i;
            UInt end = mi->Address + (UInt)mi->DataCount;
            if (mi->DataCount == 0) continue;
            if ((mi->Address < minPrg || end > maxPrg) &&
                (mi->Address < minData || end > maxData)) {
                /* remove region */
                mi->DataCount = 0;
            }
        }
    }

    return TRUE;
}

#else 

#ifdef SUPPORT_BIG_ENDIAN

static UInt UIntToCorrectEndian(UInt u)
{
    if (reverse) ReverseArray((Byte*)&u, sizeof(UInt));
    return u;
}

#else 

#define UIntToCorrectEndian(u) (u)

#endif

Bool GetMap(const void *data, size_t dataSize, void *mapData, size_t mapSize, Bool obfuscated,
    const application_info_t *appInfo, unsigned blockNumber)
{
    int i;
    map_t *map = (map_t*)mapData;
#define drm ((const drm_t*)data)
#define blockData ((const data_t*)data)
    UInt typeVal;
    Byte eraseRegions;
    UInt type, minOrXorMask, maxOrReqFeatures;
    Bool result = FALSE;

    if (mapSize < sizeof(map_t)) return FALSE;

    if (blockNumber == 0) {

        if (dataSize < sizeof(UInt) + sizeof(UInt) + sizeof(UInt) + sizeof(drm_item_t)) return FALSE;

#ifdef SUPPORT_BIG_ENDIAN
        reverse = FALSE;
        if (drm->DrmSignature != DRM_SIGNATURE) {
            if (drm->DrmSignature != DRM_REV_SIGNATURE) return FALSE;
            reverse = TRUE;
        }
#else 
        if (drm->DrmSignature != DRM_SIGNATURE) return FALSE;
#endif
        eraseRegions = (Byte)UIntToCorrectEndian(drm->EraseRegions);
        if (!obfuscated) eraseRegions &= ERASE_REGIONS_CFG + ERASE_REGIONS_CFG_DATA;
        dataSize -= sizeof(UInt) + sizeof(UInt) + sizeof(UInt); /* Signature + EraseRegions + Count */

        dataSize /= sizeof(drm_item_t);
        if (UIntToCorrectEndian(drm->Count) > 21) return FALSE;
        if (dataSize < UIntToCorrectEndian(drm->Count)) return FALSE;
        for (i = 0; i < (int)dataSize; i++) {
            const drm_item_t *drmItem = drm->Items + i;
            type = UIntToCorrectEndian(drmItem->Type);
            if (obfuscated && type >= APPINFO_ID) return FALSE;
            minOrXorMask = UIntToCorrectEndian(drmItem->MinOrXorMask);
            maxOrReqFeatures = UIntToCorrectEndian(drmItem->MaxOrReqFeatures);
            switch (type) {
            case HWINFO_MANUFACTURER_ID: typeVal = hwInfo.ManufacturerId; break;
            case HWINFO_HARDWARE_ID: typeVal = hwInfo.HardwareId; result = TRUE; break;
            case HWINFO_HARDWARE_VERSION: typeVal = hwInfo.HardwareVersion; break;
            case HWINFO_INTERNAL_DEVICE_NUMBER: typeVal = hwInfo.InternalDeviceNumber; break;
            case HWINFO_DEVICE_NUMBER: typeVal = hwInfo.DeviceNumber; break;
            case HWINFO_CUSTOMER: typeVal = hwInfo.Customer; break;
            case HWINFO_MANUFACTURING_DATE: typeVal = hwInfo.ManufacturingDate; break;
            case HWINFO_HARDWARE_FEATURES: typeVal = hwInfo.HardwareFeatures; break;
            case APPINFO_ID: typeVal = appInfo->Id; break;
            case APPINFO_VERSION: typeVal = appInfo->Version; break;
            case APPINFO_BUILD: typeVal = appInfo->Build; break;
            case APPINFO_CONFIG: typeVal = appInfo->Config; break;
            case APPINFO_DATE: typeVal = appInfo->Date; break;
            case APPINFO_FEATURES: typeVal = appInfo->Features; break;
            default: return FALSE;
            }
            if ((type & FEATURE_MASK) == 0) {
                if ((minOrXorMask && typeVal < minOrXorMask) ||
                    (maxOrReqFeatures && typeVal > maxOrReqFeatures)) return FALSE;
            }
            else {
                if (((typeVal ^ minOrXorMask) & maxOrReqFeatures) != maxOrReqFeatures) return FALSE;
            }
        }

        if (result) {
            map->EraseCount = map->MemoryCount = map->LastBlock = 0;
            map->EraseRegions = eraseRegions;
            return TRUE;
        }

        return FALSE;
    }
    else {
        UInt address;
        UInt count;
        if (dataSize < sizeof(UInt) + sizeof(UInt)) return FALSE;
        address = UIntToCorrectEndian(blockData->Address);
        count = UIntToCorrectEndian(blockData->Count);
        if (count > MAP_DATA_SIZE) return FALSE;
        if (dataSize < sizeof(UInt) + sizeof(UInt) + count) return FALSE;
        memmove(map->Data, blockData->Data, count);
        map->MemoryItem.Address = address;
        map->MemoryItem.DataItems = map->Data;
        map->MemoryItem.DataCount = count;
        map->MemoryItems = &map->MemoryItem;
        map->MemoryCount = 1;
        map->LastBlock = (address == hwInfo.AppInfo);
        if (!IsAreaProgrammable((const void*)address, count)) return FALSE;
        if (!obfuscated) {
            /* restrict unobfuscated access */
            UInt minPrg = 0;
            UInt maxPrg = 0;
            UInt minData = 0;
            UInt maxData = 0;
            memory_item_t *mi;

            if (appInfo) {
                minPrg = appInfo->ProgramBegin;
                maxPrg = minPrg + appInfo->ProgramSize;
                minData = appInfo->ProgramDataBegin;
                maxData = minData + appInfo->ProgramDataSize;
            }
            mi = map->MemoryItems;
            UInt end = mi->Address + (UInt)mi->DataCount;
            if ((mi->Address < minPrg || end > maxPrg) &&
                (mi->Address < minData || end > maxData)) {
                /* remove region */
                mi->DataCount = 0;
            }
        }

        map->EraseRegions = map->EraseCount = map->MemoryCount = 0;

        return TRUE;
    }
}

#endif