﻿/*----------------------------------------------------------------------------*/
/* Firmware Obfuscation V1.0.4                                                */
/* Written in 2019 by Rudy Tellert Elektronik                                 */
/*                                                                            */
/* To the extent possible under law, the author has dedicated all copyright   */
/* and related and neighboring rights to the software "firmware obfuscation"  */
/* and its documentation to the public domain. This software and its          */
/* documentation are distributed without any warranty.                        */
/*                                                                            */
/* See also                                                                   */
/* http://creativecommons.org/publicdomain/zero/1.0/                          */
/* ---------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/* Excerpt from Tagged Stream Format V1.1.16                                  */
/* Requirements: .NET Framework 2 (or better)                                 */
/*----------------------------------------------------------------------------*/

using System;
using System.IO;

namespace Tellert.Format
{
    public class Tsf
    {
        public enum HeaderType { None, Normal, Rev, LittleEndian, BigEndian };
        public enum Status { Data, End, EndOfStream, Error };
        public enum Extension { Default, Null, Redef, FixedArray = 0x11, VarArray = 0x12 };
        public enum Token
        {
            None, End, EndOfStream, Error, EndOfArray, EndOfCollection,
            Data, Collection, Array, Ext
        }
        static byte[] Prefix = new byte[7] { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };
        struct LevelData
        {
            public int Count;
            public int CurrId;
        }

        byte[] Data;
        HeaderType Ht;
        int pos;
        bool reverse;
        public bool Reverse { get { return reverse; } set { reverse = value; } }

        Status status;
        int dataSize;

        int Level;
        LevelData[] levelData;
        int Count;
        int Id;
        public Tsf() { }
        public Tsf(int size, HeaderType ht = HeaderType.Normal, bool writeHeader = true) { Create(size, ht, writeHeader); }
        public Tsf(byte[] data) { Open(data); }
        public Tsf(byte[] data, int offset, int size) { Open(data, offset, size); }
        public Tsf(string fileName) { Open(fileName); }
        public bool Good { get { return status != Status.Error; } }
        public bool Bad { get { return !Good; } }

        void Init()
        {
            Data = null;
            Ht = HeaderType.None;
            pos = 0;
            reverse = false;
            status = Status.Data;
            dataSize = 0;
            Level = 0;
            levelData = new LevelData[8];
            Count = 0;
            Id = 0;
        }
        public int DataSize { get { return dataSize; } }

        public int Pos
        {
            get { return pos; }
            set { pos = value; }
        }

        public void Create(int size, HeaderType ht = HeaderType.Normal, bool writeHeader = true)
        {
            Init();
            Data = new byte[size];

            if (ht == HeaderType.LittleEndian || ht == HeaderType.BigEndian)
            {
                bool isHtLittleEndian = ht == HeaderType.LittleEndian;
                ht = (isHtLittleEndian ^ BitConverter.IsLittleEndian) ? HeaderType.Rev : HeaderType.Normal;
            }
            Ht = ht;

            if (ht != HeaderType.None && size >= 4)
            {
                if (writeHeader)
                {
                    Data[0] = 0x01;
                    Data[1] = 0xe1;
                    if (ht == HeaderType.Rev ^ BitConverter.IsLittleEndian)
                    {
                        Data[2] = 0x74;
                        Data[3] = 0x73;
                    }
                    else
                    {
                        Data[2] = 0x73;
                        Data[3] = 0x74;
                    }
                    pos = 4;
                }
                reverse = ht == HeaderType.Rev;
            }
        }

        public byte[] ToBytes()
        {
            byte[] dest = new byte[pos];
            Array.Copy(Data, dest, pos);
            return dest;
        }

        bool Write(byte b)
        {
            if (status == Status.Error) return false;
            if (pos + 1 >= Data.Length)
            {
                status = Status.Error;
                return false;
            }
            Data[pos++] = b;
            return true;
        }

        bool Write(byte[] src)
        {
            return Write(src, 0, src.Length);
        }

        bool Write(byte[] src, int srcOffset, int count)
        {
            if (status == Status.Error) return false;
            if (pos + count >= Data.Length)
            {
                status = Status.Error;
                return false;
            }
            Array.Copy(src, srcOffset, Data, pos, count);
            pos += count;
            return true;
        }

        bool WriteSorted(byte[] src)
        {
            return WriteSorted(src, 0, src.Length);
        }

        bool WriteSorted(byte[] src, int srcOffset, int count)
        {
            if (reverse)
            {
                if (!Write(src, srcOffset, count)) return false;
                Array.Reverse(Data, pos - count, count);
                return true;
            }
            else
            {
                return Write(src, srcOffset, count);
            }
        }

        bool WriteNumber(ulong num)
        {
            if (num < byte.MaxValue)
            {
                byte[] arr = new byte[1];
                arr[0] = (byte)num;
                return Write(arr, 0, 1);
            }
            else if (num < ushort.MaxValue)
            {
                byte[] arr = BitConverter.GetBytes((ushort)num);

                if (!Write((byte)0xff)) return false;
                return WriteSorted(arr, 0, 2);
            }
            else if (num < uint.MaxValue)
            {
                byte[] arr = BitConverter.GetBytes((uint)num);

                if (!Write(Prefix, 0, 3)) return false;
                return WriteSorted(arr, 0, 4);
            }
            else if (num < ulong.MaxValue)
            {
                byte[] arr = BitConverter.GetBytes(num);

                if (!Write(Prefix, 0, 7)) return false;
                return WriteSorted(arr, 0, 8);
            }

            status = Status.Error;
            return false;
        }

        bool WriteId(int id, int type)
        {
            if (id == 0)
            {
                status = Status.Error;
                return false;
            }

            byte idByte = (byte)((id < 0x1f) ? id : 0x1f);
            idByte = (byte)((idByte << 3) | type);
            if (!Write(idByte)) return false;
            if (id >= 0x1f)
            {
                if (!WriteNumber((ulong)id)) return false;
            }
            return true;
        }

        public bool WriteArray(int id, int size)
        {
            if (size == 0) return false;

            if (!WriteId(id, (size == 1) ? 0x05 : 0x06)) return false;
            if (size > 1) return WriteNumber((ulong)size);
            return true;
        }

        public bool WriteEnd()
        {
            return Write((byte)0);
        }

        public bool Write(int id, byte[] src)
        {
            return Write(id, src, 0, src.Length);
        }

        public bool Write(int id, byte[] src, int offset, int count)
        {
#if TSF_WRITE_VAL0 != true
            uint c;

            for (c = 0; c < count; c++) if (src[offset + c] != 0) break;
            if (c == count) return true;
#endif
            switch (count)
            {
                case 0:
#if TSF_WRITE_SIZE0
                    if (!WriteId(id, 4)) return false;
                    return WriteNumber(0);
#else
                    return true;
#endif
                case 1:
                    if (!WriteId(id, 0)) return false;
                    break;
                case 2:
                    if (!WriteId(id, 1)) return false;
                    break;
                case 4:
                    if (!WriteId(id, 2)) return false;
                    break;
                case 8:
                    if (!WriteId(id, 3)) return false;
                    break;
                default:
                    if (!WriteId(id, 4)) return false;
                    if (!WriteNumber((ulong)count)) return false;
                    break;
            }

            return Write(src, offset, count);
        }

        public bool Write0(int id, byte[] src)
        {
            return Write0(id, src, 0, src.Length);
        }

        public bool Write0(int id, byte[] src, int offset, int count)
        {
            switch (count)
            {
                case 0:
#if TSF_WRITE_SIZE0
                    if (!WriteId(id, 4)) return false;
                    return WriteNumber(0);
#else
                    return true;
#endif
                case 1:
                    if (!WriteId(id, 0)) return false;
                    break;
                case 2:
                    if (!WriteId(id, 1)) return false;
                    break;
                case 4:
                    if (!WriteId(id, 2)) return false;
                    break;
                case 8:
                    if (!WriteId(id, 3)) return false;
                    break;
                default:
                    if (!WriteId(id, 4)) return false;
                    if (!WriteNumber((ulong)count)) return false;
                    break;
            }

            return Write(src, offset, count);
        }

        public bool SortedWrite(int id, byte[] src)
        {
            return SortedWrite(id, src, 0, src.Length);
        }

        public bool SortedWrite(int id, byte[] src, int offset, int count)
        {
#if TSF_WRITE_VAL0 != true
            uint c;

            for (c = 0; c < count; c++) if (src[offset + c] != 0) break;
            if (c == count) return true;
#endif
            switch (count)
            {
                case 0:
#if TSF_WRITE_SIZE0
                    if (!WriteId(id, 4)) return false;
                    return WriteNumber(0);
#else
                    return true;
#endif
                case 1:
                    if (!WriteId(id, 0)) return false;
                    break;
                case 2:
                    if (!WriteId(id, 1)) return false;
                    break;
                case 4:
                    if (!WriteId(id, 2)) return false;
                    break;
                case 8:
                    if (!WriteId(id, 3)) return false;
                    break;
                default:
                    if (!WriteId(id, 4)) return false;
                    if (!WriteNumber((ulong)count)) return false;
                    break;
            }

            return WriteSorted(src, offset, count);
        }

        public bool Write(int id, sbyte val)
        {
            byte[] arr = new byte[1];
            arr[0] = (byte)val;
            return Write(id, arr, 0, 1);
        }

        public bool Write(int id, byte val)
        {
            byte[] arr = new byte[1];
            arr[0] = val;
            return Write(id, arr, 0, 1);
        }

        public bool Write(int id, short val)
        {
            if (val >= sbyte.MinValue && val <= sbyte.MaxValue) return Write(id, (sbyte)val);

            byte[] arr = BitConverter.GetBytes(val);
            return SortedWrite(id, arr, 0, 2);
        }

        public bool Write(int id, ushort val)
        {
            if (val >= byte.MinValue && val <= byte.MaxValue) return Write(id, (byte)val);

            byte[] arr = BitConverter.GetBytes(val);
            return SortedWrite(id, arr, 0, 2);
        }

        public bool Write(int id, int val)
        {
            if (val >= short.MinValue && val <= short.MaxValue) return Write(id, (short)val);

            byte[] arr = BitConverter.GetBytes(val);
            return SortedWrite(id, arr, 0, 4);
        }

        public bool Write(int id, uint val)
        {
            if (val >= ushort.MinValue && val <= ushort.MaxValue) return Write(id, (ushort)val);

            byte[] arr = BitConverter.GetBytes(val);
            return SortedWrite(id, arr, 0, 4);
        }

        public bool Write(int id, long val)
        {
            if (val >= int.MinValue && val <= int.MaxValue) return Write(id, (int)val);

            byte[] arr = BitConverter.GetBytes(val);
            return SortedWrite(id, arr, 0, 8);
        }

        public bool Write(int id, ulong val)
        {
            if (val >= uint.MinValue && val <= uint.MaxValue) return Write(id, (uint)val);

            byte[] arr = BitConverter.GetBytes(val);
            return SortedWrite(id, arr, 0, 8);
        }

        public bool Write(int id, float val)
        {
            byte[] arr = BitConverter.GetBytes(val);
            return SortedWrite(id, arr, 0, 4);
        }

        public bool Write(int id, double val)
        {
            byte[] arr = BitConverter.GetBytes(val);
            return SortedWrite(id, arr, 0, 8);
        }

        public void WriteFile(string fileName)
        {
            if (status == Status.Error) throw new FormatException("Invalid TSF document");

            using (FileStream fs = new FileStream(fileName, FileMode.Create))
            {
                fs.Write(Data, 0, pos);
            }
        }

        public bool TryWriteFile(string fileName)
        {
            bool result = true;

            try
            {
                WriteFile(fileName);
            }
            catch
            {
                result = false;
            }

            return result;
        }

        //======================================================================

        public void Open(byte[] data)
        {
            Open(data, 0, data.Length);
        }

        public void Open(byte[] data, int offset, int size)
        {
            Init();
            if (size >= 0)
            {
                this.Data = new byte[size];
                if (size != 0) Array.Copy(data, offset, this.Data, 0, size);
                ReadHeader();
            }
        }

        public void Open(string fileName)
        {
            ReadFile(fileName);
        }

        public bool TryOpen(string fileName)
        {
            bool result = true;

            try
            {
                Open(fileName);
            }
            catch
            {
                result = false;
            }

            return result;
        }

        public void Close()
        {
        }

        void ReadHeader()
        {
            if (Data.Length >= 4)
            {
                if (Data[0] == 0x01 && Data[1] == 0xe1)
                {
                    if (Data[2] == 0x74 && Data[3] == 0x73)
                    {
                        reverse = !BitConverter.IsLittleEndian;
                        pos = 4;
                    }
                    else if (Data[2] == 0x73 && Data[3] == 0x74)
                    {
                        reverse = BitConverter.IsLittleEndian;
                        pos = 4;
                    }
                    Ht = reverse ? HeaderType.Rev : HeaderType.Normal;
                }
            }
        }

        public void ReadFile(string fileName)
        {
            Init();
            using (FileStream fs = new FileStream(fileName, FileMode.Open))
            {
                Data = new byte[fs.Length];
                fs.Read(Data, 0, Data.Length);
                ReadHeader();
            }
        }

        public int Available
        {
            get { return Data.Length - pos; }
        }

        public bool Read(byte[] data, int offset, int size)
        {
            if (status == Status.Error) return false;
            if (size == 0) return true;
            if (Available < size)
            {
                status = Status.Error;
                return false;
            }
            Array.Copy(this.Data, pos, data, offset, size);
            pos += size;
            return true;
        }

        public bool Read(ref byte[] bytes)
        {
            bytes = new byte[DataSize];
            return Read(bytes, 0, DataSize);
        }

        public void SkipData()
        {
            if (Available >= dataSize) pos += dataSize;
        }

        public bool ReadSorted(byte[] data)
        {
            return ReadSorted(data, 0, data.Length);
        }

        public bool ReadSorted(byte[] data, int offset, int size)
        {
            if (!Read(data, offset, size)) return false;
            if (reverse) Array.Reverse(data, offset, size);
            return true;
        }

        bool GetNumber(ref ulong num)
        {
            byte[] tmp = new byte[8];

            if (!Read(tmp, 0, 1)) return false;
            if (tmp[0] != byte.MaxValue)
            {
                num = tmp[0];
                return true;
            }
            if (!ReadSorted(tmp, 0, 2)) return false;

            ushort val16 = BitConverter.ToUInt16(tmp, 0);

            if (val16 != ushort.MaxValue)
            {
                num = val16;
                return true;
            }
            if (!ReadSorted(tmp, 0, 4)) return false;

            uint val32 = BitConverter.ToUInt32(tmp, 0);

            if (val32 != uint.MaxValue)
            {
                num = val32;
                return true;
            }
            if (!ReadSorted(tmp, 0, 8)) return false;

            ulong val64 = BitConverter.ToUInt64(tmp, 0);

            if (val64 != ulong.MaxValue)
            {
                num = val64;
                return true;
            }
            status = Status.Error;
            return false;
        }

        public bool IsAvailable(int size)
        {
            return (Available >= size);
        }

        public Token ReadToken()
        {
            byte[] t = new byte[1];
            byte id;
            ulong num = 0;

            if (status >= Status.End) return (Token)status;

            dataSize = 0;
            Count = 0;

            for (;;)
            {
                if (!Read(t, 0, 1))
                {
                    if (Level != 0) goto Error;
                    status = Status.EndOfStream;
                    return Token.EndOfStream;
                }
                this.Id = id = (byte)(t[0] >> 3);
                if (id != 0)
                {
                    if (id == 0x1f)
                    {
                        if (!GetNumber(ref num)) goto Error;
                        this.Id = (int)num;
                        if (this.Id < 0x1f || num > int.MaxValue) goto Error;
                    }
                    if (this.Id <= levelData[Level].CurrId) goto Error;
                    levelData[Level].CurrId = this.Id;

                    switch (t[0] & 7)
                    {
                        case 0:
                            if (!IsAvailable(dataSize = 1)) goto Error;
                            return Token.Data;
                        case 1:
                            if (!IsAvailable(dataSize = 2)) goto Error;
                            return Token.Data;
                        case 2:
                            if (!IsAvailable(dataSize = 4)) goto Error;
                            return Token.Data;
                        case 3:
                            if (!IsAvailable(dataSize = 8)) goto Error;
                            return Token.Data;
                        case 4:
                            if (!GetNumber(ref num)) goto Error;
                            if (num > int.MaxValue) goto Error;
                            dataSize = (int)num;
                            if (dataSize == 1 || dataSize == 2 || dataSize == 4 ||
                                dataSize == 8) goto Error;
                            if (!IsAvailable(dataSize)) goto Error;
                            return Token.Data;
                        case 5:
                            levelData[Level].Count = Count = 1;
                            if (++Level == levelData.Length) goto Error;
                            return Token.Collection;
                        case 6:
                            if (!GetNumber(ref num)) goto Error;
                            if (num > int.MaxValue) goto Error;
                            Count = (int)num;
                            if (Count <= 1) goto Error;
                            levelData[Level].Count = Count;
                            if (++Level == levelData.Length) goto Error;
                            return Token.Array;
                        case 7:
                            goto Error;
                    }
                }
                else
                {
                    switch (t[0])
                    {
                        case 0:
                            levelData[Level].CurrId = 0;
                            if (Level != 0)
                            {
                                if (--levelData[Level - 1].Count == 0)
                                {
                                    Level--;
                                    return Token.EndOfArray;
                                }
                                else
                                {
                                    return Token.EndOfCollection;
                                }
                            }
                            else
                            {
                                status = Status.End;
                                return Token.End;
                            }

                        case 2: // NOP
                            continue;
                    }
                }
                break;
            }

        Error:
            status = Status.Error;
            return Token.Error;
        }

        public bool Read(ref sbyte data)
        {
            long tmp64 = 0;

            if (!Read(ref tmp64)) return false;

            if (tmp64 > sbyte.MaxValue) return false;
            if (tmp64 < sbyte.MinValue) return false;
            data = (sbyte)tmp64;

            return true;
        }

        public bool Read(ref short data)
        {
            long tmp64 = 0;

            if (!Read(ref tmp64)) return false;

            if (tmp64 > short.MaxValue) return false;
            if (tmp64 < short.MinValue) return false;
            data = (short)tmp64;

            return true;
        }

        public bool Read(ref int data)
        {
            long tmp64 = 0;

            if (!Read(ref tmp64)) return false;

            if (tmp64 > int.MaxValue) return false;
            if (tmp64 < int.MinValue) return false;
            data = (int)tmp64;

            return true;
        }

        public bool Read(ref long data)
        {
            byte[] b = new byte[8];

            if (dataSize == 1)
            {
                Read(b, 0, 1);
                data = b[0];
                return true;
            }
            else if (dataSize == 2)
            {
                ReadSorted(b, 0, 2);
                data = BitConverter.ToInt16(b, 0);
                return true;
            }
            else if (dataSize == 4)
            {
                ReadSorted(b, 0, 4);
                data = BitConverter.ToInt32(b, 0);
                return true;
            }
            else if (dataSize == 8)
            {
                ReadSorted(b, 0, 8);
                data = BitConverter.ToInt64(b, 0);
                return true;
            }

            SkipData();
            return false;
        }

        public bool Read(ref byte data)
        {
            ulong tmp64 = 0;

            if (!Read(ref tmp64)) return false;

            if (tmp64 > byte.MaxValue) return false;
            data = (byte)tmp64;

            return true;
        }

        public bool Read(ref ushort data)
        {
            ulong tmp64 = 0;

            if (!Read(ref tmp64)) return false;

            if (tmp64 > ushort.MaxValue) return false;
            data = (ushort)tmp64;

            return true;
        }

        public bool Read(ref uint data)
        {
            ulong tmp64 = 0;

            if (!Read(ref tmp64)) return false;

            if (tmp64 > uint.MaxValue) return false;
            data = (uint)tmp64;

            return true;
        }

        public bool Read(ref ulong data)
        {
            byte[] b = new byte[8];

            if (dataSize == 1)
            {
                Read(b, 0, 1);
                data = b[0];
                return true;
            }
            else if (dataSize == 2)
            {
                ReadSorted(b, 0, 2);
                data = BitConverter.ToUInt16(b, 0);
                return true;
            }
            else if (dataSize == 4)
            {
                ReadSorted(b, 0, 4);
                data = BitConverter.ToUInt32(b, 0);
                return true;
            }
            else if (dataSize == 8)
            {
                ReadSorted(b, 0, 8);
                data = BitConverter.ToUInt64(b, 0);
                return true;
            }

            SkipData();
            return false;
        }

        public bool Read(ref float data)
        {
            byte[] b = new byte[8];

            if (dataSize == 4)
            {
                ReadSorted(b, 0, 4);
                data = BitConverter.ToSingle(b, 0);
                return true;
            }
            else if (dataSize == 8)
            {
                ReadSorted(b, 0, 8);
                data = (float)BitConverter.ToDouble(b, 0);
                return true;
            }
            else if (dataSize == 1)
            {
                ReadSorted(b, 0, 1);
                data = (sbyte)b[0];
                return true;
            }
            else if (dataSize == 2)
            {
                ReadSorted(b, 0, 2);
                data = BitConverter.ToInt16(b, 0);
                return true;
            }

            SkipData();
            return false;
        }

        public bool Read(ref double data)
        {
            byte[] b = new byte[8];

            if (dataSize == 8)
            {
                ReadSorted(b, 0, 8);
                data = BitConverter.ToDouble(b, 0);
                return true;
            }
            else if (dataSize == 4)
            {
                ReadSorted(b, 0, 4);
                data = BitConverter.ToSingle(b, 0);
                return true;
            }
            else if (dataSize == 1)
            {
                ReadSorted(b, 0, 1);
                data = (sbyte)b[0];
                return true;
            }
            else if (dataSize == 2)
            {
                ReadSorted(b, 0, 2);
                data = BitConverter.ToInt16(b, 0);
                return true;
            }

            SkipData();
            return false;
        }

        public bool Select(int id)
        {
            int level;
            int pos;

            if (status >= Status.End) return false;

            level = this.Level;

            for (;;)
            {
                pos = Pos;
                switch (ReadToken())
                {
                    case Token.Ext:
                        goto case Token.Data;
                    case Token.Data:
                        if (this.Level == level && this.Id >= id)
                        {
                            if (this.Id == id) return true;
                            Pos = pos;
                            levelData[this.Level].CurrId = 0;
                            return false;
                        }
                        SkipData();
                        continue;
                    case Token.Collection:
                        goto case Token.Array;
                    case Token.Array:
                        if (this.Level == level + 1 && this.Id >= id)
                        {
                            Pos = pos;
                            --this.Level;
                            levelData[this.Level].CurrId = 0;
                            return false;
                        }
                        continue;
                    case Token.EndOfCollection:
                        if (this.Level == level)
                        {
                            Pos = pos;
                            levelData[this.Level - 1].Count++;
                            levelData[this.Level].CurrId = 0;
                            return false;
                        }
                        continue;
                    case Token.EndOfArray:
                        if (this.Level + 1 == level)
                        {
                            Pos = pos;
                            levelData[this.Level++].Count++;
                            levelData[this.Level].CurrId = 0;
                            return false;
                        }
                        continue;
                }
                return false;
            }
        }

        public bool SelectObject(int id)
        {
            return Select(id);
        }

        public int ReadArraySize(int id)
        {
            int level;
            int pos;

            if (status >= Status.End) return 0;

            level = this.Level;

            for (;;)
            {
                pos = Pos;
                switch (ReadToken())
                {
                    case Token.Ext:
                        goto case Token.Data;
                    case Token.Data:
                        if (this.Level == level && this.Id >= id)
                        {
                            Pos = pos;
                            levelData[this.Level].CurrId = 0;
                            return 0;
                        }
                        SkipData();
                        continue;
                    case Token.Collection:
                        goto case Token.Array;
                    case Token.Array:
                        if (this.Level == level + 1 && this.Id >= id)
                        {
                            if (this.Id == id) return Count;
                            Pos = pos;
                            --this.Level;
                            levelData[this.Level].CurrId = 0;
                            return 0;
                        }
                        continue;
                    case Token.EndOfCollection:
                        if (this.Level == level)
                        {
                            Pos = pos;
                            levelData[this.Level - 1].Count++;
                            levelData[this.Level].CurrId = 0;
                            return 0;
                        }
                        continue;
                    case Token.EndOfArray:
                        if (this.Level + 1 == level)
                        {
                            Pos = pos;
                            levelData[this.Level++].Count++;
                            levelData[this.Level].CurrId = 0;
                            return 0;
                        }
                        continue;
                }
                return 0;
            }
        }

        public bool SelectArray(int id)
        {
            return ReadArraySize(id) != 0;
        }

        bool SkipCollectionOrArray(bool skipCollectionOnly)
        {
            int level;

            if (status >= Status.End) return false;

            level = this.Level;

            for (;;)
            {
                switch (ReadToken())
                {
                    case Token.Ext:
                        goto case Token.Data;
                    case Token.Data:
                        SkipData();
                        continue;
                    case Token.Collection:
                        goto case Token.Array;
                    case Token.Array:
                        continue;
                    case Token.EndOfCollection:
                        if (skipCollectionOnly && this.Level == level) return true;
                        continue;
                    case Token.EndOfArray:
                        if (this.Level + 1 == level) return true;
                        continue;
                }
                return false;
            }
        }

        public bool SkipCollection()
        {
            return SkipCollectionOrArray(true);
        }

        public bool SkipArray()
        {
            return SkipCollectionOrArray(false);
        }
    }
}