/*----------------------------------------------------------------------------*/
/* Firmware Obfuscation V1.0.4                                                */
/* Written in 2019 by Rudy Tellert Elektronik                                 */
/*                                                                            */
/* To the extent possible under law, the author has dedicated all copyright   */
/* and related and neighboring rights to the software "firmware obfuscation"  */
/* and its documentation to the public domain. This software and its          */
/* documentation are distributed without any warranty.                        */
/*                                                                            */
/* See also                                                                   */
/* http://creativecommons.org/publicdomain/zero/1.0/                          */
/* ---------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/* Excerpt from Tagged Stream Format V1.1.16                                  */
/*----------------------------------------------------------------------------*/

#ifndef __TSF_H
#define __TSF_H

/*- Switches for target platform ---------------------------------------------*/

/* #define TSF_DEBUG */ /* inserts debug code (warning: compile entire project if changed) */
/* #define TSF_SUPPORT_C99 */ /* compiler supports C99 */
#define TSF_SUPPORT_STRICT_PAR_CHECK /* enable strict parameter check */
#define TSF_SUPPORT_TAG_ORDER_CHECK /* enable ids sort order check */

#define TSF_LEVEL_COUNT 8 /* max. level of embedded collections */

#ifdef SUPPORT_BIG_ENDIAN
#define TSF_SUPPORT_REV /* support for both little and big endian */
#define TSF_SUPPORT_REV_ACCESS /* support for access of the reverse property */
/* #define TSF_SUPPORT_REV_FIELD */ /* support for additional reverse field (faster code) */
#endif
                              
/*- Include ------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <limits.h>
#ifdef TSF_SUPPORT_C99
#  include <stdint.h>
#  include <stdbool.h>
#endif

/*- Basic definitions --------------------------------------------------------*/

#ifndef TSF_EXPORT
#  define TSF_EXPORT
#endif
#ifndef TSF_API
#  define TSF_API
#endif

#if defined(UNICODE) || defined(_UNICODE)
#  define TSF_UNICODE
#endif

#ifndef TSF_SUPPORT_REV
#ifdef TSF_SUPPORT_REV_ACCESS
#undef TSF_SUPPORT_REV_ACCESS
#endif
#ifdef TSF_SUPPORT_REV_CHECK
#undef TSF_SUPPORT_REV_CHECK
#endif
#ifdef TSF_SUPPORT_REV_FIELD
#undef TSF_SUPPORT_REV_FIELD
#endif
#endif

#if !defined(TSF_POINTER_MODIFIER)
#  define TSF_POINTER_MODIFIER const
#endif

#ifdef __cplusplus
#  define TSF_USE_EXTERN_C
#  define TSF_USE_INLINE
#endif

#ifndef TSF_SUPPORT_C99
#  define INT8_MAX   0x7f
#  define INT16_MAX  0x7fff
#  define INT32_MAX  0x7fffffff
#  define INT64_MAX  0x7fffffffffffffff
#  define INT8_MIN   (-0x7f-1)
#  define INT16_MIN  (-0x7fff-1)
#  define INT32_MIN  (-0x7fffffff-1)
#  define INT64_MIN  (-0x7fffffffffffffff-1)
#  define UINT8_MAX  0xffU
#  define UINT16_MAX 0xffffU
#  define UINT32_MAX 0xffffffffUL
#  define UINT64_MAX 0xffffffffffffffffULL
#endif

#if SHRT_MAX == INT8_MAX
#  define TSF_SHRT_BITS 8
#elif SHRT_MAX == INT16_MAX
#  define TSF_SHRT_BITS 16
#elif SHRT_MAX == INT32_MAX
#  define TSF_SHRT_BITS 32
#elif SHRT_MAX == INT64_MAX
#  define TSF_SHRT_BITS 64
#endif

#if INT_MAX == INT8_MAX
#  define TSF_INT_BITS 8
#elif INT_MAX == INT16_MAX
#  define TSF_INT_BITS 16
#elif INT_MAX == INT32_MAX
#  define TSF_INT_BITS 32
#elif INT_MAX == INT64_MAX
#  define TSF_INT_BITS 64
#endif

#if LONG_MAX == INT8_MAX
#  define TSF_LONG_BITS 8
#elif LONG_MAX == INT16_MAX
#  define TSF_LONG_BITS 16
#elif LONG_MAX == INT32_MAX
#  define TSF_LONG_BITS 32
#elif LONG_MAX == INT64_MAX
#  define TSF_LONG_BITS 64
#endif

#if LLONG_MAX == INT8_MAX
#  define TSF_LLONG_BITS 8
#elif LLONG_MAX == INT16_MAX
#  define TSF_LLONG_BITS 16
#elif LLONG_MAX == INT32_MAX
#  define TSF_LLONG_BITS 32
#elif LLONG_MAX == INT64_MAX
#  define TSF_LLONG_BITS 64
#endif

#if USHRT_MAX == UINT8_MAX
#  define TSF_USHRT_BITS 8
#elif USHRT_MAX == UINT16_MAX
#  define TSF_USHRT_BITS 16
#elif USHRT_MAX == UINT32_MAX
#  define TSF_USHRT_BITS 32
#elif USHRT_MAX == UINT64_MAX
#  define TSF_USHRT_BITS 64
#endif

#if UINT_MAX == UINT8_MAX
#  define TSF_UINT_BITS 8
#elif UINT_MAX == UINT16_MAX
#  define TSF_UINT_BITS 16
#elif UINT_MAX == UINT32_MAX
#  define TSF_UINT_BITS 32
#elif UINT_MAX == UINT64_MAX
#  define TSF_UINT_BITS 64
#endif

#if ULONG_MAX == UINT8_MAX
#  define TSF_ULONG_BITS 8
#elif ULONG_MAX == UINT16_MAX
#  define TSF_ULONG_BITS 16
#elif ULONG_MAX == UINT32_MAX
#  define TSF_ULONG_BITS 32
#elif ULONG_MAX == UINT64_MAX
#  define TSF_ULONG_BITS 64
#endif

#if ULLONG_MAX == UINT8_MAX
#  define TSF_ULLONG_BITS 8
#elif ULLONG_MAX == UINT16_MAX
#  define TSF_ULLONG_BITS 16
#elif ULLONG_MAX == UINT32_MAX
#  define TSF_ULLONG_BITS 32
#elif ULLONG_MAX == UINT64_MAX
#  define TSF_ULLONG_BITS 64
#endif

#ifndef SIZE_MAX
#  define SIZE_MAX UINT_MAX
#endif

#if SIZE_MAX == UINT8_MAX
#  define TSF_SIZE_BITS 8
#elif SIZE_MAX == UINT16_MAX
#  define TSF_SIZE_BITS 16
#elif SIZE_MAX == UINT32_MAX
#  define TSF_SIZE_BITS 32
#elif SIZE_MAX == UINT64_MAX
#  define TSF_SIZE_BITS 64
#endif

#ifdef TSF_SUPPORT_C99
typedef int8_t   TsfInt8;
typedef int16_t  TsfInt16;
typedef int32_t  TsfInt32;
typedef int64_t  TsfInt64;
typedef uint8_t  TsfUInt8;
typedef uint16_t TsfUInt16;
typedef uint32_t TsfUInt32;
typedef uint64_t TsfUInt64;
#else
typedef signed char        TsfInt8;
typedef signed short       TsfInt16;
typedef signed long        TsfInt32;
typedef signed long long   TsfInt64;
typedef unsigned char      TsfUInt8;
typedef unsigned short     TsfUInt16;
typedef unsigned long      TsfUInt32;
typedef unsigned long long TsfUInt64;
#endif

#ifdef TSF_UNICODE
#  define TSF_T(str) L ## str
#  define TSF_TEXT(str) L ## str
#  ifdef TSF_SUPPORT_WCHAR_T
typedef wchar_t TSF_TCHAR;
#  else
typedef TsfUInt16 TSF_TCHAR;
#  endif
#else
#  define TSF_T(str) str
#  define TSF_TEXT(str) str
typedef char TSF_TCHAR;
#endif
typedef TSF_TCHAR *TSF_TSTR;
typedef const TSF_TCHAR *TSF_CTSTR;

/*----------------------------------------------------------------------------*/

#ifdef TSF_USE_EXTERN_C
extern "C" {
#endif

    typedef TsfUInt8  TsfByte;
    typedef TsfUInt16 TsfWord;
    typedef TsfByte TsfId;
    typedef TsfByte TsfLevel;
    typedef TsfByte TsfUuid[16];

#ifdef TSF_SUPPORT_C99
#define TSF_FALSE false
#define TSF_TRUE true
    typedef bool TsfBool;
#else
    enum { TSF_FALSE, TSF_TRUE };
    typedef TsfByte TsfBool;
#endif

    enum { TSFS_DATA, TSFS_END, TSFS_END_OF_STREAM, TSFS_ERROR };
    typedef TsfByte TsfStatusType;

    enum { TSFH_NONE, TSFH_NORMAL, TSFH_REV };
    typedef TsfByte TsfHeaderType;

    enum {
        TSFE_DEFAULT = 0x00, TSFE_NULL = 0x01, TSFE_REDEF = 0x02,
        TSFE_FIXED_ARRAY = 0x11, TSFE_VAR_ARRAY = 0x12
    };

    enum {
        TSFT_NONE, TSFT_END, TSFT_END_OF_STREAM, TSFT_ERROR,
        TSFT_END_OF_ARRAY, TSFT_END_OF_COLLECTION,
        TSFT_DATA,
        TSFT_COLLECTION, TSFT_ARRAY,
        TSFT_EXT
    };
    typedef TsfByte TsfTokenType;

    typedef struct s_TsfLevelData {
        size_t Count;
#ifdef TSF_SUPPORT_TAG_ORDER_CHECK
        TsfId  CurrId;
#endif
    } TsfLevelData;

    typedef struct s_TsfObject {
#ifdef TSF_DEBUG
        volatile TsfUInt32 StartPattern;
#endif

        /* layer 0 */
        size_t        TotalSize;

        TsfByte       TSF_POINTER_MODIFIER *Data; /* stream */
        size_t        RemainingSize; /* stream */
        /* layer 1 */
        size_t        DataSize;
        TsfHeaderType Header;
#ifdef TSF_SUPPORT_REV_FIELD
        TsfBool       Reverse;
#endif
        TsfStatusType Status;
        TsfLevel      Level;
        TsfLevelData  LevelData[TSF_LEVEL_COUNT];
        size_t        Count;
        TsfId         Id;

#ifdef TSF_DEBUG
        volatile TsfUInt32 EndPattern;
#endif
    } TsfObject;

    typedef struct s_TsfStruct {
        TsfByte Id;
        TsfByte Type;
        TsfWord Number;
    } TsfStruct;


/*- tsf ----------------------------------------------------------------------*/

#define TsfGetStatus(tsf) (tsf->Status)
#define TsfIsGood(tsf) (tsf->Status != TSFS_ERROR)
#define TsfIsBad(tsf) (tsf->Status == TSFS_ERROR)
#define TsfGetHeaderType(tsf) (tsf->Header)
#define TsfGetId(tsf) (tsf->Id)
#define TsfGetCount(tsf) (tsf->Count)
#define TsfGetLevel(tsf) (tsf->Level)
#define TsfGetExt(tsf) (0)
#define TsfGetData(tsf) ((const void *)tsf->Data)
#define TsfGetDataChars(tsf) ((const char *)tsf->Data)
#define TsfGetDataBytes(tsf) ((const TsfByte *)tsf->Data)
#define TsfGetDataSize(tsf) (tsf->DataSize)
#ifdef TSF_DEBUG
    TSF_EXPORT void TSF_API TsfClose(TsfObject *tsf);
#else
#  define TsfClose(tsf)
#endif

/*- read (layer 1) -----------------------------------------------------------*/

    TSF_EXPORT size_t TSF_API TsfGetPos(TsfObject *tsf);
    TSF_EXPORT TsfBool TSF_API TsfSetPos(TsfObject *tsf, size_t pos);
#  define TsfReadFile(fileName, data, size, fileLength) (TSF_FALSE)
#  define TsfOpenFile(tsf, fileName, data, size) (TSF_FALSE)
    TSF_EXPORT void TSF_API TsfOpen(TsfObject *tsf, const void *data, size_t size);
    TSF_EXPORT TsfTokenType TSF_API TsfReadToken(TsfObject *tsf);

#ifdef TSF_SUPPORT_REV_ACCESS
    TSF_EXPORT TsfBool TSF_API TsfGetReverse(TsfObject *tsf);
    TSF_EXPORT void TSF_API TsfSetReverse(TsfObject *tsf, TsfBool isRev);
#else
#  define TsfGetReverse(tsf) (TSF_FALSE)
#  define TsfSetReverse(tsf, b)
#endif

#ifdef TSF_SUPPORT_REV_FIELD
#  define TsfIsReverse(tsf) (tsf->Reverse)
#else
#  define TsfIsReverse(tsf) ((tsf->Header == TSFH_REV) ? TSF_TRUE : TSF_FALSE)
#endif
    TSF_EXPORT TsfBool TSF_API TsfReadData(TsfObject *tsf, void *data, size_t size);
    TSF_EXPORT TsfBool TSF_API TsfReadObject(TsfObject *tsf, void *data, size_t size);
#define TsfSkipData(tsf) TsfReadData(tsf, NULL, tsf->DataSize)

#ifdef TSF_SUPPORT_REV
    TSF_EXPORT void TSF_API TsfSortData(TsfObject *tsf, void *data, size_t size);
    TSF_EXPORT TsfBool TSF_API TsfReadSortedData(TsfObject *tsf, void *data, size_t size);
    TSF_EXPORT TsfBool TSF_API TsfReadSortedObject(TsfObject *tsf, void *data, size_t size);
#else
#  define TsfSortData(tsf, data, n)
#  define TsfReadSortedData(tsf, data, n) TsfReadData(tsf, data, n)
#endif

    TSF_EXPORT TsfBool TSF_API TsfReadUInt8(TsfObject *tsf, TsfUInt8 *data);
    TSF_EXPORT TsfBool TSF_API TsfReadUInt16(TsfObject *tsf, TsfUInt16 *data);
    TSF_EXPORT TsfBool TSF_API TsfReadUInt32(TsfObject *tsf, TsfUInt32 *data);
    TSF_EXPORT TsfBool TSF_API TsfReadUInt64(TsfObject *tsf, TsfUInt64 *data);

/*- read (layer2 ) -----------------------------------------------------------*/

    TSF_EXPORT TsfBool TSF_API TsfSelectObject(TsfObject *tsf, TsfId id);
#define TsfSelect(tsf, id) TsfSelectObject(tsf, id)
    TSF_EXPORT size_t TSF_API TsfReadArraySize(TsfObject *tsf, TsfId id);
#define TsfSelectArray(tsf, id) (TsfReadArraySize(tsf, id) != 0)
    TSF_EXPORT TsfBool TSF_API TsfSkipCollectionOrArray(TsfObject *tsf, TsfBool skipCollectionOnly);
#define TsfSkipCollection(tsf) TsfSkipCollectionOrArray(tsf, TSF_TRUE)
#define TsfSkipArray(tsf) TsfSkipCollectionOrArray(tsf, TSF_FALSE)

#  define TsfReadUChar(tsf, data) TsfReadUInt8(tsf, (TsfUInt8*)data)
#  if TSF_USHRT_BITS == 8
#    define TsfReadUShort(tsf, data) TsfReadUInt8(tsf, (TsfUInt8*)data)
#  elif TSF_USHRT_BITS == 16
#    define TsfReadUShort(tsf, data) TsfReadUInt16(tsf, (TsfUInt16*)data)
#  elif TSF_USHRT_BITS == 32
#    define TsfReadUShort(tsf, data) TsfReadUInt32(tsf, (TsfUInt32*)data)
#  elif TSF_USHRT_BITS == 64
#    define TsfReadUShort(tsf, data) TsfReadUInt64(tsf, (TsfUInt64*)data)
#  endif

#  if TSF_UINT_BITS == 8
#    define TsfReadUInt(tsf, data) TsfReadUInt8(tsf, (TsfUInt8*)data)
#  elif TSF_UINT_BITS == 16
#    define TsfReadUInt(tsf, data) TsfReadUInt16(tsf, (TsfUInt16*)data)
#  elif TSF_UINT_BITS == 32
#    define TsfReadUInt(tsf, data) TsfReadUInt32(tsf, (TsfUInt32*)data)
#  elif TSF_UINT_BITS == 64
#    define TsfReadUInt(tsf, data) TsfReadUInt64(tsf, (TsfUInt64*)data)
#  endif

#  if TSF_ULONG_BITS == 8
#    define TsfReadULong(tsf, data) TsfReadUInt8(tsf, (TsfUInt8*)data)
#  elif TSF_ULONG_BITS == 16
#    define TsfReadULong(tsf, data) TsfReadUInt16(tsf, (TsfUInt16*)data)
#  elif TSF_ULONG_BITS == 32
#    define TsfReadULong(tsf, data) TsfReadUInt32(tsf, (TsfUInt32*)data)
#  elif TSF_ULONG_BITS == 64
#    define TsfReadULong(tsf, data) TsfReadUInt64(tsf, (TsfUInt64*)data)
#  endif

#  if TSF_ULLONG_BITS == 8
#    define TsfReadULLong(tsf, data) TsfReadUInt8(tsf, (TsfUInt8*)data)
#  elif TSF_ULLONG_BITS == 16
#    define TsfReadULLong(tsf, data) TsfReadUInt16(tsf, (TsfUInt16*)data)
#  elif TSF_ULLONG_BITS == 32
#    define TsfReadULLong(tsf, data) TsfReadUInt32(tsf, (TsfUInt32*)data)
#  elif TSF_ULLONG_BITS == 64
#    define TsfReadULLong(tsf, data) TsfReadUInt64(tsf, (TsfUInt64*)data)
#  endif

#  if TSF_SIZE_BITS == 8
#    define TsfReadSize(tsf, data) TsfReadUInt8(tsf, (TsfUInt8*)data)
#  elif TSF_SIZE_BITS == 16
#    define TsfReadSize(tsf, data) TsfReadUInt16(tsf, (TsfUInt16*)data)
#  elif TSF_SIZE_BITS == 32
#    define TsfReadSize(tsf, data) TsfReadUInt32(tsf, (TsfUInt32*)data)
#  elif TSF_SIZE_BITS == 64
#    define TsfReadSize(tsf, data) TsfReadUInt64(tsf, (TsfUInt64*)data)
#  endif

#endif
